﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_Selector.h>
#include <nn/util/util_StringUtil.h>
#include <nn/fs.h>
#include <nn/oe.h>

#include "ConsumableServiceItem_LibraryApi.h"

namespace
{
    const char* SampleApplicationId = "0x01004B9000490000";
    const char* SampleNsaId = "0x0000000000000000";

    template<typename T>
    void ReadSaveData(T* pOutSavedata, const char* path) NN_NOEXCEPT
    {
        nn::fs::FileHandle fileHandle;
        {
            //! Open SaveData
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            //! Get SaveDataSize
            int64_t readFileSize;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&readFileSize, fileHandle));
            NN_ABORT_UNLESS(readFileSize == sizeof(T));

            //! Read SaveData
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, pOutSavedata, readFileSize));
        }
    }

    template<typename T>
    void WriteSaveData(const T& savedata, const char* path) NN_NOEXCEPT
    {
        nn::fs::FileHandle fileHandle;
        {
            //! Open SaveData
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            //! Write SaveData
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(fileHandle, 0, &savedata, sizeof(T), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }
    }
}

namespace nn { namespace ConsumableServiceItem {

struct LocalRight
{
    bool isServerRightConsumed;
    Right right;
};

struct SaveDataStructure
{
    static const int MaxRightCount = 64;

    int rightCount = 0;
    LocalRight localRights[MaxRightCount];
};

void PrintSaveData(const char* label, const SaveDataStructure& savedata) NN_NOEXCEPT
{
    NN_LOG("+++++++++++++++++++++++++ %40s +++++++++++++++++++++++++\n", label);
    NN_LOG("Right Count: %d\n", savedata.rightCount);
    for (int i = 0; i < savedata.rightCount; i++)
    {
        NN_LOG("\tRights[%d].rightId:\t%s:%d\n", i, savedata.localRights[i].right.rightId, savedata.localRights[i].isServerRightConsumed);
    }
    NN_LOG("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n");
}

void PrintSaveData(const char* label, const SaveDataStructure& savedata, const int itemCount) NN_NOEXCEPT
{
    PrintSaveData(label, savedata);
    NN_LOG("In-Application Item: %2d\n", itemCount);
}

void MergeServerRightsToLocalRights(SaveDataStructure* pSaveData, const Right rServer[], const int rServerCount) NN_NOEXCEPT
{
    //! 重複判定
    auto IsDuplicated = [](const Right& right, const LocalRight localRights[], const int count)->bool
    {
        for (int i = 0; i < count; i++)
        {
            if (nn::util::Strncmp(right.rightId, localRights[i].right.rightId, IdLength) == 0)
            {
                return true;
            }
        }
        return false;
    };

    //! セーブデータ上の権利との重複を確認しながら、サーバ上の権利を取り込む
    //! そのため、セーブデータ上の権利は、サーバ上の権利によって上書きされることはない
    //! セーブデータに書き込める最大権利数に到達した時点で、サーバからの権利の取り込みをやめる
    //! 取り込みきれなかったサーバ上の権利は、再び権利移行を実行した際にセーブデータに取り込まれる

    int allCount = pSaveData->rightCount;
    for (int i = 0; i < rServerCount && allCount < pSaveData->MaxRightCount; i++)
    {
        if (!IsDuplicated(rServer[i], pSaveData->localRights, allCount))
        {
            pSaveData->localRights[allCount].isServerRightConsumed = false;
            pSaveData->localRights[allCount].right = rServer[i];
            allCount++;
        }
    }
    pSaveData->rightCount = allCount;
}

void TransferServerRightsToLocalRights(const char* filePath) NN_NOEXCEPT
{
    //! セーブデータを読み込み、デバイス上の権利を取得する
    SaveDataStructure savedata;
    ReadSaveData(&savedata, filePath);

    PrintSaveData("local rights", savedata);

    //! サーバ上の権利を取得する
    int serverRightCount = 0;
    Right serverRights[SaveDataStructure::MaxRightCount];
    GetRights(serverRights, &serverRightCount, SaveDataStructure::MaxRightCount, SampleApplicationId, SampleNsaId);

    //! デバイス上の権利とサーバ上の権利をまとめる
    MergeServerRightsToLocalRights(&savedata, serverRights, serverRightCount);

    PrintSaveData("local rights + server rights", savedata);

    //! 各権利に権利消費 API を実行していく
    for (int i = 0; i < savedata.rightCount; i++)
    {
        //! サーバ上の権利を消費する
        if (ConsumeRight(savedata.localRights[i].right.rightId, SampleApplicationId, SampleNsaId).IsSuccess())
        {
            //! 権利を使用可能にする
            savedata.localRights[i].isServerRightConsumed = true;

            //! 権利をセーブデータにコミットする
            WriteSaveData(savedata, filePath);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

            PrintSaveData("local rights + server rights", savedata);

            //! 挙動確認のための電源断を、このタイミングで行えるようにウェイトする
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }
}

void TransformLocalRightsToApplicationItems(const char* rightsFilePath, const char* itemsFilePath)
{
    //! アプリ内アイテムの所有数を取得する
    int inApplicationItem = 0;
    ReadSaveData(&inApplicationItem, itemsFilePath);

    //! セーブデータを読み込み、デバイス上の権利を取得する
    SaveDataStructure savedata;
    ReadSaveData(&savedata, rightsFilePath);

    PrintSaveData("local rights, in-application item count", savedata, inApplicationItem);

    //! index 番目の権利をセーブデータから削除する
    auto EraseRight = [](SaveDataStructure* pSaveData, const int index)->void
    {
        for (int i = index; i < pSaveData->rightCount - 1; i++)
        {
            pSaveData->localRights[i] = pSaveData->localRights[i + 1];
        }
        --pSaveData->rightCount;
    };

    //! 使用可能な権利をアプリ内アイテムに変換していく
    for (int i = 0; i < savedata.rightCount;)
    {
        if (savedata.localRights[i].isServerRightConsumed)
        {
            //! 権利を削除する
            EraseRight(&savedata, i);

            //! アプリ内アイテムを増加させる
            ++inApplicationItem;

            //! 権利、アプリ内アイテムをセーブデータにコミットする
            WriteSaveData(savedata, rightsFilePath);
            WriteSaveData(inApplicationItem, itemsFilePath);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

            PrintSaveData("local rights, in-application item count", savedata, inApplicationItem);

            //! 挙動確認のための電源断を、このタイミングで行えるようにウェイトする
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        else
        {
            ++i;
        }
    }
}

extern "C" void nnMain() NN_NOEXCEPT
{
    oe::Initialize();
    account::Initialize();

    const char* mountName = "save";
    const char* rightsFilePath = "save:/rights";
    const char* itemsFilePath = "save:/items";

    //! Mount SaveData
    nn::account::Uid user;
    auto accountResult = nn::account::ShowUserSelector(&user);
    if (nn::account::ResultCancelledByUser::Includes(accountResult))
    {
        NN_LOG("User selection cancelled\n");
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(accountResult);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::EnsureSaveData(user));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveData(mountName, user));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(mountName);
    };

    //! Create SaveData
    Result result;
    result = fs::CreateFile(rightsFilePath, sizeof(SaveDataStructure));
    if (fs::ResultUsableSpaceNotEnough::Includes(result) || fs::ResultPathNotFound::Includes(result))
    {
        NN_LOG("Create file result failure\n");
        return;
    }

    result = fs::CreateFile(itemsFilePath, sizeof(int));
    if (fs::ResultUsableSpaceNotEnough::Includes(result) || fs::ResultPathNotFound::Includes(result))
    {
        NN_LOG("Create file result failure\n");
        return;
    }

    //! サーバ上の権利をセーブデータに移行する
    TransferServerRightsToLocalRights(rightsFilePath);

    //! セーブデータ上の権利をアプリ内アイテムに変換する
    TransformLocalRightsToApplicationItems(rightsFilePath, itemsFilePath);
}

}}
