﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/usb/testUsb_Fx3Utility.h>

namespace nnt {
namespace usb {

    TestFx3Utility::TestFx3Utility()
    {
        m_IsInitialized = false;
        m_IfCount = 0;
        m_Fx3DeviceFilter = {
            .matchFlags = nn::usb::DeviceFilterMatchFlags_Vendor | nn::usb::DeviceFilterMatchFlags_Product,
            .idVendor = Fx3BootLoaderVID,
            .idProduct = Fx3BootLoaderPID,
        };
    }


    TestFx3Utility::~TestFx3Utility()
    {
    }


    nn::Result TestFx3Utility::Initialize(int seconds)
    {
        nn::Result result = nn::ResultSuccess();
        NN_LOG("Fx3Utility Intialize...\n");

        result = m_HsClient.Initialize();
        result = m_HsClient.CreateInterfaceAvailableEvent(&m_IfAvailableEvent,
                nn::os::EventClearMode_ManualClear,
                0,
                &m_Fx3DeviceFilter);
        NN_LOG("Looking for FX3 control interface...");
        if (nn::os::TimedWaitSystemEvent(&m_IfAvailableEvent, nn::TimeSpan::FromSeconds(seconds)))
        {
            nn::os::ClearSystemEvent(&m_IfAvailableEvent);
            NN_LOG("Interface ready.\n");
        }
        else
        {
            NN_ABORT("FAILED to find FX3 control interface after %d seconds.\n", seconds);
        }

        result = m_HsClient.QueryAvailableInterfaces(&m_IfCount, m_IfList, sizeof(m_IfList), &m_Fx3DeviceFilter);

        NN_LOG("m_IfCount = %d\n", m_IfCount);

        for (int i = 0; i < m_IfCount; i++)
        {
            nn::usb::InterfaceProfile& profile = m_IfList[i].ifProfile;

            if (profile.ifDesc.bInterfaceClass == Fx3ControlInterfaceClass)
            {
                NN_LOG("Initializing vendor-specific interface 0x%x\n", profile.ifDesc.bInterfaceNumber);
                result = m_Fx3Interface.Initialize(&m_HsClient, profile.handle);
                if (result.IsSuccess())
                {
                    m_pIfStateChangedEvent = m_Fx3Interface.GetStateChangeEvent();
                }
                break;
            }
        }

        m_IsInitialized = true;
        return result;
    }

    bool TestFx3Utility::IsInitialized()
    {
        return m_IsInitialized;
    }

    void TestFx3Utility::Finalize()
    {
        m_Fx3Interface.Finalize();
        m_HsClient.DestroyInterfaceAvailableEvent(&m_IfAvailableEvent, 0);
        m_HsClient.Finalize();
        m_IsInitialized = true;
    }


    Fx3FwErrorCode TestFx3Utility::UpdateFirmware(const char *fx3Firmware, int size)
    {
        nn::Result result = nn::ResultSuccess();
        uint8_t ramOpCode = 0xA0;

        static NN_USB_DMA_ALIGN unsigned char downloadBuffer[UPortMaxBufSize];
        static NN_USB_DMA_ALIGN unsigned char uploadBuffer[UPortMaxBufSize];
        bool isTrue = true;
        unsigned int computeCheckSum = 0;
        unsigned int expectedCheckSum = 0;
        int fwImagePtr = 0;
        unsigned int sectionLength = 0;
        unsigned int sectionAddress = 0;
        unsigned int downloadAddress = 0;
        unsigned int programEntry = 0;

        /* Initialize computed checksum */
        computeCheckSum = 0;

        if ((fx3Firmware[fwImagePtr] != 0x43) || (fx3Firmware[fwImagePtr + 1] != 0x59))
        {
            // Signature doesn't match
            return nnt::usb::Fx3FwErrorCode_InvalidFwSignature;
        }

        // Skip the two bytes signature and the following two bytes
        fwImagePtr += 4;

        while (isTrue)
        {
            // Get section length(4 bytes) and convert it from 32 - bit word count to byte count
            NNT_USB_CYWB_BL_4_BYTES_COPY(&sectionLength, &fx3Firmware[fwImagePtr]);
            fwImagePtr += 4;

            sectionLength = sectionLength << 2;
            if (sectionLength == 0)
            {
                break;
            }

            NNT_USB_CYWB_BL_4_BYTES_COPY(&sectionAddress, &fx3Firmware[fwImagePtr]);
            fwImagePtr += 4;

            int bytesLeftToDownload = sectionLength;
            downloadAddress = sectionAddress;

            while (bytesLeftToDownload > 0)
            {
                int bytesToTransfer = UPortMaxBufSize;
                if (bytesLeftToDownload < UPortMaxBufSize)
                {
                    bytesToTransfer = bytesLeftToDownload;
                }

                if (bytesToTransfer > size)
                {
                    NN_LOG("Error: bytesToTransfer = %d. Size of firmware = %d\n", bytesToTransfer, size);
                    return nnt::usb::Fx3FwErrorCode_CorruptFwImageFile;
                }

                memcpy(downloadBuffer, static_cast<const void *>(fx3Firmware + fwImagePtr), bytesToTransfer);

                //  Compute checksum: Here transferLength is assumed to be a multiple of 4. If it is not, the checksum will fail anyway
                for (int index = 0; index < bytesToTransfer; index += 4)
                {
                    unsigned int buf32Bits = 0;
                    NNT_USB_CYWB_BL_4_BYTES_COPY(&buf32Bits, &downloadBuffer[index]);
                    computeCheckSum += buf32Bits;
                }

                int maxTryCount = 3;
                for (int tryCount = 1; tryCount <= maxTryCount; tryCount++)
                {
                    // Send downloadBuffer to device over control point
                    result = DownloadBufferToDevice(downloadAddress, bytesToTransfer, downloadBuffer, ramOpCode);

                    if (!result.IsSuccess())
                    {
                        if (tryCount == maxTryCount)
                        {
                            return nnt::usb::Fx3FwErrorCode_Failed;
                        }
                        else
                        {
                            NN_LOG("F/W buffer download failure. Trying writing/verifying current buffer again... \n");
                            continue;
                        }
                    }

                    memset(uploadBuffer, 0, bytesToTransfer);

                    // Recieve uploadBuffer from device over control point for verification
                    result = UploadBufferFromDevice(downloadAddress, bytesToTransfer, uploadBuffer, ramOpCode);

                    if (!result.IsSuccess())
                    {
                        if (tryCount == maxTryCount)
                        {
                            return nnt::usb::Fx3FwErrorCode_Failed;
                        }
                        else
                        {
                            NN_LOG("F/W buffer upload failure. Trying writing/verifying current buffer again... \n");
                            continue;
                        }
                    }

                    for (int count = 0; count < bytesToTransfer; count++)
                    {
                        if (downloadBuffer[count] != uploadBuffer[count])
                        {
                            if (tryCount == maxTryCount)
                            {
                                return nnt::usb::Fx3FwErrorCode_Failed;
                            }
                            else
                            {
                                NN_LOG("Uploaded data does not match downloaded data. Trying writing/verifying current buffer again...\n");
                                continue;
                            }
                        }
                    }

                    break;
                }

                downloadAddress += bytesToTransfer;
                fwImagePtr += bytesToTransfer;
                bytesLeftToDownload -= bytesToTransfer;

                if (fwImagePtr > size)
                {
                    return nnt::usb::Fx3FwErrorCode_IncorrectImageLength;
                }
            }
        }

        NNT_USB_CYWB_BL_4_BYTES_COPY(&programEntry, &fx3Firmware[fwImagePtr]);
        fwImagePtr += 4;

        NNT_USB_CYWB_BL_4_BYTES_COPY(&expectedCheckSum, &fx3Firmware[fwImagePtr]);
        fwImagePtr += 4;

        if (computeCheckSum != expectedCheckSum)
        {
            NN_LOG("Computed check sum: %d does not match expected: %d", computeCheckSum, expectedCheckSum);
            return Fx3FwErrorCode_CorruptFwImageFile;
        }

        unsigned char dummyBuffer[1];
        // Few of the xHCI driver stack have issue with Control IN transfer,due to that below request fails ,
        // Control IN transfer is ZERO lenght packet , and it does not have any impact on execution of firmware.
        //if (!DownloadBufferToDevice(ProgramEntry, 0, dummyBuffer, opCode))
        //{
        //    /* Downloading Program Entry failed */
        //    return FAILED;
        //}

        // Download dummyBuffer to device
        result = DownloadBufferToDevice(programEntry, 0, dummyBuffer, ramOpCode);
        if (!result.IsSuccess())
        {
            NN_LOG("Warning: Dummy buffer failed to send to device\n");
        }

        if (nn::os::TimedWaitSystemEvent(m_pIfStateChangedEvent, nn::TimeSpan::FromSeconds(10)))
        {
            NN_LOG("Fx3 BootLoader interface disconnected.\n");
        }
        else
        {
            NN_LOG("FX3 BootLoader interface failed to disconnect after 10 seconds.\n");
            return nnt::usb::Fx3FwErrorCode_Failed;
        }
        NN_LOG("Fx3 firmware installed.\n");
        return nnt::usb::Fx3FwErrorCode_Success;
    } // NOLINT(impl/function_size)


    nn::Result TestFx3Utility::DownloadBufferToDevice(unsigned int start_addr, long size, unsigned char *data_buf, uint8_t opCode)
    {
        nn::Result result = nn::ResultSuccess();
        uint16_t wValue = (start_addr & 0xFFFF);
        uint16_t wIndex = ((start_addr >> 16) & 0xFFFF);
        size_t bytesTransferred = 0;

        result = m_Fx3Interface.ControlRequest(
            &bytesTransferred,
            reinterpret_cast<uint8_t*>(data_buf),   //void      *pData,
            0x40,                                   //uint8_t    bmRequestType,
            opCode,                                 //uint8_t    bRequest,
            wValue,                                 //uint16_t   wValue,
            wIndex,                                 //uint16_t   wIndex,
            size);                                 //uint16_t   wLength) NN_NOEXCEPT;

        return result;
    }


    nn::Result TestFx3Utility::UploadBufferFromDevice(unsigned int start_addr, long size, unsigned char *data_buf, uint8_t opCode)
    {
        nn::Result result = nn::ResultSuccess();
        uint16_t wValue = (start_addr & 0xFFFF);
        uint16_t wIndex = ((start_addr >> 16) & 0xFFFF);
        size_t bytesTransferred = 0;

        result = m_Fx3Interface.ControlRequest(
            &bytesTransferred,
            reinterpret_cast<uint8_t*>(data_buf),   //void      *pData,
            0xC0,                                   //uint8_t    bmRequestType,
            opCode,                                 //uint8_t    bRequest,
            wValue,                                 //uint16_t   wValue,
            wIndex,                                 //uint16_t   wIndex,
            size);                                 //uint16_t   wLength) NN_NOEXCEPT;

        return result;
    }

}
}
