﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/graphics/testGraphics_Png.h>

#include <cstring>
#include <cstdlib>
#include <zlib.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Endian.h>
#include <nn/fs.h>
#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_CreateDirectories.h>

namespace nnt{ namespace graphics{

//------------------------------------------------------------------------------
// IHDR
//------------------------------------------------------------------------------
    void PngIhdr::SetDefault() NN_NOEXCEPT
    {
        width = 0;
        height = 0;
        bitDepth = 1;
        colorType = 0;
        compressionType = 0;
        filterType = 0;
        interlaceType = 0;
        channels = 0;
    }
    uint8_t PngIhdr::GetPixelDepth() const NN_NOEXCEPT
    {
        return bitDepth * channels;
    }
    uint32_t PngIhdr::GetRowBytes() const NN_NOEXCEPT
    {
        return ((width * static_cast<uint32_t>(GetPixelDepth()) + 7) >> 3);
    }

//------------------------------------------------------------------------------
//  PNGファイル読み込み
//------------------------------------------------------------------------------
    namespace{
        //! 8bitデータをストリームから入力
        uint8_t Input8( uint8_t** pStream )
        {
            uint8_t data = *(*pStream);
            *pStream = *pStream + 1;
            return data;
        }
        // 32bitデータをストリームから入力
        uint32_t Input32( uint8_t** pStream )
        {
            uint32_t temp;
            std::memcpy( &temp, *pStream, 4 );
            uint32_t data = nn::util::LoadBigEndian<uint32_t>(&temp);
            *pStream = *pStream + 4;
            return data;
        }
        //! バイト列データをストリームから入力
        void InputBytes( uint8_t** pStream, void* pData, size_t size )
        {
            std::memcpy( pData, *pStream, size );
            *pStream = *pStream + size;
        }
        // フィルタを適用したピクセルデータを復元
        void DecompressFilter( const uint8_t* pInput, int width, int height, uint8_t pixelBytes, uint8_t* pOutput )
        {
            int stride = width * pixelBytes;
            for ( int y = 0; y < height; ++y )
            {
                const uint8_t* pInputLine = &pInput[y * ( 1 + width * pixelBytes )];
                uint8_t* pOutputLine = &pOutput[y * ( width * pixelBytes )];
                std::memcpy(pOutputLine, pInputLine, stride);
            }
        }
    }
    bool PngIO::Read(
        uint8_t** pOutImageData,
        size_t* pOutImageDataSize,
        PngIhdr* pOutIhdr,
        const char* pPath,
        AllocatorFunction* allocateFunction )
    {
        // ファイルからデータを入力
        uint8_t* pData;
        uint32_t size;
        {
            nn::Result result;
            nn::fs::FileHandle file = {};
            int64_t fileSize = 0;
            result = nn::fs::OpenFile(&file, pPath, nn::fs::OpenMode_Read);
            NN_ASSERT(result.IsSuccess());
            result= nn::fs::GetFileSize(&fileSize, file);
            NN_ASSERT(result.IsSuccess());
            size = static_cast<uint32_t>(fileSize);
            pData = reinterpret_cast<uint8_t*>(allocateFunction->Allocate(size, NN_ALIGNOF(int)));
            NN_ASSERT_NOT_NULL(pData);
            result = nn::fs::ReadFile(file, 0, pData, size);
            NN_ASSERT(result.IsSuccess());
            nn::fs::CloseFile(file);
        }

        uint8_t* dataHead = pData;

        uint8_t pixelDepth;  // ピクセル深度
        uint32_t rowBytes;

        // PNGヘッダを入力
        {
            uint8_t header[8];
            InputBytes( &pData, &header, sizeof( header ) );
        }

        // IHDRチャンクを出力
        {
            // チャンク長を入力
            Input32( &pData );

            // チャンクデータを入力
            uint8_t ihdrHeader[4];
            InputBytes( &pData, ihdrHeader, sizeof( ihdrHeader ) );
            pOutIhdr->width         = Input32( &pData );
            pOutIhdr->height        = Input32( &pData );
            pOutIhdr->bitDepth      = Input8( &pData );
            pOutIhdr->colorType     = Input8( &pData );
            pOutIhdr->compressionType = Input8( &pData );
            pOutIhdr->filterType    = Input8( &pData );
            pOutIhdr->interlaceType = Input8( &pData );

            // CRCを入力
            Input32( &pData );

            switch (pOutIhdr->colorType )
            {
            case PngColorType_Gray:
            case PngColorType_Palette:
                pOutIhdr->channels = 1;
                break;
            case PngColorType_Rgb:
                pOutIhdr->channels = 3;
                break;
            case PngColorType_GrayAlpha:
                pOutIhdr->channels = 2;
                break;
            case PngColorType_RgbAlpha:
                pOutIhdr->channels = 4;
                break;
            default:
                pOutIhdr->channels = 3;
                break;
            }

            pixelDepth = pOutIhdr->bitDepth * pOutIhdr->channels;
            rowBytes = ((pOutIhdr->width * static_cast<uint32_t>(pixelDepth) + 7) >> 3);

        }

        // 圧縮データのバッファを作成
        uint32_t decompressBufferSize = (rowBytes) * pOutIhdr->height;
        *pOutImageData = reinterpret_cast<uint8_t*>(allocateFunction->Allocate(static_cast<size_t>(decompressBufferSize), NN_ALIGNOF(int)));
        *pOutImageDataSize = static_cast<int>(decompressBufferSize);

        // IDATチャンクを入力
        {
            // チャンク長を入力
            uint32_t length = Input32( &pData );

            // チャンクデータを入力
            uint8_t  idatHeader[4];
            InputBytes( &pData, idatHeader, sizeof( idatHeader ) );

            uint8_t* pCompressBuffer = reinterpret_cast<uint8_t*>(allocateFunction->Allocate(static_cast<size_t>(length), NN_ALIGNOF(int)));
            InputBytes( &pData, pCompressBuffer, length );

            // zstreamを作成
            z_stream zstream;

            zstream.zalloc = Z_NULL;
            zstream.zfree = Z_NULL;
            zstream.opaque = Z_NULL;

            uint32_t filtedImageSize = (1 + rowBytes) * pOutIhdr->height;
            uint8_t* pFilteredImage = reinterpret_cast<uint8_t*>(allocateFunction->Allocate(static_cast<size_t>(filtedImageSize), NN_ALIGNOF(int)));

            zstream.next_in   = pCompressBuffer;
            zstream.avail_in  = length;
            zstream.next_out  = pFilteredImage;
            zstream.avail_out = filtedImageSize;

            int res = inflateInit2(&zstream, MAX_WBITS);
            NN_ASSERT(res == Z_OK);

            // データを解凍
            res = inflate(&zstream, Z_FINISH);
            NN_ASSERT(res == Z_STREAM_END);

            // zstreamを破棄
            res = inflateEnd(&zstream);
            NN_ASSERT(res == Z_OK);

            // フィルターを展開
            DecompressFilter( pFilteredImage, pOutIhdr->width, pOutIhdr->height, pixelDepth / 8, *pOutImageData );

            allocateFunction->Free( pCompressBuffer );
            allocateFunction->Free( pFilteredImage );

            // CRCを入力
            Input32( &pData );
        }

        // IENDチャンクを入力
        {
            // チャンク長を入力
            Input32( &pData );

            // チャンクデータを入力
            uint8_t  iendHeader[4];
            InputBytes( &pData, iendHeader, sizeof( iendHeader ) );

            // CRCを入力
            Input32( &pData );
        }

        allocateFunction->Free(dataHead);

        return true;
    }


    //------------------------------------------------------------------------------
    //  PNGファイル書き込み
    //------------------------------------------------------------------------------
    namespace{
        class OutputStream
        {
        public:
            static const int AllocateUnitSize = 1024;
        public:
            explicit OutputStream(AllocatorFunction* allocateFunction)
                : m_AllocateFunction(allocateFunction)
                , m_pData(nullptr)
                , m_DataSize(0)
                , m_DataCapacity(0)
            {
            }
            ~OutputStream()
            {
                if(m_pData)
                {
                    m_AllocateFunction->Free(m_pData);
                }
            }

            size_t GetDataSize() const
            {
                return m_DataSize;
            }
            const char* GetData() const
            {
                return reinterpret_cast<const char*>(m_pData);
            }

            void Output8(const void* pData, uint32_t* pCrc)
            {
                WriteBytes( static_cast<const char*>( pData ), 1 );

                if (pCrc != NULL)
                {
                    *pCrc = crc32( *pCrc, static_cast<const Bytef*>( pData ), 1 );
                }
            }

            //  32bitデータをストリームに出力
            void Output32( const void* pData, uint32_t* pCrc )
            {
                // ビッグエンディアンに変換
                uint32_t data;
                {
                    uint32_t temp;
                    std::memcpy(&temp, pData, 4);
                    nn::util::StoreBigEndian<uint32_t>(&data, temp);
                }

                WriteBytes( reinterpret_cast<const char*>( &data ), 4 );

                if (pCrc != NULL)
                {
                    *pCrc = crc32( *pCrc, reinterpret_cast<const Bytef*>( &data ), 4 );
                }
            }

            //  バイト列データをストリームに出力
            void OutputBytes( const void* pData, size_t size, uint32_t* pCrc )
            {
                WriteBytes( static_cast<const char*>( pData ), size );

                if (pCrc != NULL)
                {
                    *pCrc = crc32( *pCrc, static_cast<const Bytef*>( pData ), static_cast<nnutilZlib_uInt>(size) );
                }
            }

        private:
            void WriteBytes(const char* pData, size_t size)
            {
                size_t postSize = m_DataSize + size;
                if(postSize > m_DataCapacity)
                {
                    size_t allocateSize = ((postSize + AllocateUnitSize - 1) / AllocateUnitSize) * AllocateUnitSize;
                    void* pNewData = m_AllocateFunction->Allocate(allocateSize, NN_ALIGNOF(int));
                    if(m_pData)
                    {
                        std::memcpy(pNewData, m_pData, m_DataSize);
                        m_AllocateFunction->Free(m_pData);
                    }
                    m_pData = pNewData;
                    m_DataCapacity = m_DataSize;
                }
                std::memcpy(static_cast<char*>(m_pData) + m_DataSize, pData, size);
                m_DataSize = postSize;
            }

        private:
            AllocatorFunction* m_AllocateFunction;
            void* m_pData;
            size_t m_DataSize;
            size_t m_DataCapacity;
        };

        //  圧縮率を上げるため画像にフィルターをかける
        void CompressFilter( const uint8_t* pInput, int width, int height, uint8_t pixelBytes, uint8_t* pOutput )
        {
            for ( int y = 0; y < height; ++y )
            {
                int rowBytes = width * pixelBytes;

                const uint8_t* pInputLine = &pInput[y * rowBytes];
                uint8_t* pOutputLine = &pOutput[y * ( 1 + rowBytes )];

                pOutputLine[0] = 0;

                for ( int x = 0; x < rowBytes; ++x )
                {
                    pOutputLine[1 + x] = pInputLine[x];
                }
            }
        }

    }// anonymous namespace

    bool PngIO::Write(
        const char* pPath,
        const uint8_t* pImageData,
        size_t imageDataSize,
        const PngIhdr* pIhdr,
        AllocatorFunction* allocateFunction)
    {
        uint32_t width = pIhdr->width;
        uint32_t height = pIhdr->height;
        uint8_t bitDepth = pIhdr->bitDepth;
        uint8_t colorType = pIhdr->colorType;
        uint8_t compressType = pIhdr->compressionType;
        uint8_t filterType = pIhdr->filterType;
        uint8_t interlaceType = pIhdr->interlaceType;
        uint8_t channels = pIhdr->channels;

        NN_UNUSED(imageDataSize);

        // メモリーストリーム
        OutputStream stream(allocateFunction);

        // PNGヘッダを出力
        {
            uint8_t header[] = { 0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a };
            stream.OutputBytes(header, sizeof( header ), nullptr);
        }

        switch (colorType )
        {
        case PngColorType_Gray:
        case PngColorType_Palette:
            channels = 1;
            break;
        case PngColorType_Rgb:
            channels = 3;
            break;
        case PngColorType_GrayAlpha:
            channels = 2;
            break;
        case PngColorType_RgbAlpha:
            channels = 4;
            break;
        default:
            channels = 3;
            break;
        }

        uint8_t pixelDepth = bitDepth * channels;
        uint32_t rowBytes = ((width * static_cast<uint32_t>(pixelDepth) + 7) >> 3);

        // IHDRチャンクを出力
        {
            uint32_t ihdrLength        = 13;
            uint8_t  ihdrHeader[]      = { 'I', 'H', 'D', 'R' };

            // CRCを初期化
            uint32_t crc = crc32( 0L, Z_NULL, 0 );

            // チャンク長を出力
            stream.Output32(&ihdrLength, nullptr);

            // チャンクデータを出力
            stream.OutputBytes(ihdrHeader, sizeof( ihdrHeader ), &crc);
            stream.Output32(&width, &crc);
            stream.Output32(&height, &crc);
            stream.Output8(&bitDepth, &crc);
            stream.Output8(&colorType, &crc);
            stream.Output8(&compressType, &crc);
            stream.Output8(&filterType, &crc);
            stream.Output8(&interlaceType, &crc);

            // CRCを出力
            stream.Output32(&crc, nullptr);
        }

        // 圧縮データのバッファを作成
        uint32_t compressBufferSize = (1 + rowBytes) * height + 128 * 1024;  // 念のため少し大きめのバッファサイズを確保
        uint8_t* pCompressBuffer = reinterpret_cast<uint8_t*>(allocateFunction->Allocate(static_cast<size_t>(compressBufferSize), NN_ALIGNOF(int)));
        NN_ASSERT_NOT_NULL(pCompressBuffer);

        uint32_t compressedSize;

        // ピクセルデータを圧縮
        {
            // zstreamを作成
            z_stream zstream;

            zstream.zalloc = Z_NULL;
            zstream.zfree = Z_NULL;
            zstream.opaque = Z_NULL;

            int res = deflateInit(&zstream, Z_DEFAULT_COMPRESSION);
            NN_ASSERT(res == Z_OK);

            // 圧縮率を上げるためピクセルデータにフィルターをかける
            uint32_t filteredSize = (1 + rowBytes) * height;
            uint8_t* pFilteredImage = reinterpret_cast<uint8_t*>(allocateFunction->Allocate(static_cast<size_t>(filteredSize), NN_ALIGNOF(int)));
            NN_ASSERT_NOT_NULL(pFilteredImage);

            CompressFilter( pImageData, width, height, pixelDepth / 8, pFilteredImage );

            zstream.next_in   = pFilteredImage;
            zstream.avail_in  = filteredSize;
            zstream.next_out  = pCompressBuffer;
            zstream.avail_out = compressBufferSize;

            // データを圧縮
            res = deflate(&zstream, Z_NO_FLUSH);
            NN_ASSERT(res == Z_OK);

            res = deflate(&zstream, Z_FINISH);
            NN_ASSERT(res == Z_STREAM_END);

            compressedSize = zstream.total_out;

            // zstreamを破棄
            res = deflateEnd(&zstream);
            NN_ASSERT(res == Z_OK);

            allocateFunction->Free(pFilteredImage);
        }

        // IDATチャンクを出力
        {
            uint32_t idatLength = compressedSize;
            uint8_t  idatHeader[] = { 'I', 'D', 'A', 'T' };

            // CRCを初期化
            uint32_t crc = crc32( 0L, Z_NULL, 0 );

            // チャンク長を出力
            stream.Output32(&idatLength, nullptr);

            // チャンクデータを出力
            stream.OutputBytes(idatHeader, sizeof( idatHeader ), &crc);
            stream.OutputBytes(pCompressBuffer, compressedSize, &crc);

            // CRCを出力
            stream.Output32(&crc, nullptr);
        }

        // 圧縮データを破棄
        allocateFunction->Free(pCompressBuffer);

        // IENDチャンクを出力
        {
            uint32_t iendLength = 0;
            uint8_t  iendHeader[] = { 'I', 'E', 'N', 'D' };

            // CRCを初期化
            uint32_t crc = crc32( 0L, Z_NULL, 0 );

            // チャンク長を出力
            stream.Output32(&iendLength, nullptr);

            // チャンクデータを出力
            stream.OutputBytes(iendHeader, sizeof( iendHeader ), &crc);

            // CRCを出力
            stream.Output32(&crc, nullptr);
        }

        // ファイルにデータを保存
        {
            nn::Result result;
            result = CreateDirectories(Path(pPath).GetParent());
            NN_ASSERT(result.IsSuccess());
            size_t size = stream.GetDataSize();
            nn::fs::FileHandle file = {};
            result = nn::fs::DeleteFile(pPath);
            NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));
            result = nn::fs::CreateFile(pPath, size);
            NN_ASSERT(result.IsSuccess());
            result = nn::fs::OpenFile(&file, pPath, nn::fs::OpenMode_Write);
            NN_ASSERT(result.IsSuccess());
            result = nn::fs::WriteFile(file, 0, stream.GetData(), size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
            NN_ASSERT(result.IsSuccess());
            nn::fs::FlushFile(file);
            nn::fs::CloseFile(file);
        }

        return true;
    }// NOLINT(impl/function_size)

    //------------------------------------------------------------------------------
    //  PNGファイル解放
    //------------------------------------------------------------------------------
    void PngIO::Destroy(uint8_t** pImageData, size_t* pImageDataSize, PngIhdr* pIhdr, AllocatorFunction* allocateFunction)
    {
        allocateFunction->Free(pImageData);
        *pImageData = nullptr;
        *pImageDataSize = 0;
        pIhdr->SetDefault();
    }




}}
