﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nnt/gfx/testGfx_FileSystem.h>

namespace nnt {
namespace gfx {

//------------------------------------------------------------------------------
//  ファイル読み込み
//------------------------------------------------------------------------------
bool NntGfxFileSystem::Load( uint8_t** ppDest, size_t* pSize, const char* pPath  )
{
    *ppDest = NULL;
    *pSize = 0;

    // ファイルパスをHFIOパスに変換
    char hfioPath[_MAX_PATH];
    sprintf( hfioPath, "/vol/hfio01/%s", pPath );
    hfioPath[13] = '\\';  // ':' を '\\' に置換

    // FSクライアントを作成
    FSClient fsClient;
    FSAddClient( &fsClient, FS_RET_NO_ERROR );

    FSStateChangeParams params;
    params.userCallback = NULL;
    params.userContext  = NULL;
    params.ioMsgQueue   = NULL;

    FSSetStateChangeNotification( &fsClient, &params );

    // コマンドブロックを初期化
    FSCmdBlock cmd;
    FSInitCmdBlock( &cmd );

    // マウントソースを検索
    FSMountSource mountSrc;
    FSStatus st = FSGetMountSource( &fsClient, &cmd, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR );

    // HFIOをマウント
    char mountPath[FS_MAX_MOUNTPATH_SIZE];
    FSMount( &fsClient, &cmd, &mountSrc, mountPath, sizeof(mountPath), FS_RET_NO_ERROR );

    // ファイル情報を取得
    FSStat fileStat;
    FSGetStat( &fsClient, &cmd, hfioPath, &fileStat, FS_RET_NO_ERROR );

    // IO Buffer用にアライメントを調整したバッファを作成
    size_t size = fileStat.size;
    uint8_t* pBuffer = nn::lmem::AllocateFromExpHeap(m_HeapHandle, *pSize, FS_IO_BUFFER_ALIGN);

    // ファイルを開く
    FSFileHandle fileHandle;
    FSOpenFile( &fsClient, &cmd, hfioPath, "r", &fileHandle, FS_RET_NO_ERROR );

    // ファイルからデータを読み込む
    FSReadFile( &fsClient, &cmd, pBuffer, size, 1, fileHandle, 0, FS_RET_NO_ERROR );

    // ファイルを閉じる
    FSCloseFile( &fsClient, &cmd, fileHandle, FS_RET_NO_ERROR );

    // HFIOをアンマウント
    FSUnmount( &fsClient, &cmd, mountPath, FS_RET_NO_ERROR );

    // FSクライアントを破棄
    FSDelClient( &fsClient, FS_RET_NO_ERROR );

    *ppDest = pBuffer;
    *pSize = size;

    return true;
}

//------------------------------------------------------------------------------
//  ファイル書き込み
//------------------------------------------------------------------------------
bool NntGfxFileSystem::Write( const uint8_t* pData, size_t size, const char* pPath )
{
    // ファイルパスをHFIOパスに変換
    char hfioPath[_MAX_PATH];
    sprintf(hfioPath, "/vol/hfio01/%s", pPath);
    hfioPath[13] = '\\';  // ':' を '\\' に置換

    // IO Buffer用にアライメントを調整したバッファを作成
    uint8_t* pAlignedBuffer = static_cast<uint8_t*>( nn::lmem::AllocateFromExpHeap(m_HeapHandle, size, FS_IO_BUFFER_ALIGN) );

    memcpy( pAlignedBuffer, pData, size );

    // FSクライアントを作成
    FSClient fsClient;
    FSAddClient(&fsClient, FS_RET_NO_ERROR);

    FSStateChangeParams params;
    params.userCallback = NULL;
    params.userContext  = NULL;
    params.ioMsgQueue   = NULL;

    FSSetStateChangeNotification(&fsClient, &params);

    // コマンドブロックを初期化
    FSCmdBlock cmd;
    FSInitCmdBlock(&cmd);

    // マウントソースを検索
    FSMountSource mountSrc;
    FSStatus st = FSGetMountSource(&fsClient, &cmd, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR);

    // HFIOをマウント
    char mountPath[FS_MAX_MOUNTPATH_SIZE];
    FSMount(&fsClient, &cmd, &mountSrc, mountPath, sizeof(mountPath), FS_RET_NO_ERROR);

    // ファイルを開く
    FSFileHandle fileHandle;
    FSOpenFile(&fsClient, &cmd, hfioPath, "w", &fileHandle, FS_RET_NO_ERROR);

    // ファイルにデータを書き込む
    FSWriteFile(&fsClient, &cmd, pAlignedBuffer, size, 1, fileHandle, 0, FS_RET_NO_ERROR);

    // ファイルを閉じる
    FSCloseFile(&fsClient, &cmd, fileHandle, FS_RET_NO_ERROR);

    // HFIOをアンマウント
    FSUnmount(&fsClient, &cmd, mountPath, FS_RET_NO_ERROR);

    // FSクライアントを破棄
    FSDelClient(&fsClient, FS_RET_NO_ERROR);

    // バッファを破棄
    //m_pHeap->Free( pAlignedBuffer );
    nn::lmem::FreeToExpHeap(m_HeapHandle, pAlignedBuffer);

    return true;
}

}
}
