﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib> // std::[malloc,free]
#include <cstring> // std::strstr
#include <string> // std::string
#include <cctype> // std::isxdigit

#include <nnt.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/crypto.h> // nn::crypto::Sha1Generator

#include <nn/codec.h>
#include <nn/codec/codec_OpusEncoder.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h> // nn::codec::detail::OpusPacketInternal

#include <nn/util/util_ScopeExit.h>

#include <nnt/codecUtil/testCodec_Util.h>


// #define ENABLE_SAVE_ENCODE_DATA
// #define ENABLE_DEBUG_PRINT
// #define ENABLE_DEBUG_PRINT_IN_DETAIL

#if defined(ENABLE_DEBUG_PRINT_IN_DETAIL) && !defined(ENABLE_DEBUG_PRINT)
#define ENABLE_DEBUG_PRINT
#endif

#if defined(ENABLE_DEBUG_PRINT)
#include <nn/nn_Log.h>
#define DEBUG_PRINT(...) NN_LOG(__VA_ARGS__)
#else  // defined(ENABLE_DEBUG_PRINT)
#define DEBUG_PRINT(...)
#endif // defined(ENABLE_DEBUG_PRINT)

namespace {

// テスト対象ファイルの拡張子
const std::string targetFileExtension = "opus.sig";
const std::string inputFileExtension = "raw";

const std::string GetTestDataRootDirectoryPath() NN_NOEXCEPT
{
    return nnt::codec::util::GetCodecTestBinariesPath()  + "\\testCodec_OpusCodec";
}

#if defined(ENABLE_SAVE_ENCODE_DATA)
std::string RemoveFileExtention(const std::string& path)
{
    std::string::size_type position = path.find_last_of(".");
    return (position == std::string::npos) ? path : path.substr(0, position);
}
#endif // defined(ENABLE_SAVE_ENCODE_DATA)

int IsReal(int c)
{
    const bool isReal = (0 != std::isdigit(c))
      || '.' == static_cast<char>(c)
      || '+' == static_cast<char>(c)
      || '-' == static_cast<char>(c)
      || 'e' == static_cast<char>(c)
      || 'E' == static_cast<char>(c);

    return static_cast<int>(isReal);
}

std::string GetSsvSubString(const char* head, const char* ss, int (*isxxx)(int) )
{
    const char* endp = std::strstr(head, ss);
    if (nullptr != endp)
    {
        for (const char* ptr = endp - 1; ptr != head; --ptr)
        {
            if (0 == isxxx( static_cast<int>(*ptr)) )
            {
                const char* startp = ptr + 1;
                const size_t length = endp - startp;
                const std::string valueString(startp);
                return valueString.substr(0, length);
            }
        }
    }
    return "0";
}

float GetSsvFloatValueFromString(const char* head, const char* ss)
{
    const std::string valueString = GetSsvSubString(head, ss, IsReal);
    return std::stof(valueString);
}


int GetSsvIntegerValueFromString(const char* head, const char* ss)
{
    const std::string valueString = GetSsvSubString(head, ss, std::isdigit);
    return std::stoi(valueString);
}

int GetSampleRateFromString(const char* filename)
{
    return GetSsvIntegerValueFromString(filename, "kHz");
}

int GetChannelCountFromString(const char* filename)
{
    return GetSsvIntegerValueFromString(filename, "ch");
}

int GetBitRateFromString(const char* filename)
{
    return GetSsvIntegerValueFromString(filename, "bps");
}

float GetFrameFromString(const char* filename)
{
    return GetSsvFloatValueFromString(filename, "ms");
}

nn::codec::OpusCodingMode GetCodingModeFromString(const char* filename)
{
    if (nullptr != std::strstr(filename, "Silk"))
    {
        return nn::codec::OpusCodingMode_Silk;
    }
    return nn::codec::OpusCodingMode_Celt;
}

std::string GetSourceFilenameFromString(const char* filename)
{
    std::string path(filename);
    std::string::size_type position = path.find_last_of("-");
    return (position == std::string::npos) ? path : path.substr(0, position);
}

void Tester(void* arg)
{
    const auto entry = *reinterpret_cast<nn::fs::DirectoryEntry*>(arg);
    DEBUG_PRINT("[Case: %s]\n", entry.name);

    nn::fs::FileHandle srcHandle;
    const std::string srcPath = nnt::codec::util::GetInputDirectoryPath() + "/" + GetSourceFilenameFromString(entry.name) + "." + inputFileExtension;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&srcHandle, srcPath.c_str(), nn::fs::OpenMode_Read));
    int64_t fileSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, srcHandle));

#if defined(ENABLE_SAVE_ENCODE_DATA)
    const std::string dstPath = nnt::codec::util::GetOutputDirectoryPath() + "/" + RemoveFileExtention(entry.name);
    nn::fs::FileHandle dstHandle;
    nn::fs::DeleteFile(dstPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(dstPath.c_str(), 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&dstHandle, dstPath.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
#endif // defined(ENABLE_SAVE_ENCODE_DATA)

    // read input file
    auto pcmData = new int16_t[static_cast<int32_t>(fileSize) / sizeof(int16_t)];
    ASSERT_NE(nullptr, pcmData);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(srcHandle, 0, pcmData, static_cast<int32_t>(fileSize)));

    const int sampleRate = GetSampleRateFromString(entry.name) * 1000;
    ASSERT_TRUE(sampleRate == 8000 || sampleRate == 12000 || sampleRate == 16000 || sampleRate == 24000 || sampleRate == 48000);
    const int channelCount = GetChannelCountFromString(entry.name);
    ASSERT_TRUE(channelCount == 1 || channelCount == 2);
    const int frameSizeInUsec = static_cast<int>(GetFrameFromString(entry.name) * 1000);
    ASSERT_TRUE(frameSizeInUsec == 2500 || frameSizeInUsec == 5000 || frameSizeInUsec == 10000 || frameSizeInUsec == 20000);
    int bitRate = GetBitRateFromString(entry.name);
    ASSERT_GE(bitRate, nn::codec::GetOpusBitRateMin(channelCount));
    ASSERT_LE(bitRate, nn::codec::GetOpusBitRateMax(channelCount));
    const auto codingMode = GetCodingModeFromString(entry.name);
    ASSERT_TRUE(codingMode == nn::codec::OpusCodingMode_Silk || codingMode == nn::codec::OpusCodingMode_Celt);
    DEBUG_PRINT("\tParameter : sampleRate=%6d, channelCount=%2d, frameSize=%5d, bitRate=%6d, codingMode=%d\n", sampleRate, channelCount, frameSizeInUsec, bitRate, codingMode);

    // initialize OpusEncoder
    nn::codec::OpusEncoder encoder;
    EXPECT_FALSE(encoder.IsInitialized());
    void* workBuffer = std::malloc(encoder.GetWorkBufferSize(sampleRate, channelCount));
    ASSERT_NE(nullptr, workBuffer);
    EXPECT_EQ(nn::codec::OpusResult_Success, encoder.Initialize(sampleRate, channelCount, workBuffer, encoder.GetWorkBufferSize(sampleRate, channelCount)));
    EXPECT_TRUE(encoder.IsInitialized());
    EXPECT_EQ(sampleRate, encoder.GetSampleRate());
    EXPECT_EQ(channelCount, encoder.GetChannelCount());
    EXPECT_EQ(nn::codec::OpusBitRateControl_Vbr, encoder.GetBitRateControl());

    // set the bitrate
    encoder.SetBitRate(bitRate);
    EXPECT_EQ(bitRate, encoder.GetBitRate());

    // set the codingMode
    encoder.BindCodingMode(codingMode);
    EXPECT_EQ(codingMode, encoder.GetCodingMode());

    // prepare input buffer to hold pcm data.
    const int frameSampleCount = encoder.CalculateFrameSampleCount(frameSizeInUsec);
    auto input = new int16_t[channelCount * frameSampleCount];
    ASSERT_NE(nullptr, input);

    // prepare output pcm buffer
    const size_t encodeBufferSize = nn::codec::OpusPacketSizeMaximum;
    auto opusData = new uint8_t[encodeBufferSize];
    ASSERT_NE(nullptr, opusData);

#if defined(ENABLE_SAVE_ENCODE_DATA)
    int64_t writeOffset = 0;
#endif // defined(ENABLE_SAVE_ENCODE_DATA)

    // calculate sha1 value for verifying the encoded data.
    nn::crypto::Sha1Generator sha1;
    sha1.Initialize();

    // get the head of pcm data
    const int16_t* p = pcmData;
    int remainedSampleCount = static_cast<int>(fileSize / sizeof(int16_t) / channelCount);

    // encode loop
    // remainedSampleCount が 0 になった後、一回だけ 0 のバッファでエンコードする。
    while (remainedSampleCount >= 0)
    {
        const int sampleCount = std::min<int>(frameSampleCount, remainedSampleCount);
        const std::size_t copyUnitSize = channelCount * sizeof(int16_t);
        const std::size_t copyBufferSize = sampleCount * copyUnitSize;
        const std::size_t copyBlankSize = (frameSampleCount - sampleCount) * copyUnitSize;
        std::memcpy(input, p, copyBufferSize);
        // バッファの末尾の端数と、0 バッファエンコードのため。
        if (copyBlankSize > 0)
        {
            std::memset(input + copyBufferSize / sizeof(int16_t), 0, copyBlankSize);
        }
        size_t encodedDataSize = 0;
        EXPECT_EQ(nn::codec::OpusResult_Success, encoder.EncodeInterleaved(&encodedDataSize, opusData, encodeBufferSize, input, frameSampleCount));
#if defined(ENABLE_DEBUG_PRINT_IN_DETAIL)
        struct nn::codec::detail::OpusPacketInternal::Header header;
        nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, opusData);
        const uint32_t size = header.packetSize;
        const uint32_t finalRange = header.finalRange;
        DEBUG_PRINT("\tEncoding  : size=%08X, rng=%08X, samples=%d, remaining=%d\n", size, finalRange, sampleCount * channelCount, remainedSampleCount);
#endif // #if defined(ENABLE_DEBUG_PRINT_IN_DETAIL)
        sha1.Update(opusData, encodedDataSize);
#if defined(ENABLE_SAVE_ENCODE_DATA)
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(dstHandle, writeOffset, opusData, encodedDataSize, nn::fs::WriteOption()));
        writeOffset += encodedDataSize;
#endif // defined(ENABLE_SAVE_ENCODE_DATA)
        if (0 == remainedSampleCount)
        {
            break;
        }
        p += sampleCount * channelCount;
        remainedSampleCount -= sampleCount;
    }
    delete[] input;
    // finalization
    nn::fs::CloseFile(srcHandle);
#if defined(ENABLE_SAVE_ENCODE_DATA)
    nn::fs::FlushFile(dstHandle);
    nn::fs::CloseFile(dstHandle);
#endif // defined(ENABLE_SAVE_ENCODE_DATA)

    encoder.Finalize();
    EXPECT_FALSE(encoder.IsInitialized());
    std::free(workBuffer);

    delete[] pcmData;
    delete[] opusData;

    // 結果検証
    enum Id
    {
        Calculated,
        Reference,
        IdCount
    };

    const size_t hashSize = nn::crypto::Sha1Generator::HashSize;
    uint8_t hash[IdCount][hashSize];

    // 結果のハッシュ値
    sha1.GetHash(hash[Calculated], hashSize);

    // opus_demo で同じ処理をさせた結果のハッシュ値
    const std::string refPath = nnt::codec::util::GetReferenceDirectoryPath() + "/" + entry.name;
    ASSERT_TRUE(nnt::codec::util::GetHashValueFromFile(hash[Reference], hashSize, refPath));
    // 照合
    EXPECT_TRUE(0 == std::memcmp(hash[Calculated], hash[Reference], hashSize));
    DEBUG_PRINT("\tCalculated: %s\n", nnt::codec::util::GetHashValueString(hash[Calculated], hashSize).c_str());
    DEBUG_PRINT("\tReference : %s\n", nnt::codec::util::GetHashValueString(hash[Reference], hashSize).c_str());
} // NOLINT(readability/fn_size)

}  // namespace

/**
 * @brief       単体エンコーダによるエンコードテストです。
 * @details
 * MultiInstanceEncodeInterleaved がパスするなら不要なので、"DISABLED_" を付しておく。
 */
TEST(OpusEncoderEncodeTest, DISABLED_EncodeInterleaved)
{
    nn::fs::DirectoryHandle directoryHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, nnt::codec::util::GetReferenceDirectoryPath().c_str(), nn::fs::OpenDirectoryMode_File));
    int64_t entryCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));

    auto entries = new nn::fs::DirectoryEntry[static_cast<int32_t>(entryCount)];
    ASSERT_NE(nullptr, entries);
    {
        int64_t tmp;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&tmp, entries, directoryHandle, entryCount));
        EXPECT_EQ(tmp, entryCount);
    }
    nn::fs::CloseDirectory(directoryHandle);
    for (int i = 0; i < entryCount; ++i)
    {
        if (nnt::codec::util::GetFileExtension(entries[i].name, 2) != targetFileExtension)
        {
            continue;
        }
        SCOPED_TRACE(entries[i].name);
        ASSERT_EQ(entries[i].directoryEntryType, nn::fs::DirectoryEntryType_File);
        Tester(&entries[i]);
    }
    delete[] entries;
}

namespace {

// 実行スレッド数 (実行コア数とは別)
const int ThreadCount = 4;

}

/**
 * @brief       複数のエンコーダインスタンスにより、エンコード処理を並走させます。
 */
TEST(OpusEncoderMultipleInstance, MultiInstanceEncodeInterleaved)
{
    nn::fs::DirectoryHandle directoryHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, nnt::codec::util::GetReferenceDirectoryPath().c_str(), nn::fs::OpenDirectoryMode_File));
    int64_t entryCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));

    auto entries = new nn::fs::DirectoryEntry[static_cast<int32_t>(entryCount)];
    ASSERT_NE(nullptr, entries);
    {
        int64_t tmp;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&tmp, entries, directoryHandle, entryCount));
        EXPECT_EQ(tmp, entryCount);
    }
    nn::fs::CloseDirectory(directoryHandle);
    // テストスレッド関連の初期化
    nnt::codec::util::TesterManager manager;
    manager.Initialize(ThreadCount, std::malloc);

    // スレッドを実行するコアを管理するモジュールの初期化＆確認
    nnt::codec::util::CoreSelector coreSelector;
    ASSERT_TRUE(coreSelector.IsInitialized());
    ASSERT_TRUE(coreSelector.GetCoreCount() > 0);

    for (int i = 0; i < entryCount; ++i)
    {
        if (nnt::codec::util::GetFileExtension(entries[i].name, 2) != targetFileExtension)
        {
            continue;
        }
        SCOPED_TRACE(entries[i].name);
        ASSERT_EQ(entries[i].directoryEntryType, nn::fs::DirectoryEntryType_File);
        ASSERT_TRUE(manager.Run(entries[i], Tester, coreSelector.PostIncrement()));
    }
    manager.Finalize(std::free);

    delete[] entries;
}

extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    size_t mountRomCacheBufferSize = 4 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    char* mountRomCacheBuffer = new char[mountRomCacheBufferSize];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] mountRomCacheBuffer;
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom(nnt::codec::util::GetMountPointRom().c_str(), mountRomCacheBuffer, mountRomCacheBufferSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost(nnt::codec::util::GetMountPointHost().c_str(), GetTestDataRootDirectoryPath().c_str()));

    int resultTest = RUN_ALL_TESTS();

    nn::fs::Unmount(nnt::codec::util::GetMountPointRom().c_str());
    nn::fs::Unmount(nnt::codec::util::GetMountPointHost().c_str());

    nnt::Exit(resultTest);
}
