﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib> // std::[malloc,free]

#include <nn/nn_Assert.h>

#include <nn/codec.h>
#include <nn/codec/detail/codec_HardwarePlatform.h>
#include <nn/codec/detail/codec_HardwareOpusCommon-spec.NX.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

#include <nnt.h>
#include <nnt/codecUtil/testCodec_Util.h>
#include <nnt/codecUtil/testCodec_FixtureBase.h>

#include "testCodec_OpusDecoderApiTestParameters.h"

namespace {

template <typename DecoderType>
class OpusDecoderApiTestBase
{
private:
    const int DefaultIterationCount = 10;
    const int32_t DefaultOption = 0;

protected:
    OpusDecoderApiTestBase() NN_NOEXCEPT
        : sampleRate(0)
        , channelCount(0)
        , iteration(DefaultIterationCount)
    {}
    virtual ~OpusDecoderApiTestBase() NN_NOEXCEPT
    {
        Finalize();
    }
    nnt::codec::TestParameterSet GetTestParameterSet(const std::string& valueString) NN_NOEXCEPT
    {
        return GetTestParameterSet(valueString.c_str());
    }
    nnt::codec::TestParameterSet GetTestParameterSet(const char* valueString) NN_NOEXCEPT
    {
        nnt::codec::TestParameterSet parameterSet;
        parameterSet.sampleRate = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "SampleRate");
        parameterSet.channelCount = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "ChannelCount");
        parameterSet.frame = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "Frame");
        return parameterSet;
    }
    void Initialize(const nnt::codec::TestParameterSet& parameterSet, bool executeInitialization) NN_NOEXCEPT
    {
        sampleRate = parameterSet.sampleRate;
        channelCount = parameterSet.channelCount;
        NN_ASSERT(sampleRate != 0);
        NN_ASSERT(channelCount != 0);
        // Initializing
        pDecoder = nnt::codec::CreateOpusDecoder<DecoderType>(DefaultOption);
        NN_ASSERT_NOT_NULL(this->pDecoder);
        decoderWorkBufferSize = pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount);
        decoderWorkBufferAddress = std::malloc(decoderWorkBufferSize);
        ASSERT_NE(nullptr, this->decoderWorkBufferAddress);
        if (executeInitialization)
        {
            ASSERT_EQ(
                nn::codec::OpusResult_Success,
                pDecoder->Initialize(this->sampleRate, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
        }
    }
    virtual void Finalize() NN_NOEXCEPT
    {
        if (nullptr != pDecoder)
        {
            if (this->pDecoder->IsInitialized())
            {
                pDecoder->Finalize();
            }
            nnt::codec::DeleteOpusDecoder(this->pDecoder);
        }
        this->pDecoder = nullptr;
        if (nullptr != decoderWorkBufferAddress)
        {
            std::free(decoderWorkBufferAddress);
        }
        decoderWorkBufferAddress = nullptr;
        decoderWorkBufferSize = 0;
    }
    DecoderType* pDecoder;
    void* decoderWorkBufferAddress;
    std::size_t decoderWorkBufferSize;
    int sampleRate;
    int channelCount;
    int iteration;
};

template <typename T>
class OpusDecoderApiSimpleTest
    : public OpusDecoderApiTestBase<T>
    , public nnt::codec::DecodedDataBufferBase
    , public nnt::codec::EncodedDataBufferBase
    , public ::testing::Test
{
protected:
    typedef T DecoderType;
    ~OpusDecoderApiSimpleTest() NN_NOEXCEPT
    {
        Finalize();
    }
    virtual void Initialize(const char* parameterString, bool executeInitialization) NN_NOEXCEPT
    {
        auto parameterSet = OpusDecoderApiTestBase<T>::GetTestParameterSet(std::string(parameterString));
        // Simple では Frame を指定していないので "0" になっている。
        // Frame を変えて実施するテストは含まれていない。
        // 固定で 20 ms を与えておく。
        parameterSet.frame = 20000;
        OpusDecoderApiTestBase<T>::Initialize(parameterSet, executeInitialization);
        EncodedDataBufferBase::Initialize(parameterSet);
        DecodedDataBufferBase::Initialize(parameterSet);
    }
    virtual void Finalize() NN_NOEXCEPT
    {
        EncodedDataBufferBase::Finalize();
        DecodedDataBufferBase::Finalize();
        OpusDecoderApiTestBase<T>::Finalize();
    }
};

template <typename T>
class OpusDecoderApiSimpleDeathTest
    : public OpusDecoderApiSimpleTest<T>
{
};

template <typename T>
class OpusDecoderApiComplexTest
    : public OpusDecoderApiTestBase<T>
    , public nnt::codec::DecodedDataBufferBase
    , public nnt::codec::EncodedDataBufferBase
    , public ::testing::Test
{
protected:
    typedef T DecoderType;
    virtual ~OpusDecoderApiComplexTest() NN_NOEXCEPT
    {
        Finalize();
    }
    virtual void Initialize(const char* parameterString, bool executeInitialization) NN_NOEXCEPT
    {
        const auto parameterSet = OpusDecoderApiTestBase<T>::GetTestParameterSet(std::string(parameterString));
        Initialize(parameterSet, executeInitialization);
    }
    virtual void Initialize(const nnt::codec::TestParameterSet& parameterSet, bool executeInitialization) NN_NOEXCEPT
    {
        OpusDecoderApiTestBase<T>::Initialize(parameterSet, executeInitialization);

        frame = parameterSet.frame;
        EncodedDataBufferBase::Initialize(parameterSet);
        DecodedDataBufferBase::Initialize(parameterSet);
        // 正常な opus パケットを準備する
        nnt::codec::util::OpusPacketConfiguration config;
        ASSERT_EQ(true, nnt::codec::util::DefaultOpusPacket(&config));
        config.frameInMicroSeconds = frame;
        ASSERT_EQ(true,
            nnt::codec::util::MakeOpusPacket(
                &encodedDataSize,
                encodedDataBuffer,
                encodedDataBufferSize,
                config)
        );
        ASSERT_GT(encodedDataSize, 0u);
    }
    virtual void Finalize() NN_NOEXCEPT
    {
        EncodedDataBufferBase::Finalize();
        DecodedDataBufferBase::Finalize();
        OpusDecoderApiTestBase<T>::Finalize();
    }
    int frame;
};

template <typename T>
class OpusDecoderApiComplexDeathTest
    : public OpusDecoderApiComplexTest<T>
{
};

// Fixture 定義 (型をテストケースで使うためだけの定義)
template <class T>
class HardwareOpusDecoderApiTest : public ::testing::Test
{
protected:
    typedef T DecoderType;
};

}  // anonymous namespace

/**
 * @brief       インスタンス化する型を列挙します。
 */
typedef ::testing::Types<nn::codec::OpusDecoder, nn::codec::HardwareOpusDecoder> CommonTestImplementations;

TYPED_TEST_CASE(OpusDecoderApiSimpleTest, CommonTestImplementations);
TYPED_TEST_CASE(OpusDecoderApiComplexTest, CommonTestImplementations);

TYPED_TEST(OpusDecoderApiSimpleTest, InitializeFinalize)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        for (auto k = 0; k < this->iteration; ++k)
        {
            // 戻値を確認する。
            EXPECT_EQ(
                nn::codec::OpusResult_Success,
                this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
            // 戻値はないので、停止しないことを確認する。
            this->pDecoder->Finalize();
        }
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleTest, GetWorkBufferSize)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        for (auto k = 0; k < this->iteration; ++k)
        {
            // GetWorkBufferSize() はいつでも有効
            const auto workBufferSize = this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount);
            EXPECT_GT(workBufferSize, 0u);
            ASSERT_EQ(
                nn::codec::OpusResult_Success,
                this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
            // 取得したサイズをリファレンスとして照合する。
            EXPECT_EQ(workBufferSize, this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount));
            this->pDecoder->Finalize();
            EXPECT_EQ(workBufferSize, this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount));
        }
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleTest, GetSampleRate)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
        );
        EXPECT_EQ(this->sampleRate, this->pDecoder->GetSampleRate());
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleTest, GetChannelCount)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
        );
        EXPECT_EQ(this->channelCount, this->pDecoder->GetChannelCount());
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleTest, Initialized)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        for (auto k = 0; k < this->iteration; ++k)
        {
            EXPECT_EQ(false, this->pDecoder->IsInitialized());
            ASSERT_EQ(
                nn::codec::OpusResult_Success,
                this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
            EXPECT_EQ(true, this->pDecoder->IsInitialized());
            this->pDecoder->Finalize();
            EXPECT_EQ(false, this->pDecoder->IsInitialized());
        }
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleTest, DoubleInitializing)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, true);
        SCOPED_TRACE(parameterString);
        auto pAnotherDecoder = nnt::codec::CreateOpusDecoder<typename TestFixture::DecoderType>(0);
#if defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR) // ハードウェアアクセラレータを持っている場合
        // TransferMemory で失敗する (既にプロセスに権利を委譲している領域に対して再度 TransferMemory することになる)
        EXPECT_DEATH_IF_SUPPORTED(
            pAnotherDecoder->Initialize(
                this->pDecoder->GetSampleRate(),
                this->pDecoder->GetChannelCount(),
                this->decoderWorkBufferAddress,
                this->decoderWorkBufferSize), "");
#else  // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
        // ワークバッファとインスタンスの紐付けはないので、使用中のワークバッファを別インスタンスの Initialize() に指定しても通る。
        EXPECT_EQ(
            nn::codec::OpusResult_Success,
            pAnotherDecoder->Initialize(
                this->pDecoder->GetSampleRate(),
                this->pDecoder->GetChannelCount(),
                this->decoderWorkBufferAddress,
                this->decoderWorkBufferSize)
        );
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
        nnt::codec::DeleteOpusDecoder(pAnotherDecoder);
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleTest, Reset)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        for (auto k = 0; k < this->iteration; ++k)
        {
            ASSERT_EQ(
                nn::codec::OpusResult_Success,
                this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
            this->pDecoder->Reset();
            this->pDecoder->Finalize();
        }
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiComplexTest, DecodeInterleaved)
{
    for (const auto parameterString : ComplexTestParameterListStrings)
    {
        this->Initialize(parameterString, true);
        SCOPED_TRACE(parameterString);
        for (auto k = 0; k < this->iteration; ++k)
        {
            const auto outputSampleCountCorrect = nn::codec::CalculateOpusFrameSampleCount(this->frame, this->pDecoder->GetSampleRate());
            std::size_t consumed;
            int outputSampleCount;
            ASSERT_EQ(
                nn::codec::OpusResult_Success,
                this->pDecoder->DecodeInterleaved(
                    &consumed,
                    &outputSampleCount,
                    this->decodedDataBuffer,
                    this->decodedDataBufferSize,
                    this->encodedDataBuffer,
                    this->encodedDataSize)
            );
            EXPECT_GT(consumed, 0u);
            EXPECT_EQ(outputSampleCountCorrect, outputSampleCount);

            // Providing illegal sizes
            EXPECT_EQ(
                nn::codec::OpusResult_InsufficientOpusBuffer,
                this->pDecoder->DecodeInterleaved(
                    &consumed, &outputSampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, 0u)
            );

            // 出力バッファサイズは、必要サイズ以上であれば良い (以下では、指定サイズだけを大きくしている。バッファは大きくなっていない)。
            EXPECT_EQ(
                nn::codec::OpusResult_Success,
                this->pDecoder->DecodeInterleaved(
                    &consumed, &outputSampleCount, this->decodedDataBuffer, this->decodedDataBufferSize + 1, this->encodedDataBuffer, this->encodedDataSize)
            );
            // 必要サイズ未満
            EXPECT_EQ(
                nn::codec::OpusResult_InsufficientPcmBuffer,
                this->pDecoder->DecodeInterleaved(
                    &consumed, &outputSampleCount, this->decodedDataBuffer, this->decodedDataBufferSize - 1, this->encodedDataBuffer, this->encodedDataSize)
            );
            // 必要サイズ未満
            // 先頭でない位置を指定 (サイズ指定は正確な値を指定するために帳尻合わせをしている)
            using OpusHeader = nn::codec::detail::OpusPacketInternal::Header;
            OpusHeader header;
            nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, this->encodedDataBuffer);
            const auto headerSize = sizeof(OpusHeader);
            EXPECT_EQ(
                nn::codec::OpusResult_InsufficientOpusBuffer,
                this->pDecoder->DecodeInterleaved(
                    &consumed, &outputSampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, headerSize + header.packetSize - 1)
            );
        }
        this->Finalize();
    }
}


#if !defined(NN_SDK_BUILD_RELEASE)

TYPED_TEST_CASE(OpusDecoderApiSimpleDeathTest, CommonTestImplementations);
TYPED_TEST_CASE(OpusDecoderApiComplexDeathTest, CommonTestImplementations);

TYPED_TEST(OpusDecoderApiSimpleDeathTest, GetWorkBufferSize)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        // Unsupported channel count.
        // "+1/-1" だとテストにならない。
        // channelCount=2 だと 3/1、channelCount=1 だと 2/0 となり、正常値を含んでしまっている。
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount - 2), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount + 2), "");

        // Unsupported sample rate.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate - 1, this->channelCount), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate + 1, this->channelCount), "");
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleDeathTest, FailedInInitializing)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        // Unsupported channel count.
        // "+1/-1" だとテストにならない。
        // channelCount=2 だと 3/1、channelCount=1 だと 2/0 となり、正常値を含んでしまっている。
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount - 2, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount + 2, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");

        // Unsupported sample rate.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate - 1, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate + 1, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");

        // nullptr
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount, nullptr, this->decoderWorkBufferSize), "");

        // Smaller work buffer size
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->decoderWorkBufferAddress, this->decoderWorkBufferSize - 1), "");
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleDeathTest, InitializeUnderInitializedState)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, true);
        SCOPED_TRACE(parameterString);
        void* tmp = std::malloc(this->decoderWorkBufferSize);
        // Initialization under "Initialized" state.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount, tmp, this->decoderWorkBufferSize), "");
        std::free(tmp);
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleDeathTest, CallingUnderUninitializedState)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        // Calling some APIs requiring "Initialized" state.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetSampleRate(), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetChannelCount(), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Reset(), "");
        std::size_t consumed;
        int sampleCount;
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                &consumed, &sampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, this->encodedDataSize), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Finalize(), "");
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiSimpleDeathTest, CallingUnderFinalizedState)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, true);
        SCOPED_TRACE(parameterString);
        this->pDecoder->Finalize();

        // Calling some APIs requiring "Initialized" state.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetSampleRate(), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetChannelCount(), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Reset(), "");
        std::size_t consumed;
        int sampleCount;
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                &consumed, &sampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, this->encodedDataSize), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Finalize(), "");
        this->Finalize();
    }
}

TYPED_TEST(OpusDecoderApiComplexDeathTest, DecodeInerleavedUnderInitializedState)
{
    for (const auto parameterString : ComplexTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        std::size_t consumed;
        int sampleCount;
        // Providing "nullptr"
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                nullptr, &sampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, this->encodedDataSize), "");
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                &consumed, nullptr, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, this->encodedDataSize), "");
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                &consumed, &sampleCount, nullptr, this->decodedDataBufferSize, this->encodedDataBuffer, this->encodedDataSize), "");
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                &consumed, &sampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, nullptr, this->encodedDataSize), "");
        // ワークメモリをクリアしてデコードすると "error: SEH exception with code 0xc0000005 thrown in the test body." が起こる (@Windows)。
        std::memset(this->decoderWorkBufferAddress, 0, this->decodedDataBufferSize);
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                &consumed, &sampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, this->encodedDataSize), "");
        this->Finalize();
    }
}

#endif // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       インスタンス化する型を列挙します。
 */
typedef ::testing::Types<nn::codec::HardwareOpusDecoder> Implementations;

TYPED_TEST_CASE(HardwareOpusDecoderApiTest, Implementations);

/**
 * @brief       複数のインスタンスを初期化します。
 *
 * @brief
 * ハードウェアアクセラレータを搭載している環境では、インスタンス初期化の背後でシステム
 * プロセスにもインスタンスを作成している。
 * 作成できるインスタンス数に上限を設けている。
 * 一方で、ハードウェアアクセラレータを搭載していない環境では、インスタンス作成において
 * このようなシステムの都合による制約はない。
 * よって、実装分岐をしている。
 * 実装分岐をなくすのがベストだが、そのためだけにハードウェアアクセラレータを搭載してい
 * ない環境向けにセッション管理機構を開発するコストを払う意味は、現時点ではない。
 *
 */
TYPED_TEST(HardwareOpusDecoderApiTest, InitializeMultipleInstances)
{
    // テストパラメータ定義
    const int sampleRates[] =
    {
         8000,
        12000,
        16000,
        24000,
        48000
    };

    const int channelCounts[] =
    {
        1,
        2
    };
    for (auto sampleRate : sampleRates)
    {
        for (auto channelCount : channelCounts)
        {
            const std::size_t sessionCountMax = nn::codec::detail::HardwareOpusDecoderIpcConstants_SessionCountMax;
            const auto sessionCountUpperLimit = sessionCountMax + 1; // + 1 は上限値をオーバーさせるため。
            typename TestFixture::DecoderType* decoders[sessionCountUpperLimit] = { nullptr, };
            void* workBuffers[sessionCountUpperLimit] = { nullptr, };
            for (auto k = 0u; k < sessionCountUpperLimit; ++k)
            {
                decoders[k] = nnt::codec::CreateOpusDecoder<typename TestFixture::DecoderType>(0);
                ASSERT_NE(nullptr, decoders[k]);
                const auto workBufferSize = decoders[k]->GetWorkBufferSize(sampleRate, channelCount);
                workBuffers[k] = std::malloc(workBufferSize);
                ASSERT_NE(nullptr, workBuffers[k]);
                const auto result = decoders[k]->Initialize(sampleRate, channelCount, workBuffers[k], workBufferSize);
#if defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR) // ハードウェアアクセラレータを持っている場合
                const auto correct = k < sessionCountMax ? nn::codec::OpusResult_Success : nn::codec::OpusResult_OutOfResource;
#else                                           // ハードウェアアクセラレータを持っていない場合、セッションの概念がなく、制約もない。
                const auto correct = nn::codec::OpusResult_Success;
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
                EXPECT_EQ(correct, result);
            }
            for (auto k = 0u; k < sessionCountUpperLimit; ++k)
            {
                if (nullptr != decoders[k] && decoders[k]->IsInitialized())
                {
                    decoders[k]->Finalize();
                }
                if (nullptr != workBuffers[k])
                {
                    std::free(workBuffers[k]);
                    workBuffers[k] = nullptr;
                }
            }
        }
    }
}

