﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <string>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <direct.h>  // _getcwd
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)

#include <nnt.h>
#include <nnt/nnt_Argument.h.>
#include <nnt/codecUtil/testCodec_Util.h>

#include <nn/codec.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

// opus 内の関数
extern "C" int OpusDemo(int argc, const char **_argv); // Externals/libopus/src/opus_demo.c
extern "C" int OpusCompare(int argc, const char **_argv); // Externals/libopus/src/opus_compare.c
extern "C" int TestOpusApi(int argc, const char **_argv); // Externals/libopus/tests/test_opus_api.c
extern "C" int TestOpusDecode(int argc, const char **_argv); // Externals/libopus/tests/test_opus_decode.c
extern "C" int TestOpusEncode(int argc, const char **_argv); // Externals/libopus/tests/test_opus_encode.c
extern "C" int TestOpusPadding(int argc, const char **_argv); // Externals/libopus/tests/test_opus_padding.c

namespace {

const int SupportedSampleRates[] = {
     8000,
    12000,
    16000,
    24000,
    48000
};

const char* OpusDemo_ProgramPath = "tests/opus_demo";
const char* OpusDemo_InstructionDecode = "-d";
const char* OpusDemo_ChannelMono = "1";
const char* OpusDemo_ChannelStereo = "2";

const char* OpusCompare_ProgramPath = "tests/opus_compare";
const char* OpusCompare_ChannelStereo = "-s";
const char* OpusCompare_OptionRate = "-r";

const int OpusStatus_Success = 0;

const std::string GetTestDataRootDirectoryPath() NN_NOEXCEPT
{
    return nnt::codec::util::GetCodecTestBinariesPath()  + "\\testCodec_OpusCodec";
}

std::string RemoveFileExtention(const std::string& path)
{
    std::string::size_type position = path.find_last_of(".");
    return (position == std::string::npos) ? path : path.substr(0, position);
}

int RunOpusDemoDecoder(const std::string& outputPath, const std::string& inputPath, const std::string& sampleRate, int channelCount)
{
    const int argc = 6;
    const char* argv[argc];

    argv[0] = OpusDemo_ProgramPath; //プログラムパス、適当な文字列を与えておくが、動作には影響なし。
    argv[1] = OpusDemo_InstructionDecode;
    argv[2] = sampleRate.c_str();
    argv[3] = (channelCount == 1) ? OpusDemo_ChannelMono : OpusDemo_ChannelStereo;
    argv[4] = inputPath.c_str();
    argv[5] = outputPath.c_str();

    return OpusDemo(argc, argv);
}

int RunOpusCompare(const std::string& referencePath, const std::string& resultPath, const std::string& sampleRate, int channelCount)
{
    const int MaximumArgc = 6;
    const char* argv[MaximumArgc];

    int argc = 0;

    argv[argc++] = OpusCompare_ProgramPath; //プログラムパス、適当な文字列を与えておくが、動作には影響なし。
    if (2 == channelCount)
    {
        argv[argc++] = OpusCompare_ChannelStereo;
    }
    argv[argc++] = OpusCompare_OptionRate;
    argv[argc++] = sampleRate.c_str();
    argv[argc++] = referencePath.c_str();
    argv[argc++] = resultPath.c_str();

    return OpusCompare(argc, argv);
}

void RunDecodeAndCompare(const std::string& entry, int sampleRate, int channelCount)
{
    NN_SDK_ASSERT(channelCount == 1 || channelCount == 2);
    NN_SDK_ASSERT(sampleRate == 8000 || sampleRate == 12000 || sampleRate == 16000 || sampleRate == 24000 || sampleRate == 48000);

    const std::string srcPath = nnt::codec::util::GetMountPointRom() + ":/Inputs/" + entry;
    const std::string valPath = nnt::codec::util::GetMountPointRom() + ":/References/" + RemoveFileExtention(entry) + ".dec";
    const std::string dstPath = nnt::codec::util::GetMountPointHost()
        + ":/Outputs/" + RemoveFileExtention(entry)
        + "_Fs" + std::to_string(sampleRate)
        + "_Ch" + std::to_string(channelCount) + ".raw";

    nn::fs::DeleteFile(dstPath.c_str());

    const int decodeStatus = RunOpusDemoDecoder(dstPath, srcPath, std::to_string(sampleRate), channelCount);
    EXPECT_EQ(decodeStatus, OpusStatus_Success);
    if (OpusStatus_Success == decodeStatus)
    {
        EXPECT_EQ(RunOpusCompare(valPath, dstPath, std::to_string(sampleRate), channelCount), OpusStatus_Success);
    }
}

}  // namespace

/**
 * @brief       Normal case tests
 */
TEST(OpusDecoderWithVectors, NormalCase)
{
    size_t mountRomCacheBufferSize = 4 * 1024;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    char* mountRomCacheBuffer = new char[mountRomCacheBufferSize];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] mountRomCacheBuffer;
    };

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountRom(nnt::codec::util::GetMountPointRom().c_str(), mountRomCacheBuffer, mountRomCacheBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountHost(nnt::codec::util::GetMountPointHost().c_str(), GetTestDataRootDirectoryPath().c_str()));

    nn::fs::DirectoryHandle directoryHandle;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, (nnt::codec::util::GetMountPointRom() + ":/Inputs/").c_str(), nn::fs::OpenDirectoryMode_File));
    int64_t entryCount = 0;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));

    auto entries = new nn::fs::DirectoryEntry[static_cast<uint32_t>(entryCount)];
    {
        int64_t tmp;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&tmp, entries, directoryHandle, entryCount));
        EXPECT_EQ(tmp, entryCount);
    }
    for (int i = 0; i < entryCount; ++i)
    {
        const auto& entry = entries[i];
        // open input/output files
        EXPECT_EQ(entry.directoryEntryType, nn::fs::DirectoryEntryType_File);
        for (const int sampleRate : SupportedSampleRates)
        {
            RunDecodeAndCompare(entry.name, sampleRate, 1); // Test for Mono.
            RunDecodeAndCompare(entry.name, sampleRate, 2); // Test for Stereo
        }
    }
    delete[] entries;

    nn::fs::CloseDirectory(directoryHandle);
}

TEST(OpusApiTests, NormalCase)
{
    EXPECT_EQ(TestOpusApi(0, NULL), OpusStatus_Success);
    EXPECT_EQ(TestOpusPadding(0, NULL), OpusStatus_Success);
}

TEST(OpusDecodeTests, NormalCase)
{
    EXPECT_EQ(TestOpusDecode(0, NULL), OpusStatus_Success);
}

// Keep disabled until confirmed that the test went well.
TEST(OpusEncodeTests, NormalCase)
{
    EXPECT_EQ(TestOpusEncode(0, NULL), OpusStatus_Success);
}

