﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <string>
#include <random>

#include <nn/nn_Assert.h>
#include <nn/codec.h>
#include <nn/codec/codec_OpusEncoder.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h.>
#include <nnt/codecUtil/testCodec_Util.h>
#include <nnt/codecUtil/testCodec_FixtureBase.h>

#include "testCodec_SingleStreamTestParameters.h"
#include "testCodec_MultiStreamTestParameters.h"

#define DebugPrint(...)

#ifndef DebugPrint
#define DebugPrint(...) NN_LOG(__VA_ARGS__)
#endif

struct OpusDecoder;
struct OpusMSDecoder;

struct OpusEncoder;
struct OpusMSEncoder;

namespace {

nn::os::EventType g_FinishedDecoding;
nn::os::EventType g_FinishedEncoding;

const std::size_t KiB = 1024;
const std::size_t MiB = KiB * KiB;
const std::size_t ThreadStackSize = 256 * KiB;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackForDecoder[ThreadStackSize];
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackForEncoder[ThreadStackSize];
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackTemporary[ThreadStackSize];

const int ChannelCountMax = 255;

const std::size_t PcmDataBufferSampleCount = 960 * 3; // 60ms 対応
int16_t g_PcmDataBuffer[PcmDataBufferSampleCount * ChannelCountMax];

// 足りなかったようなので (本当?) 2 倍にしておく (SIGLO-47342)。
const std::size_t EncodedDataBufferSize = 2 * ChannelCountMax * nn::codec::OpusPacketSizeMaximum; // Here, "ChannelCount" is meaning "StreamCount".
uint8_t g_EncodedDataBuffer[EncodedDataBufferSize];

const std::size_t DecodedDataBufferSampleCount = 960 * 3; // 60ms 対応
int16_t g_DecodedDataBuffer[DecodedDataBufferSampleCount * ChannelCountMax];

std::size_t g_EncodedDataSize = 0;

const std::size_t WorkBufferSize = 32 * MiB;
char* g_WorkBufferForOpusDecoder[WorkBufferSize];
char* g_WorkBufferForOpusEncoder[WorkBufferSize];

// Horizon 環境の Develop/Debug ビルドにて、スレッド作成時にこの値で埋められる。
// Release ビルドでは埋められない。
const uint8_t MemoryDefaultValue = 0x58;

const int ProcessingIteration = 3;

struct DecoderSet
{
    void* pDecoder;
    int iteration;
};

struct EncoderSet
{
    int frame;
    int bitRate;
    nn::codec::OpusCodingMode codingMode;
    int streamCount;
    int coupledStreamCount;
    void* pEncoder;
    int iteration;
};

void Reverse(void* _pDestination, const void* _pSource, std::size_t size) NN_NOEXCEPT
{
    auto pSource = static_cast<const uint8_t*>(_pSource);
    auto pDestination = static_cast<uint8_t*>(_pDestination);
    for (auto k = 0u; k < size; ++k)
    {
        pDestination[k] = pSource[size - k - 1];
    }
}

std::size_t AnalyzeMemoryUsage(std::size_t pSizes[], std::size_t pSizesCount, const void* stackBase, std::size_t stackSize, uint8_t defaultValue) NN_NOEXCEPT
{
    const uint8_t* ptr = static_cast<const uint8_t*>(stackBase);
    bool used = false;
    int used_end = static_cast<int>(stackSize) - 1;
    int unused_end = 0;
    int fill_value_count = 0;
    const int consec_bytes = 32;
    std::size_t total_used = 0;
    std::size_t total_unused = 0;
    auto idx = 0u;
    std::memset(pSizes, 0, sizeof(std::size_t) * pSizesCount);
    for (int k = used_end; k >= 0; --k)
    {
        if (ptr[k] != defaultValue)
        {
            if(false == used)
            {
                if (unused_end > 0)
                {
                    const int unused_start = k + 1;
                    const int unused_size = unused_end - unused_start + 1;
                    DebugPrint("* Unused: %d\n", unused_size);
                    DebugPrint("        %02x %02x %02x %02x | %02x %02x %02x %02x ... ",
                        ptr[unused_start - 4], ptr[unused_start - 3], ptr[unused_start - 2], ptr[unused_start - 1], ptr[unused_start], ptr[unused_start + 1], ptr[unused_start + 2], ptr[unused_start + 3]);
                    DebugPrint("%02x %02x %02x %02x | %02x %02x %02x %02x (%d-%d) (%08x-%08x)\n",
                        ptr[unused_end - 3], ptr[unused_end - 2], ptr[unused_end - 1], ptr[unused_end], ptr[unused_end + 1], ptr[unused_end + 2], ptr[unused_end + 3], ptr[unused_end + 4],
                        unused_start, unused_end, &ptr[unused_start], &ptr[unused_end]);
                    total_unused += unused_size;
                    if (idx < pSizesCount)
                    {
                        pSizes[idx] = unused_size;
                        ++idx;
                    }
                }
                used = true;
                used_end = k;
            }
            fill_value_count = 0;
        }
        else // if(ptr[k] == defaultValue)
        {
            if (true == used)
            {
                fill_value_count++;
                if (fill_value_count == consec_bytes)
                {
                    const int used_start = k + consec_bytes;
                    const int used_size = used_end - used_start + 1;
                    fill_value_count = 0;
                    used = false;
                    DebugPrint("* Used: %d\n", used_size);
                    DebugPrint("        %02x %02x %02x %02x | %02x %02x %02x %02x ... ",
                        ptr[used_start - 4], ptr[used_start - 3], ptr[used_start - 2], ptr[used_start - 1], ptr[used_start], ptr[used_start + 1], ptr[used_start + 2], ptr[used_start + 3]);
                    DebugPrint("%02x %02x %02x %02x | %02x %02x %02x %02x (%08x-%08x)\n",
                        ptr[used_end - 3], ptr[used_end - 2], ptr[used_end - 1], ptr[used_end], ptr[used_end + 1], ptr[used_end + 2], ptr[used_end + 3], ptr[used_end + 4],
                        used_start, used_end);
                    total_used += used_size;
                    unused_end = used_start - 1;
                    if (idx < pSizesCount)
                    {
                        pSizes[idx] = used_size;
                        ++idx;
                    }
                }
            }
        }
    }
    {
        const int used_start = 0;
        NN_UNUSED(used_start);
        const int used_size = used_end + 1;
        DebugPrint("* Used: %d\n", used_size);
        DebugPrint("                    | %02x %02x %02x %02x ... ",
            ptr[used_start], ptr[used_start + 1], ptr[used_start + 2], ptr[used_start + 3]);
        DebugPrint("%02x %02x %02x %02x | %02x %02x %02x %02x (%08x-%08x)\n",
            ptr[used_end - 3], ptr[used_end - 2], ptr[used_end - 1], ptr[used_end], ptr[used_end + 1], ptr[used_end + 2], ptr[used_end + 3], ptr[used_end + 4],
            used_start, used_end);
        total_used += used_size;
                    if (idx < pSizesCount)
        {
            pSizes[idx] = used_size;
            ++idx;
        }
    }
    DebugPrint("  ==> Stack consumed: %d (%08x) + %d (%08x) \n", total_used, total_used, total_unused, total_unused);
    return total_used + total_unused;
}

template <typename DecoderType>
void DecoderFunction(void* arg) NN_NOEXCEPT
{
    // Decoding
    NN_UNUSED(arg);
    if (nullptr == arg)
    {
        nn::os::SignalEvent(&g_FinishedDecoding);
        return ;
    }
    auto pDecoderSet = static_cast<struct DecoderSet*>(arg);
    auto pDecoder = static_cast<nnt::codec::NativeOpusDecoder<DecoderType>*>(pDecoderSet->pDecoder);
    NN_ASSERT_NOT_EQUAL(pDecoder->GetChannelCount(), 0);
    NN_ASSERT_NOT_EQUAL(pDecoder->GetSampleRate(), 0);
    for (auto k = 0; k < pDecoderSet->iteration; ++k)
    {
        std::size_t consumed;
        int sampleCount;
        nn::os::WaitEvent(&g_FinishedEncoding);
//        nnt::codec::util::Dump(g_EncodedDataBuffer, g_EncodedDataSize, 32);
//        nnt::codec::util::DumpOpusPacket(g_EncodedDataBuffer, g_EncodedDataSize, pDecoder->GetStreamCount());
        EXPECT_EQ(
            nn::codec::OpusResult_Success,
            pDecoder->DecodeInterleaved(
                &consumed,
                &sampleCount,
                g_DecodedDataBuffer,
                sizeof(g_DecodedDataBuffer),
                g_EncodedDataBuffer,
                g_EncodedDataSize)
        );
        // 全設定、かつ最長フレームのデコード (EncodeInterleaved() の結果はほぼ無視)。
        // 全設定に関しては RFC6716 の "3.1.  The TOC Byte"を、
        // フレームサイズに関しては、RFC6716 の "3.2.1.  Frame Length Coding" を参照。
        for (auto config = 0u; config < 64; ++config)
        {
            using OpusHeader = nn::codec::detail::OpusPacketInternal::Header;
            const auto opusPacketOffset = sizeof(OpusHeader);
            const auto streamCount = pDecoder->GetStreamCount();
            g_EncodedDataSize = streamCount * 4 - 1 + opusPacketOffset;
            const OpusHeader header =
            {
                static_cast<uint32_t>(g_EncodedDataSize - opusPacketOffset),
                static_cast<uint32_t>(0)
            };
            nn::codec::detail::OpusPacketInternal::SetHeaderInPacket(g_EncodedDataBuffer, header);
            auto p = g_EncodedDataBuffer + opusPacketOffset;
            for (auto s = 0; s < streamCount; ++s)
            {
                p[0] = static_cast<uint8_t>(config << 2);
                if (s != streamCount - 1)
                {
                    p[1] = 2;
                    ++p;
                }
                p[1] = 255;
                p[2] = 255;
                p += 3;
            }
            EXPECT_EQ(
                nn::codec::OpusResult_Success,
                pDecoder->DecodeInterleaved(
                    &consumed,
                    &sampleCount,
                    g_DecodedDataBuffer,
                    sizeof(g_DecodedDataBuffer),
                    g_EncodedDataBuffer,
                    g_EncodedDataSize)
            );
        }
        nn::os::SignalEvent(&g_FinishedDecoding);
    }
}

template <typename EncoderType>
void EncoderFunction(void* arg)  NN_NOEXCEPT
{
    // Encoding
    NN_UNUSED(arg);
    if (nullptr == arg)
    {
        return ;
    }
    auto pEncoderSet = static_cast<struct EncoderSet*>(arg);
    auto pEncoder = static_cast<nnt::codec::NativeOpusEncoder<EncoderType>*>(pEncoderSet->pEncoder);
    NN_ASSERT_NOT_EQUAL(pEncoder->GetChannelCount(), 0);
    NN_ASSERT_NOT_EQUAL(pEncoder->GetSampleRate(), 0);
    pEncoder->SetBitRate(pEncoderSet->bitRate);
    //pEncoder->SetBitRateControl(nn::codec::OpusBitRateControl_Cbr);
    pEncoder->BindCodingMode(pEncoderSet->codingMode);
    const auto encodeSampleCount = pEncoder->CalculateFrameSampleCount(pEncoderSet->frame);
    std::mt19937 mt(static_cast<int>(nn::os::GetSystemTick().GetInt64Value()));
    const auto a = std::numeric_limits<uint16_t>::max();
    const auto b = std::numeric_limits<int16_t>::min();
    for (auto k = 0; k < pEncoderSet->iteration; ++k)
    {
        for (auto i = 0u; i < PcmDataBufferSampleCount * pEncoder->GetChannelCount(); ++i)
        {
            g_PcmDataBuffer[i] = (mt() % a) + b;
        }
        nn::os::WaitEvent(&g_FinishedDecoding);
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            pEncoder->EncodeInterleaved(
                &g_EncodedDataSize,
                g_EncodedDataBuffer,
                EncodedDataBufferSize,
                g_PcmDataBuffer,
                encodeSampleCount)
        );
        nn::codec::OpusCodingMode codingMode;
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            nn::codec::detail::OpusPacketInternal::GetOpusPacketCodingMode(&codingMode, g_EncodedDataBuffer, g_EncodedDataSize)
        );
        ASSERT_EQ(pEncoderSet->codingMode, codingMode);
        nn::os::SignalEvent(&g_FinishedEncoding);
    }
}

template <typename D, typename E>
class NativeOpusCodecTestBase
    : public nnt::codec::NativeOpusEncoderBase<E>
    , public nnt::codec::NativeOpusDecoderBase<D>
    , public ::testing::WithParamInterface<const char*>
{
protected:

    using DecoderType = D;
    using EncoderType = E;
    using DecoderClass = nnt::codec::NativeOpusDecoderBase<DecoderType>;
    using EncoderClass = nnt::codec::NativeOpusEncoderBase<EncoderType>;

    NativeOpusCodecTestBase() NN_NOEXCEPT
        : sampleRate(0)
        , channelCount(0)
        , frame(0)
        , bitRate(0)
        , codingMode(nn::codec::OpusCodingMode_Auto)
        , decoding()
        , encoding()
        , decoderSet()
        , encoderSet()
    {}

    nnt::codec::TestParameterSet GetTestParameterSet(const char* valueString) NN_NOEXCEPT
    {
        const int _sampleRate = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "SampleRate");
        const int _channelCount = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "ChannelCount");
        const int _frame = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "Frame");
        const int _bitRate = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "BitRate");
        const int _codingMode = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "CodingMode");
        int _streamCount = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "StreamCount");
        int _coupledStreamCount = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "CoupledStreamCount");
        // streamCount が "0" の場合、SingleStream と見なして設定する。
        if (0 == _streamCount)
        {
            _streamCount = 1;
            _coupledStreamCount = _channelCount - 1;
        }
        nnt::codec::TestParameterSet set =
        {
            _sampleRate,
            _channelCount,
            _frame,
            _bitRate,
            static_cast<nn::codec::OpusCodingMode>(_codingMode),
            _streamCount,
            _coupledStreamCount
        };
        return set;
    }

    virtual void Initialize(const nnt::codec::TestParameterSet& parameterSet) NN_NOEXCEPT
    {
        sampleRate = parameterSet.sampleRate;
        channelCount = parameterSet.channelCount;
        frame = parameterSet.frame;
        bitRate = parameterSet.bitRate;
        codingMode = parameterSet.codingMode;
        NN_ASSERT(
            codingMode == nn::codec::OpusCodingMode_Silk
            // || codingMode == nn::codec::OpusCodingMode_Auto
            || codingMode == nn::codec::OpusCodingMode_Hybrid // Should be removed
            || codingMode == nn::codec::OpusCodingMode_Celt
        );
        streamCount = parameterSet.streamCount;
        coupledStreamCount = parameterSet.coupledStreamCount;
        NN_ASSERT_GREATER_EQUAL(channelCount, streamCount + coupledStreamCount);
        EncoderClass::Initialize(parameterSet, false);
        std::memset(g_WorkBufferForOpusEncoder, MemoryDefaultValue, WorkBufferSize);
        uint8_t channelMappingTable[256];
        for (auto channel = 0; channel < channelCount; ++channel)
        {
            channelMappingTable[channel] = static_cast<uint8_t>(channel);
        }
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            EncoderClass::encoder.Initialize(
                sampleRate,
                channelCount,
                streamCount,
                coupledStreamCount,
                channelMappingTable,
                g_WorkBufferForOpusEncoder,
                WorkBufferSize
            )
        );
        DecoderClass::Initialize(parameterSet, false);
        std::memset(g_WorkBufferForOpusDecoder, MemoryDefaultValue, WorkBufferSize);
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            DecoderClass::decoder.Initialize(
                sampleRate,
                channelCount,
                streamCount,
                coupledStreamCount,
                channelMappingTable,
                g_WorkBufferForOpusDecoder,
                WorkBufferSize
            )
        );
    }

    virtual void Finalize() NN_NOEXCEPT
    {
        EncoderClass::Finalize();
        DecoderClass::Finalize();
    }

    int sampleRate;
    int channelCount;
    int frame;
    int bitRate;
    int streamCount;
    int coupledStreamCount;
    nn::codec::OpusCodingMode codingMode;

    nn::os::ThreadType decoding;
    nn::os::ThreadType encoding;
    struct DecoderSet decoderSet;
    struct EncoderSet encoderSet;
};

class OpusCodecSingleStreamTest
    : public NativeOpusCodecTestBase<::OpusDecoder, ::OpusEncoder>
    , public ::testing::Test
{
protected:

    virtual void SetUp() NN_NOEXCEPT
    {
        const auto parameterSet = GetTestParameterSet(GetParam());
        NativeOpusCodecTestBase::Initialize(parameterSet);
        Initialize(parameterSet);
    }
    virtual void TearDown() NN_NOEXCEPT
    {
        NativeOpusCodecTestBase::Finalize();
    }
};

class OpusCodecMultiStreamTest
    : public NativeOpusCodecTestBase<::OpusMSDecoder, ::OpusMSEncoder>
    , public ::testing::Test
{
protected:

    virtual void SetUp() NN_NOEXCEPT
    {
        const auto parameterSet = GetTestParameterSet(GetParam());
        NativeOpusCodecTestBase::Initialize(parameterSet);
        Initialize(parameterSet);
    }
    virtual void TearDown() NN_NOEXCEPT
    {
        NativeOpusCodecTestBase::Finalize();
    }
};

/**
 * @brief       メモリ使用量計測の基底関数
 */
template <typename EncoderType, typename DecoderType>
void RunMemoryUsageTestForOpusCodec(
    nnt::codec::NativeOpusEncoder<EncoderType>& encoder,
    nnt::codec::NativeOpusDecoder<DecoderType>& decoder,
    int frame,
    int bitRate,
    nn::codec::OpusCodingMode codingMode,
    int iteration,
    const char* paramString
) NN_NOEXCEPT
{
    SCOPED_TRACE(paramString);
    //------------------------------
    // Initialize
    //------------------------------
    nn::os::InitializeEvent(&g_FinishedDecoding, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_FinishedEncoding, false, nn::os::EventClearMode_AutoClear);

    nn::os::SignalEvent(&g_FinishedDecoding); // Signal しておく

    nn::os::ThreadType decoding;
    nn::os::ThreadType encoding;
    struct DecoderSet decoderSet;
    struct EncoderSet encoderSet;

    NN_ASSERT_EQUAL(decoder.GetSampleRate(), encoder.GetSampleRate());
    NN_ASSERT_EQUAL(decoder.GetChannelCount(), encoder.GetChannelCount());
    NN_ASSERT_EQUAL(decoder.GetStreamCount(), encoder.GetStreamCount());
    NN_ASSERT_EQUAL(decoder.GetCoupledStreamCount(), encoder.GetCoupledStreamCount());
    const auto sampleRate = encoder.GetSampleRate();
    const auto streamCount = encoder.GetStreamCount();
    const auto coupledStreamCount = encoder.GetCoupledStreamCount();

    // スレッドスタックの初期化
    std::memset(g_ThreadStackForDecoder, MemoryDefaultValue, ThreadStackSize);
    std::memset(&decoderSet, 0, sizeof(decoderSet));
    decoderSet.iteration = iteration;
    decoderSet.pDecoder = &decoder;
    nn::os::CreateThread(
        &decoding,
        DecoderFunction<DecoderType>,
        &decoderSet,
        g_ThreadStackForDecoder,
        ThreadStackSize,
        nn::os::DefaultThreadPriority,
        nn::os::GetCurrentCoreNumber()
    );
    nn::os::StartThread(&decoding);

    std::memset(g_ThreadStackForEncoder, MemoryDefaultValue, ThreadStackSize);
    std::memset(&encoderSet, 0, sizeof(encoderSet));
    encoderSet.iteration = iteration;
    encoderSet.frame = frame;
    encoderSet.bitRate = bitRate;
    encoderSet.codingMode = codingMode;
    encoderSet.streamCount = streamCount;
    encoderSet.coupledStreamCount = coupledStreamCount;
    encoderSet.pEncoder = &encoder;
    nn::os::CreateThread(
        &encoding,
        EncoderFunction<EncoderType>,
        &encoderSet,
        g_ThreadStackForEncoder,
        ThreadStackSize,
        nn::os::DefaultThreadPriority,
        nn::os::GetCurrentCoreNumber()
    );
    nn::os::StartThread(&encoding);
    //------------------------------
    // Finalize
    //------------------------------
    const std::size_t size = 32;
    std::size_t sizes[size];
    enum MemoryUsageId
    {
        MemoryUsageId_DecoderStack,
        MemoryUsageId_DecoderWorkBuffer,
        MemoryUsageId_EncoderStack,
        MemoryUsageId_EncoderWorkBuffer,
        MemoryUsageId_Count
    };
    std::size_t usages[MemoryUsageId_Count] = { 0, };

    nn::os::DestroyThread(&decoding);
    nn::os::DestroyThread(&encoding);
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    Reverse(g_ThreadStackTemporary, g_ThreadStackForDecoder, ThreadStackSize);
    usages[MemoryUsageId_DecoderStack] = AnalyzeMemoryUsage(sizes, size, g_ThreadStackTemporary, ThreadStackSize, MemoryDefaultValue);
    Reverse(g_ThreadStackTemporary, g_ThreadStackForEncoder, ThreadStackSize);
    usages[MemoryUsageId_EncoderStack] = AnalyzeMemoryUsage(sizes, size, g_ThreadStackTemporary, ThreadStackSize, MemoryDefaultValue);
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
    usages[MemoryUsageId_DecoderWorkBuffer] = AnalyzeMemoryUsage(sizes, size, g_WorkBufferForOpusDecoder, WorkBufferSize, MemoryDefaultValue);
    const std::size_t workSizeForDecoder = decoder.GetWorkBufferSize(decoder.GetSampleRate(), decoder.GetStreamCount(), decoder.GetCoupledStreamCount());
    EXPECT_LE(usages[MemoryUsageId_DecoderWorkBuffer], workSizeForDecoder);
    const std::size_t workSizeForEncoder = encoder.GetWorkBufferSize(decoder.GetSampleRate(), encoder.GetStreamCount(), encoder.GetCoupledStreamCount());
    usages[MemoryUsageId_EncoderWorkBuffer] = AnalyzeMemoryUsage(sizes, size, g_WorkBufferForOpusEncoder, WorkBufferSize, MemoryDefaultValue);
    EXPECT_LE(usages[MemoryUsageId_EncoderWorkBuffer], workSizeForEncoder);
    NN_LOG("##teamcity[MemoryUsage:%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d]\n",
        sampleRate,
        streamCount,
        coupledStreamCount,
        bitRate,
        frame,
        static_cast<int>(codingMode),
        workSizeForDecoder,
        workSizeForEncoder,
        usages[MemoryUsageId_DecoderWorkBuffer],
        usages[MemoryUsageId_EncoderWorkBuffer],
        usages[MemoryUsageId_DecoderStack],
        usages[MemoryUsageId_EncoderStack]);
}

}  // namespace

/**
 * シングルストリーム
 */
TEST_P(OpusCodecSingleStreamTest, MemoryUsage)
{
    RunMemoryUsageTestForOpusCodec<EncoderType, DecoderType>(
        encoder,
        decoder,
        frame,
        bitRate,
        codingMode,
        ProcessingIteration,
        GetParam()
    );
}

INSTANTIATE_TEST_CASE_P(
    RoundRobin,
    OpusCodecSingleStreamTest,
    ::testing::ValuesIn(CodecOpusTestParametersForSingleStream));

/**
 * マルチストリーム
 */
TEST_P(OpusCodecMultiStreamTest, MemoryUsage)
{
    RunMemoryUsageTestForOpusCodec<EncoderType, DecoderType>(
        encoder,
        decoder,
        frame,
        bitRate,
        codingMode,
        ProcessingIteration,
        GetParam()
    );
}

INSTANTIATE_TEST_CASE_P(
    RoundRobin,
    OpusCodecMultiStreamTest,
    ::testing::ValuesIn(CodecOpusTestParametersForMultiStream));

