﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Fx3Control.h"

namespace nnt {
namespace usb {
namespace cdmsc {

///////////////////////////////////////////////////////////////////////////////
//  static
///////////////////////////////////////////////////////////////////////////////
namespace
{
    nn::os::Event           g_DeviceAvailableEvent(nn::os::EventClearMode_AutoClear);
    nn::os::Event           g_DeviceDetachEvent(nn::os::EventClearMode_AutoClear);
    static int32_t          g_IfCount;
    static                  nn::usb::InterfaceQueryOutput g_IfList[nn::usb::HsLimitMaxInterfacesCount];
    static bool             g_IsFx3Initialized = false;
    nn::usb::Host           g_HsClient;
    nn::usb::HostInterface  g_Fx3Interface;
    nn::os::SystemEventType g_IfAvailableEvent;
    nn::usb::DeviceFilter   g_Fx3DeviceFilter = {
        .matchFlags = nn::usb::DeviceFilterMatchFlags_Vendor | nn::usb::DeviceFilterMatchFlags_Product,
        .idVendor = Fx3MscVID,
        .idProduct = Fx3MscPID,
    };
}

static bool Fx3InitializeInterface(int seconds)
{
    FX3_LOG("Looking for FX3 device...");
    for (int i = 0; i < seconds; i++)
    {
        if ( (g_HsClient.QueryAvailableInterfaces(&g_IfCount,
                g_IfList,
                sizeof(g_IfList),
                &g_Fx3DeviceFilter)
                ).IsFailure()
            )
        {
            return false;
        }

        for (int i = 0; i < g_IfCount; i++)
        {
            nn::usb::InterfaceProfile& profile = g_IfList[i].ifProfile;

            if (profile.ifDesc.bInterfaceClass == 0xFF)
            {
                nn::Result result;

                FX3_LOG("Initializing vendor-specific interface 0x%x\n", profile.ifDesc.bInterfaceNumber);
                result = g_Fx3Interface.Initialize(&g_HsClient, profile.handle);

                if (result.IsSuccess())
                {
                    return true;
                }
                return false;
            }
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    return false;
}


static void Fx3HandleDetach()
{
    ASSERT_TRUE(g_Fx3Interface.IsInitialized());
    NNT_ASSERT_RESULT_SUCCESS(g_Fx3Interface.Finalize());
}


static void Fx3InstallFirmware()
{
    nnt::usb::TestFx3Utility fx3Utility; // Install firmware

    FX3_LOG("Installing Fx3 Firmware...\n");
    Fx3FwErrorCode errorCode = Fx3FwErrorCode_Success;
    NNT_EXPECT_RESULT_SUCCESS(fx3Utility.Initialize(WAIT_SECONDS_FOR_ATTACH));
    errorCode = fx3Utility.UpdateFirmware(CdmscTestSuiteFx3Firmware, sizeof(CdmscTestSuiteFx3Firmware));
    fx3Utility.Finalize();

    if (errorCode != Fx3FwErrorCode_Success)
    {
        FX3_LOG("Warning: failed to UpdateFirmware to Fx3. Error code: %d\n", errorCode);
    }
}

static void Fx3WaitAllInterfacesDisconnect()
{
    for (;;)
    {
        NN_LOG("Waiting for all Fx3 interfaces to disconnect..\n");

        NNT_ASSERT_RESULT_SUCCESS(
            g_HsClient.QueryAllInterfaces(&g_IfCount, g_IfList, sizeof(g_IfList), &g_Fx3DeviceFilter)
            );

        NN_LOG("Found %d Fx3 interfaces\n", g_IfCount);
        if (g_IfCount == 0)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
}


bool Fx3Attach(int seconds)
{
    bool isSuccess = true;
    Fx3HandleDetach();
    FX3_LOG("Waiting for device available event...\n");
    if (g_DeviceAvailableEvent.TimedWait(nn::TimeSpan::FromSeconds(seconds)))
    {
        FX3_LOG("Device ready.\n");
        if (!Fx3InitializeInterface(seconds))
        {
            NN_ABORT("Fx3 device was not found after %d seconds.\n", seconds);
        }
    }
    else
    {
        FX3_LOG("Device not found after %d seconds.\n", seconds);
        isSuccess = false;;
    }
    return isSuccess;
}


void Fx3Initialize(int seconds, bool installFw)
{
    int maxRetries = 60;
    size_t bytesTransferred = 0;

    FX3_LOG("Fx3Intialize...\n");

    NNT_ASSERT_RESULT_SUCCESS(g_HsClient.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(
        g_HsClient.CreateInterfaceAvailableEvent(&g_IfAvailableEvent,
            nn::os::EventClearMode_ManualClear,
            0,
            &g_Fx3DeviceFilter)
    );

    NNT_ASSERT_RESULT_SUCCESS(
        nn::cdmsc::Initialize(
            g_DeviceAvailableEvent.GetBase(),
            std::aligned_alloc,
            [](void *p, size_t size) {
        NN_UNUSED(size);
        free(p);
    }
        )
    );

    if (Fx3InitializeInterface(seconds))
    {
        FX3_LOG("Found Fx3 device. Issuing cold reset command..\n");
        NNT_ASSERT_RESULT_SUCCESS(g_Fx3Interface.ControlRequest(
             &bytesTransferred,          //Cold reset device
             NULL,                       //void      *pData,
             0x40,                       //uint8_t    bmRequestType,
             Fx3Request_ColdReset,       //uint8_t    bRequest,
             0x00,                       //uint16_t   wValue,
             0x00,                       //uint16_t   wIndex,
             0x00));                     //uint16_t   wLength) NN_NOEXCEPT;

        Fx3WaitAllInterfacesDisconnect();

        NNT_ASSERT_RESULT_SUCCESS(g_Fx3Interface.Finalize());
    }

    for (int retry = 0; retry < maxRetries; retry++)
    {
        if (installFw)
        {
            Fx3InstallFirmware();
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        if (Fx3InitializeInterface(seconds))
        {
            g_IsFx3Initialized = true;
            return;
        }

    }

    NN_ABORT("After max retries (%d) Fx3 firmware failed to start.\n", maxRetries);
}


bool IsFx3Initialized()
{
    return g_IsFx3Initialized;
}


void Fx3Finalize()
{
    size_t bytesTransferred = 0;
    FX3_LOG("Issuing cold reset command\n");
    NNT_ASSERT_RESULT_SUCCESS(g_Fx3Interface.ControlRequest(
         &bytesTransferred,          //Cold reset device
         NULL,                       //void      *pData,
         0x40,                       //uint8_t    bmRequestType,
         Fx3Request_ColdReset,       //uint8_t    bRequest,
         0x00,                       //uint16_t   wValue,
         0x00,                       //uint16_t   wIndex,
         0x00));                     //uint16_t   wLength) NN_NOEXCEPT;

    Fx3WaitAllInterfacesDisconnect();

    NNT_ASSERT_RESULT_SUCCESS(nn::cdmsc::Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Fx3Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_HsClient.DestroyInterfaceAvailableEvent(&g_IfAvailableEvent, 0));
    NNT_ASSERT_RESULT_SUCCESS(g_HsClient.Finalize());
    g_IsFx3Initialized = false;
}


void Fx3Reconnect(int seconds)
{
    if (!Fx3InitializeInterface(seconds))
    {
        NN_ABORT("Fx3 device was not found after %d seconds.\n", seconds);
    }
}


void Fx3Reset(int seconds, bool installFw)
{
    size_t bytesTransferred = 0;
    FX3_LOG("Issuing cold reset command\n");
    NNT_ASSERT_RESULT_SUCCESS(g_Fx3Interface.ControlRequest(
        &bytesTransferred,          //Cold reset device
        NULL,                       //void      *pData,
        0x40,                       //uint8_t    bmRequestType,
        Fx3Request_ColdReset,       //uint8_t    bRequest,
        0x00,                       //uint16_t   wValue,
        0x00,                       //uint16_t   wIndex,
        0x00));                     //uint16_t   wLength) NN_NOEXCEPT;

    Fx3WaitAllInterfacesDisconnect();

    NNT_ASSERT_RESULT_SUCCESS(nn::cdmsc::Finalize());

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    NNT_ASSERT_RESULT_SUCCESS(
        nn::cdmsc::Initialize(
            g_DeviceAvailableEvent.GetBase(),
            std::aligned_alloc,
            [](void *p, size_t size) {
        NN_UNUSED(size);
        free(p);
    }
        )
    );

    if (installFw)
    {
        Fx3InstallFirmware();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    if (Fx3Attach(seconds))
    {
        FX3_LOG("Device Attached!\n");
    }
    else
    {
        FX3_LOG("Device Failed to attach with hard reset.");
        FAIL();
    }
}


void Fx3ActivateMassStorageMode(int seconds)
{
    FX3_LOG("Activate Mass Storage Mode!\n");
    size_t bytesTransferred = 0;
    NNT_ASSERT_RESULT_SUCCESS(g_Fx3Interface.ControlRequest(
        &bytesTransferred,          // Set WP to 1
        NULL,                       //void      *pData,
        0x40,                       //uint8_t    bmRequestType,
        Fx3Request_Reset,           //uint8_t    bRequest,
        0x02,                       //uint16_t   wValue,
        0x01,                       //uint16_t   wIndex,
        0x00));                     //uint16_t   wLength) NN_NOEXCEPT;

    Fx3Attach(seconds);
}


nn::Result Fx3Probe(nn::cdmsc::UnitProfile *pOutProfile)
{
    return nn::cdmsc::Probe(g_DeviceDetachEvent.GetBase(), pOutProfile);
}

bool Fx3WaitForDeviceDisconnect(int seconds)
{
    bool isSuccess = true;
    if (g_DeviceDetachEvent.TimedWait(nn::TimeSpan::FromSeconds(seconds)))
    {
        FX3_LOG("Device disconnected..\n");
    }
    else
    {
        FX3_LOG("Device not disconnected after %d seconds.\n", seconds);
        isSuccess = false;;
    }
    return isSuccess;
}

void Fx3SendVendorSpecificRequest(uint8_t *pData, uint8_t bmRequest, Fx3Request request, uint16_t wValue, uint16_t wIndex, uint16_t wLength)
{
    FX3_LOG("Issuing Vendor specific command: 0x%x\n", request);
    size_t bytesTransferred = 0;
    NNT_ASSERT_RESULT_SUCCESS(g_Fx3Interface.ControlRequest(
        &bytesTransferred,          // Set WP to 1
        pData,                      //void      *pData,
        bmRequest,                  //uint8_t    bmRequestType,
        request,                    //uint8_t    bRequest,
        wValue,                     //uint16_t   wValue,
        wIndex,                     //uint16_t   wIndex,
        wLength));                  //uint16_t   wLength) NN_NOEXCEPT;
}


void Fx3PrintString(uint8_t *pStringData, uint32_t size)
{
    size_t bytesTransferred;

    if (size <= Fx3MscMaxControlBufSize)
    {
        NNT_EXPECT_RESULT_SUCCESS(g_Fx3Interface.ControlRequest(
                                            &bytesTransferred,              // bytes transferred
                                            pStringData,                    // buffer
                                            0x40,                           // bmRequestType: device to host | vendor | device
                                            Fx3Request_PrintString,         // bRequest
                                            0,                              // wValue
                                            0,                              // wIndex
                                            size                            // wLength
                                            ));
    }
    else
    {
        FX3_LOG("ERROR: String is greater than Fx3MscMaxControlBufSize buffer (%d).\n", Fx3MscMaxControlBufSize);
        FAIL();
    }

}

} // cdmsc
} // usb
} // nnt
