﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt.h>

#include <nn/cdmsc.h>

namespace {

nn::os::Event           g_DeviceAvailableEvent(nn::os::EventClearMode_AutoClear);
nn::os::Event           g_DeviceDetachEvent(nn::os::EventClearMode_AutoClear);
nn::cdmsc::UnitProfile  g_Profile;

}

TEST(Cdmsc_Basic, Initialize)
{
    NNT_ASSERT_RESULT_SUCCESS(
        nn::cdmsc::Initialize(
            g_DeviceAvailableEvent.GetBase(),
            std::aligned_alloc,
            [](void *p, size_t size) {
                NN_UNUSED(size);
                free(p);
            }
        )
    );
}

TEST(Cdmsc_Basic, WaitForDevice)
{
    while (true)
    {
        NN_LOG("Waiting for device available event...\n");

        if (g_DeviceAvailableEvent.TimedWait(nn::TimeSpan::FromSeconds(1)))
        {
            break;
        }
    }
}

TEST(Cdmsc_Basic, ProbeDevice)
{
    NNT_ASSERT_RESULT_SUCCESS(
        nn::cdmsc::Probe(g_DeviceDetachEvent.GetBase(), &g_Profile)
    );
}

TEST(Cdmsc_Basic, Read)
{
    NN_ALIGNAS(4096) uint8_t buffer[1024 * 4];

    ASSERT_GE(sizeof(buffer), g_Profile.blockSize);

    for(uint64_t lba = 0; lba < 10; lba++)
    {
        NNT_ASSERT_RESULT_SUCCESS(
            nn::cdmsc::Read(buffer, g_Profile.handle, lba, 1)
        );
    }
}

TEST(Cdmsc_Basic, LetItRunForAWhile)
{
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
}

TEST(Cdmsc_Basic, Finalize)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::cdmsc::Finalize());
}

TEST(Cdmsc_Basic, ReInit)
{
    NNT_ASSERT_RESULT_SUCCESS(
        nn::cdmsc::Initialize(
            g_DeviceAvailableEvent.GetBase(),
            std::aligned_alloc,
            [](void *p, size_t size) {
                NN_UNUSED(size);
                free(p);
            }
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(nn::cdmsc::Finalize());
}
