﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt.h>
#include <nn/init/init_Malloc.h>

#include <nn/usb/usb_Host.h>
#include <nn/cdacm/cdacm.h>

namespace nnt {
namespace usb {
namespace cdacm {


namespace
{
    enum ThreadIndex {
        ThreadIndex_Main,
        ThreadIndex_Second,
        ThreadIndex_Max
    };
    const size_t                       g_HeapSize = 16 * 1024 * 1024;
    nn::cdacm::UnitProfile             g_Profile[ThreadIndex_Max];
    nn::os::MultiWaitType             *g_pDeviceAvailableEvent;
    nn::os::SystemEventType           *g_pThreadDetachEvent[ThreadIndex_Max];
    nn::os::ThreadType                 g_SecondThread;
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_SecondThreadStack[1024 * 16];
    nn::cdacm::CdAcmParameters  g_Parameters;
    nn::os::Event                      g_ThreadSyncEvent(nn::os::EventClearMode_AutoClear);
    uint32_t                           g_LoopTimeInTimeUnits = 12;
    nn::TimeSpan                       g_StartTime;
    bool                               g_Continue = true;
}

class CdAcmTestEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override;
    virtual void TearDown() override;
};

void
HandlePossibleDisconnect(uint8_t threadIndex)
{
    nn::os::MultiWaitHolderType *pHolder = nullptr;

    if (nn::os::TryWaitSystemEvent(g_pThreadDetachEvent[threadIndex]))
    {
        nn::Result result = nn::ResultSuccess();
        NN_LOG("Thread %d: Disconnect detected\n", threadIndex);
        result = nn::cdacm::CloseHandle(&g_Profile[threadIndex]);
        if (result.IsFailure())
        {
            NN_LOG("Thread %d: CloseHandle failed.  %d:%d\n", threadIndex,
                result.GetModule(), result.GetDescription());
        }

        NN_LOG("Waiting for device %d to be reconnected, threadIndex\n");
        pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);
        NN_LOG("Opening handle for device %d in recovery\n", threadIndex);
        do
        {
            result = OpenHandle(&g_pThreadDetachEvent[threadIndex],
                                &g_Profile[threadIndex],
                                pHolder,
                                &g_Parameters);
            if (result.IsFailure())
            {
                NN_LOG("OpenHandle failed with %d:%d.  Sleeping then trying again....\n", result.GetModule(), result.GetDescription());
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }

        } while (result.IsFailure());
    }
    else
    {
        NN_LOG("Thread %d: NO disconnect detected\n", threadIndex);
    }
}

static void TestThreadEntry(void* pThis)
{
    size_t bytesXferred = 0;
    nn::Result result;
    nn::os::MultiWaitHolderType *pHolder = nullptr;
    static NN_USB_DMA_ALIGN uint8_t buffer[512] = { 0 };

    NN_LOG("2nd thread: In thread waiting for device 1 to attach....\n");
    pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);

    NN_LOG("2nd thread: Probing for device 1 in primary thread\n");
    result = OpenHandle(&g_pThreadDetachEvent[ThreadIndex_Second],
                        &g_Profile[ThreadIndex_Second],
                        pHolder,
                        &g_Parameters);

    if (result.IsFailure())
    {
        NN_LOG("2nd thread: Failed to open second device.  Aborting...\n");
        NN_ABORT();
    }
    g_ThreadSyncEvent.Signal();

    do
    {
        NN_LOG("2nd thread: Requesting to read\n");
        result = nn::cdacm::Read(&bytesXferred,
                                 (void*)(&buffer[0]),
                                 g_Profile[ThreadIndex_Second].handle,
                                 sizeof(buffer));
        if (result.IsSuccess())
        {
            NN_LOG("2nd thread: %d bytes read\n", bytesXferred);
        }
        else
        {
            NN_LOG("2nd thread: Read failed.\n");
            HandlePossibleDisconnect(ThreadIndex_Second);
        }


        result = nn::cdacm::Write(&bytesXferred,
                                  (void*)(&buffer[0]),
                                  g_Profile[ThreadIndex_Second].handle,
                                  bytesXferred);
        if (result.IsSuccess())
        {
            NN_LOG("2nd thread: %d bytes written\n", bytesXferred);
        }
        else
        {
            NN_LOG("2nd thread: Write failed.\n");
            HandlePossibleDisconnect(ThreadIndex_Second);
        }
    } while (g_Continue);
}

void CdAcmTestEnvironment::SetUp()
{
    nn::Result result;
    nn::os::MultiWaitHolderType *pHolder = nullptr;

    NN_LOG("Initializing cdacm driver\n");

    g_StartTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

    nn::cdacm::Initialize(&g_pDeviceAvailableEvent, NULL, 0);

    NN_LOG("Waiting for device....\n");
    pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);

    g_Parameters.baudRate = 9600;
    g_Parameters.latency = 0x10;
    g_Parameters.flowCtrl = 0;
    g_Parameters.modemControl = 0x0101;
    g_Parameters.dataCharacteristics = 0x0008;
    g_Parameters.eventCharacter = 0;

    NN_LOG("Opening handle for device 0 in primary thread\n");
    OpenHandle(&g_pThreadDetachEvent[ThreadIndex_Main],
               &g_Profile[ThreadIndex_Main],
               pHolder,
               &g_Parameters);

    nn::os::CreateThread(&g_SecondThread, TestThreadEntry, NULL,
        &g_SecondThreadStack, sizeof(g_SecondThreadStack),
        nn::os::LowestThreadPriority, 0);

    nn::os::StartThread(&g_SecondThread);

    g_ThreadSyncEvent.Wait();
    NN_LOG("Secondary thread executing.  Exiting setup\n");
}

void CdAcmTestEnvironment::TearDown()
{
    NNT_ASSERT_RESULT_SUCCESS(nn::cdacm::Finalize());
}

TEST(CdAcmTestSuite, PnpTest)
{
    static NN_USB_DMA_ALIGN uint8_t buffer1[] = {
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
        0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a
    };
    static NN_USB_DMA_ALIGN uint8_t buffer2[512] = { 0 };
    size_t bytesXferred = 0;
    nn::Result result;
    nn::TimeSpan  currentTime;

    do
    {
        result = nn::cdacm::Write(&bytesXferred, (const void*)(&buffer1[0]), g_Profile[0].handle, sizeof(buffer1));

        if (result.IsSuccess())
        {
            NN_LOG("Main thread: %d bytes written\n", bytesXferred);
        }
        else
        {
            NN_LOG("Main thread: Write failed.  %d:%d\n", result.GetModule(), result.GetDescription());
            HandlePossibleDisconnect(ThreadIndex_Main);
        }

        bytesXferred = 0;
        memset((void*)buffer2, 0, sizeof(buffer2));
        result = nn::cdacm::Read(&bytesXferred, (void*)buffer2, g_Profile[0].handle, sizeof(buffer2));

        if (result.IsSuccess())
        {
            NN_LOG("Main thread: %d bytes read\n", bytesXferred);
        }
        else
        {
            NN_LOG("Main thread: Read failed.  %d:%d\n", result.GetModule(), result.GetDescription());
            HandlePossibleDisconnect(ThreadIndex_Main);
        }
        currentTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        if (currentTime.GetHours() - g_StartTime.GetHours() > g_LoopTimeInTimeUnits)
        {
            NN_LOG("Main thread: Ending loop as time has passed. Setting flag to stop.\n");
            g_Continue = false;
        }
    } while (g_Continue);
}

} // cdacm
} // usb
} // nnt

extern "C" void nninitStartup()
{
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(nnt::usb::cdacm::g_HeapSize);
    if (!result.IsSuccess())
    {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, nnt::usb::cdacm::g_HeapSize / 2);
    NNT_EXPECT_RESULT_SUCCESS(result);

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), nnt::usb::cdacm::g_HeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new nnt::usb::cdacm::CdAcmTestEnvironment);
}

extern "C" void nnMain()
{
    int    argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    const int exitCode = RUN_ALL_TESTS();

    ::nnt::Exit(exitCode);
}
