﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt.h>
#include <nn/init/init_Malloc.h>
#include <nn/usb/usb_Host.h>
#include <nn/cdacm/cdacm.h>

namespace nnt {
namespace usb {
namespace cdacm {

namespace
{
    const size_t                       g_HeapSize = 16 * 1024 * 1024;
    nn::cdacm::UnitProfile             g_Profile[2];
    nn::os::MultiWaitType *            g_pDeviceAvailableEvent;
    nn::os::SystemEventType           *g_MainThreadDetach;
    nn::os::SystemEventType           *g_2ndThreadDetach;
    nn::os::ThreadType                 g_SecondThread;
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_SecondThreadStack[16 * 1024];
    struct nn::cdacm::CdAcmParameters  g_Parameters;
    nn::os::Event                      g_ThreadSyncEvent(nn::os::EventClearMode_AutoClear);
}

class CdAcmTestEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override;
    virtual void TearDown() override;
};

static void TestThreadEntry(void* pThis)
{
    static NN_USB_DMA_ALIGN uint8_t buffer[512] = { 0 };
    size_t bytesXferred = 0;
    nn::Result result;
    nn::os::MultiWaitHolderType *pHolder = nullptr;


    NN_LOG("2nd thread: In thread waiting for device 2 to attach....\n");
    pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);

    NN_LOG("2nd thread: Probing for device 2 in primary thread\n");
    result = nn::cdacm::OpenHandle(&g_2ndThreadDetach, &g_Profile[1], pHolder, &g_Parameters);

    if (result.IsFailure())
    {
        NN_LOG("2nd thread: Failed to open second device\n");
        return;
    }
    g_ThreadSyncEvent.Signal();

    //
    // Read/Write (synchronous)
    //
    NN_LOG("2ND THREAD: SYNCHRONOUS\n");


    NN_LOG("2nd thread: Requesting to read (1)\n");
    result = nn::cdacm::Read(&bytesXferred,
                             (void*)(&buffer[0]),
                             g_Profile[1].handle,
                             sizeof(buffer));
    if (result.IsFailure())
    {
        NN_LOG("2nd thread: Read (1) failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }

    NN_LOG("2nd thread: %d bytes read (1)\n", bytesXferred);

    result = nn::cdacm::Write(&bytesXferred,
                              (void*)(&buffer[0]),
                              g_Profile[1].handle,
                              bytesXferred);

    if (result.IsSuccess())
    {
        NN_LOG("2nd thread: %d bytes written (1)\n", bytesXferred);
    }
    else
    {
        NN_LOG("2nd thread: Write (1) failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }

    //
    // Read/Write (for print)
    //
    NN_LOG("2ND THREAD: PRINT\n");

    NN_LOG("2nd thread: Requesting to read (print)\n");
    result = nn::cdacm::Read(&bytesXferred,
        (void*)buffer,
        g_Profile[1].handle,
        sizeof(buffer));
    if (result.IsFailure())
    {
        NN_LOG("2nd thread: Read (print) failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }

    NN_LOG("2nd thread: %d bytes read (print)\n", bytesXferred);

    result = nn::cdacm::Write(&bytesXferred,
        (void*)buffer,
        g_Profile[1].handle,
        bytesXferred);

    if (result.IsSuccess())
    {
        NN_LOG("2nd thread: %d bytes written\n", bytesXferred);
    }
    else
    {
        NN_LOG("2nd thread: Write failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }

    //
    // Read/Write (Asynchronous)
    //
    {
        uint32_t                 outXferId;
        nn::os::SystemEventType *pCompletionEvent;
        uint32_t count = 0;
        nn::usb::XferReport report;

        NN_LOG("2ND THREAD: ASYNCHRONOUS\n");
        NN_LOG("2nd thread: Requesting to read\n");
        result = nn::cdacm::Read(&bytesXferred,
            (void*)(&buffer[0]),
            g_Profile[1].handle,
            sizeof(buffer));
        if (result.IsFailure())
        {
            NN_LOG("2nd thread: Read failed.  Exiting\n");
            return;
        }
        NN_LOG("2nd thread: %d bytes read\n", bytesXferred);

        NN_LOG("2nd thread: Issuing asynch write\n");
        result = nn::cdacm::WriteAsync(&outXferId, &pCompletionEvent, buffer, g_Profile[1].handle, bytesXferred);
        if (result.IsFailure())
        {
            NN_LOG("WriteAsync failed.  %d:%d\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            NN_LOG("WriteAsync Succeeded.  Waiting for completion event\n");
        }

        WaitSystemEvent(pCompletionEvent);
        NN_LOG("WriteAsync completed.\n");

        result = nn::cdacm::GetWriteAsyncResult(&count, &report, g_Profile[1].handle, 1);
        if (result.IsFailure())
        {
            NN_LOG("GetWriteAsyncResult failed.  %d:%d\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            NN_LOG("GetWriteAsyncResult Succeeded.\n");
            NN_LOG("- id %d\n", report.id);
            NN_LOG("- result %d:%d\n", report.result.GetModule(), report.result.GetDescription());
            NN_LOG("- requestedSize %d\n", report.requestedSize);
            NN_LOG("- transferredSize %d\n", report.transferredSize);
            NN_LOG("- context %lld\n", report.context);
        }
    }
    result = nn::cdacm::CloseHandle(&g_Profile[1]);
    if (result.IsSuccess())
    {
        NN_LOG("2nd thread: handle closed\n");
    }
    else
    {
        NN_LOG("2nd thread: close handle failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }
} // NOLINT(impl/function_size)

void CdAcmTestEnvironment::SetUp()
{
    nn::Result result;
    nn::os::MultiWaitHolderType *pHolder = nullptr;

    NN_LOG("Initializing cdacm driver\n");

    nn::cdacm::Initialize(&g_pDeviceAvailableEvent, NULL, 0);

    NN_LOG("Waiting for device....\n");
    pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);

    g_Parameters.baudRate = 9600;
    g_Parameters.latency = 0x10;
    g_Parameters.flowCtrl = 0;
    g_Parameters.modemControl = 0x0101;
    g_Parameters.dataCharacteristics = 0x0008;
    g_Parameters.eventCharacter = 0;

    NN_LOG("Probing for device 1 in primary thread\n");
    result = nn::cdacm::OpenHandle(&g_MainThreadDetach, &g_Profile[0], pHolder, &g_Parameters);

    if (result.IsFailure())
    {
        NN_LOG("Main thread: Failed to open first device\n");
        return;
    }


    nn::os::CreateThread(&g_SecondThread, TestThreadEntry, NULL,
        g_SecondThreadStack, sizeof(g_SecondThreadStack),
        nn::os::LowestThreadPriority, 0);

    nn::os::StartThread(&g_SecondThread);

    g_ThreadSyncEvent.Wait();
    NN_LOG("Secondary thread executing.  Exiting setup\n");
}

void CdAcmTestEnvironment::TearDown()
{
    NN_LOG("Tearing down environment\n");
    WaitThread(&g_SecondThread);
    NNT_ASSERT_RESULT_SUCCESS(nn::cdacm::Finalize());
}



TEST(CdAcmTestSuite, LoopbackTest)
{
    NN_USB_DMA_ALIGN uint8_t buffer1[] = { 0x64, 0x65, 0x66, 0x67, 0x68, 0x69 };
    NN_USB_DMA_ALIGN uint8_t buffer2[512] = { 0 };
    size_t bytesXferred = 0;
    nn::Result result;

    NN_LOG("MAIN THREAD: SYNCHRONOUS\n");

    result = nn::cdacm::Write(&bytesXferred, (const void*)(&buffer1[0]), g_Profile[0].handle, sizeof(buffer1));

    if (result.IsSuccess())
    {
        NN_LOG("Main thread: %d bytes written synchronously\n", bytesXferred);
    }
    else
    {
        NN_LOG("Main thread: Synchronous write failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }

    bytesXferred = 0;
    memset((void*)(&buffer2[0]), 0, sizeof(buffer2));
    result = nn::cdacm::Read(&bytesXferred, (void*)(&buffer2[0]), g_Profile[0].handle, sizeof(buffer2));
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (result.IsSuccess())
    {
        NN_LOG("Main thread: %d bytes read (1)\n", bytesXferred);
        ASSERT_EQ(memcmp(buffer1, buffer2, bytesXferred), 0);
    }
    else
    {
        NN_LOG("Main thread: Read (1) failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }


    NN_LOG("MAIN THREAD: PRINT\n");
    result = nn::cdacm::Print(g_Profile[0].handle, "Value sent is %d", 5);
    if (result.IsSuccess())
    {
        NN_LOG("Main thread: print succeeded\n");
    }
    else
    {
        NN_LOG("Main thread: Write failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }
    bytesXferred = 0;
    memset((void*)(&buffer2[0]), 0, sizeof(buffer2));
    result = nn::cdacm::Read(&bytesXferred, (void*)(&buffer2[0]), g_Profile[0].handle, sizeof(buffer2));
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (result.IsSuccess())
    {
        NN_LOG("Main thread: %d bytes read.  String received %s\n", bytesXferred, (char*)buffer2);
    }
    else
    {
        NN_LOG("Main thread: Read failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }

    {
        uint32_t                 outXferId;
        nn::os::SystemEventType *pCompletionEvent;
        uint32_t count = 0;
        nn::usb::XferReport report;

        NN_LOG("MAIN THREAD: ASYNCHRONOUS\n");

        result = nn::cdacm::WriteAsync(&outXferId, &pCompletionEvent, buffer1, g_Profile[0].handle, sizeof(buffer1));
        if (result.IsFailure())
        {
            NN_LOG("Main thread: WriteAsync failed.  %d:%d\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            NN_LOG("Main thread: WriteAsync Succeeded.  Waiting for completion event\n");
        }
        WaitSystemEvent(pCompletionEvent);
        NN_LOG("Main thread: WriteAsync completed.\n");

        result = nn::cdacm::GetWriteAsyncResult(&count, &report, g_Profile[0].handle, 1);
        if (result.IsFailure())
        {
            NN_LOG("Main thread: GetWriteAsyncResult failed.  %d:%d\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            NN_LOG("GetWriteAsyncResult Succeeded.\n");
            NN_LOG("- id %d\n", report.id);
            NN_LOG("- result %d:%d\n", report.result.GetModule(), report.result.GetDescription());
            NN_LOG("- requestedSize %d\n", report.requestedSize);
            NN_LOG("- transferredSize %d\n", report.transferredSize);
            NN_LOG("- context %lld\n", report.context);
        }

        NN_LOG("Main thread: Requesting to read (2)\n");
        result = nn::cdacm::Read(&bytesXferred,
            (void*)buffer2,
            g_Profile[0].handle,
            sizeof(buffer2));
        if (result.IsFailure())
        {
            NN_LOG("Main thread: Read (2) failed.  %d:%d\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            NN_LOG("Main thread: %d bytes read (2)\n", bytesXferred);
        }
    }
    result = nn::cdacm::CloseHandle(&g_Profile[0]);
    if (result.IsSuccess())
    {
        NN_LOG("Main thread: handle closed\n");
    }
    else
    {
        NN_LOG("Main thread: close handle failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }
} // NOLINT(impl/function_size)

} // cdacm
} // usb
} // nnt

extern "C" void nninitStartup()
{
    nn::Result result;

    //
    result = nn::os::SetMemoryHeapSize(nnt::usb::cdacm::g_HeapSize);
    if (!result.IsSuccess()) {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", nnt::usb::cdacm::g_HeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, nnt::usb::cdacm::g_HeapSize / 2);
    NNT_EXPECT_RESULT_SUCCESS(result);

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), nnt::usb::cdacm::g_HeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new nnt::usb::cdacm::CdAcmTestEnvironment);
}

extern "C" void nnMain()
{
    int    argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    const int exitCode = RUN_ALL_TESTS();

    ::nnt::Exit(exitCode);
}
