﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt.h>
#include <nn/init/init_Malloc.h>

#include <nn/usb/usb_Host.h>
#include <nn/cdacm/cdacm.h>

namespace nnt {
namespace usb {
namespace cdacm {

namespace
{
    const size_t                       g_HeapSize = 16 * 1024 * 1024;
    nn::cdacm::UnitProfile             g_Profile;
    nn::os::MultiWaitType *            g_pDeviceAvailableEvent;
    nn::os::SystemEventType           *g_Detach;
    struct nn::cdacm::CdAcmParameters  g_Parameters;
}

class CdAcmTestEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override;
    virtual void TearDown() override;
};

void CdAcmTestEnvironment::SetUp()
{
}


void CdAcmTestEnvironment::TearDown()
{
}



TEST(CdAcmTestSuite, AsyncWriteTest1)
{
    NN_USB_DMA_ALIGN uint8_t buffer1[] = { 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69 };
    uint8_t                  asyncBuf[512];
    size_t bytesXferred = 0;
    nn::Result result;
    nn::os::MultiWaitHolderType *pHolder = nullptr;

    NN_LOG("Initializing cdacm driver\n");

    nn::cdacm::Initialize(&g_pDeviceAvailableEvent,
        static_cast<void*>(asyncBuf),
        sizeof(asyncBuf));

    nn::cdacm::EnableLogObserver();

    result = nn::cdacm::Write(&bytesXferred, (const void*)(&buffer1[0]), 0, sizeof(buffer1));

    if (nn::cdacm::ResultPending::Includes(result))
    {
        NN_LOG("Write returned result pending\n");
    }
    else
    {
        NN_LOG("Write failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }


    NN_LOG("Waiting for device....\n");
    pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);

    g_Parameters.baudRate = 9600;
    g_Parameters.latency = 0x10;
    g_Parameters.flowCtrl = 0;
    g_Parameters.modemControl = 0x0101;
    g_Parameters.dataCharacteristics = 0x0008;
    g_Parameters.eventCharacter = 0;

    NN_LOG("Probing for device 1 in primary thread\n");
    result = OpenHandle(&g_Detach, &g_Profile, pHolder, &g_Parameters);

    if (result.IsFailure())
    {
        NN_LOG("Failed to open first device\n");
        goto Exit;
    }


    result = nn::cdacm::CloseHandle(&g_Profile);
    if (result.IsSuccess())
    {
        NN_LOG("handle closed\n");
    }
    else
    {
        NN_LOG("close handle failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }

Exit:
    nn::cdacm::DisableLogObserver();
    NNT_ASSERT_RESULT_SUCCESS(nn::cdacm::Finalize());
    return;
}


TEST(CdAcmTestSuite, AsyncWriteTest2)
{
    NN_USB_DMA_ALIGN uint8_t buffer1[] = { 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f };
    uint8_t                  asyncBuf[10];
    size_t bytesXferred = 0;
    nn::Result result;
    nn::os::MultiWaitHolderType *pHolder = nullptr;

    NN_LOG("Initializing cdacm driver\n");

    nn::cdacm::Initialize(&g_pDeviceAvailableEvent,
        static_cast<void*>(asyncBuf),
        sizeof(asyncBuf));


    result = nn::cdacm::Write(&bytesXferred, (const void*)(&buffer1[0]), 0, sizeof(buffer1));

    if (nn::cdacm::ResultPending::Includes(result))
    {
        NN_LOG("Write returned result pending\n");
    }
    else
    {
        NN_LOG("Write failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }


    NN_LOG("Waiting for device....\n");
    pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);

    g_Parameters.baudRate = 9600;
    g_Parameters.latency = 0x10;
    g_Parameters.flowCtrl = 0;
    g_Parameters.modemControl = 0x0101;
    g_Parameters.dataCharacteristics = 0x0008;
    g_Parameters.eventCharacter = 0;

    NN_LOG("Probing for device 1 in primary thread\n");
    result = OpenHandle(&g_Detach, &g_Profile, pHolder, &g_Parameters);

    if (result.IsFailure())
    {
        NN_LOG("Failed to open first device\n");
        return;
    }


    result = nn::cdacm::CloseHandle(&g_Profile);
    if (result.IsSuccess())
    {
        NN_LOG("handle closed\n");
    }
    else
    {
        NN_LOG("close handle failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::cdacm::Finalize());
}

TEST(CdAcmTestSuite, AsyncWriteTest3)
{
    NN_USB_DMA_ALIGN uint8_t buffer1[] = { 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72 };
    uint8_t                  asyncBuf[20];
    size_t bytesXferred = 0;
    nn::Result result;
    nn::os::MultiWaitHolderType *pHolder = nullptr;

    NN_LOG("Initializing cdacm driver\n");

    nn::cdacm::Initialize(&g_pDeviceAvailableEvent,
        static_cast<void*>(asyncBuf),
        sizeof(asyncBuf));


    result = nn::cdacm::Write(&bytesXferred, (const void*)(&buffer1[0]), 0, sizeof(buffer1));

    if (nn::cdacm::ResultPending::Includes(result))
    {
        NN_LOG("Write returned result pending\n");
    }
    else
    {
        NN_LOG("Write failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }

    NN_LOG("Waiting for device....\n");
    pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);

    g_Parameters.baudRate = 9600;
    g_Parameters.latency = 0x10;
    g_Parameters.flowCtrl = 0;
    g_Parameters.modemControl = 0x0101;
    g_Parameters.dataCharacteristics = 0x0008;
    g_Parameters.eventCharacter = 0;

    NN_LOG("Probing for device 1 in primary thread\n");
    result = OpenHandle(&g_Detach, &g_Profile, pHolder, &g_Parameters);

    if (result.IsFailure())
    {
        NN_LOG("Failed to open first device\n");
        return;
    }


    result = nn::cdacm::CloseHandle(&g_Profile);
    if (result.IsSuccess())
    {
        NN_LOG("handle closed\n");
    }
    else
    {
        NN_LOG("close handle failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::cdacm::Finalize());
}

TEST(CdAcmTestSuite, AsyncWriteTest4)
{
    NN_USB_DMA_ALIGN uint8_t buffer1[] = { 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f };
    uint8_t                  asyncBuf[20];
    size_t bytesXferred = 0;
    nn::Result result;
    nn::os::MultiWaitHolderType *pHolder = nullptr;

    NN_LOG("Initializing cdacm driver\n");

    nn::cdacm::Initialize(&g_pDeviceAvailableEvent,
        static_cast<void*>(asyncBuf),
        sizeof(asyncBuf));


    result = nn::cdacm::Write(&bytesXferred, (const void*)(&buffer1[0]), 0, sizeof(buffer1));

    if (nn::cdacm::ResultPending::Includes(result))
    {
        NN_LOG("Write returned result pending\n");
    }
    else
    {
        NN_LOG("Write failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }
    result = nn::cdacm::Write(&bytesXferred, (const void*)(&buffer1[0]), 0, sizeof(buffer1));

    if (nn::cdacm::ResultPending::Includes(result))
    {
        NN_LOG("Write returned result pending\n");
    }
    else
    {
        NN_LOG("Write failed.  %d:%d\n", result.GetModule(), result.GetDescription());
    }

    NN_LOG("Waiting for device....\n");
    pHolder = nn::os::WaitAny(g_pDeviceAvailableEvent);

    g_Parameters.baudRate = 9600;
    g_Parameters.latency = 0x10;
    g_Parameters.flowCtrl = 0;
    g_Parameters.modemControl = 0x0101;
    g_Parameters.dataCharacteristics = 0x0008;
    g_Parameters.eventCharacter = 0;

    NN_LOG("Probing for device 1 in primary thread\n");
    result = OpenHandle(&g_Detach, &g_Profile, pHolder, &g_Parameters);

    if (result.IsFailure())
    {
        NN_LOG("Failed to open first device\n");
        return;
    }


    result = nn::cdacm::CloseHandle(&g_Profile);
    if (result.IsSuccess())
    {
        NN_LOG("handle closed\n");
    }
    else
    {
        NN_LOG("close handle failed.  %d:%d\n",
            result.GetModule(), result.GetDescription());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::cdacm::Finalize());
}

} // cdacm
} // usb
} // nnt

extern "C" void nninitStartup()
{
    nn::Result result;

    //
    result = nn::os::SetMemoryHeapSize(nnt::usb::cdacm::g_HeapSize);
    if (!result.IsSuccess()) {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", nnt::usb::cdacm::g_HeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, nnt::usb::cdacm::g_HeapSize / 2);
    NNT_EXPECT_RESULT_SUCCESS(result);

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), nnt::usb::cdacm::g_HeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new nnt::usb::cdacm::CdAcmTestEnvironment);
}

extern "C" void nnMain()
{
    int    argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    const int exitCode = RUN_ALL_TESTS();

    ::nnt::Exit(exitCode);
}
