﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_Result.h>

#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_EncryptApplicationId.h"

namespace {

    class RandomApplicationIdGenerator
    {
        static const uint32_t RandomSeed = 0x99945539;

    public:
        RandomApplicationIdGenerator() NN_NOEXCEPT
        {
            m_Engine.seed(RandomSeed);
        }

        nn::ncm::ApplicationId GenerateApplicationId() NN_NOEXCEPT
        {
            nn::ncm::ApplicationId id;
            id.value = m_Dist(m_Engine);
            return id;
        }

        nn::capsrv::server::detail::EncryptedApplicationId GenerateEncryptedApplicationId() NN_NOEXCEPT
        {
            uint64_t bytes[2];
            nn::capsrv::server::detail::EncryptedApplicationId eid;
            NN_STATIC_ASSERT(sizeof(bytes) == sizeof(eid.value));

            for(int i = 0; i < 2; i++)
            {
                bytes[i] = m_Dist(m_Engine);
            }

            std::memcpy(eid.value, bytes, sizeof(eid.value));
            return eid;
        }

    private:
        std::mt19937 m_Engine;
        std::uniform_int_distribution<uint64_t> m_Dist;
    };

}


TEST(UnitTest, EncryptoApplicationId)
{
    RandomApplicationIdGenerator idgen;

    // 固定のアプリケーション ID の暗号化結果が一致することの確認
    {
        nn::ncm::ApplicationId src;
        std::memcpy(&src.value, "Nintendo", 8);

        auto mid = nn::capsrv::server::detail::EncryptApplicationId(src, false);

        const uint8_t ref[16] = { 0x99, 0x4B, 0x64, 0x11, 0x58, 0xF4, 0xF2, 0x7A, 0x26, 0xFE, 0x2B, 0x9A, 0xAF, 0xC6, 0xB5, 0x68 };
        EXPECT_EQ(0, std::memcmp(mid.value, ref, sizeof(ref)));

        nn::ncm::ApplicationId dst;
        bool isExtra;
        auto result = nn::capsrv::server::detail::TryDecryptApplicationId(&dst, &isExtra, mid);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_EQ(false, isExtra);
        EXPECT_EQ(0, std::memcmp(&dst.value, "Nintendo", 8));
    }

    // 適当なアプリケーション ID で暗号化→復号化が正しくできることの確認
    for(int i = 0; i < 2000; i++)
    {
        auto src = idgen.GenerateApplicationId();
        bool isExtra = (i >= 1000);
        auto mid = nn::capsrv::server::detail::EncryptApplicationId(src, isExtra);
        nn::ncm::ApplicationId dst = {};
        bool isOutExtra;
        auto result = nn::capsrv::server::detail::TryDecryptApplicationId(&dst, &isOutExtra, mid);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_EQ(isExtra, isOutExtra);
        EXPECT_TRUE(std::memcmp(&src, &dst, sizeof(dst)) == 0);
    }

    // 適当なバイト列は天文学的に運がよくなければ復号化できないことの確認
    for(int i = 0; i < 1000; i++)
    {
        auto src = idgen.GenerateEncryptedApplicationId();
        nn::ncm::ApplicationId dst = {};
        bool isExtra;
        auto result = nn::capsrv::server::detail::TryDecryptApplicationId(&dst, &isExtra, src);
        EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidApplicationId::Includes(result));
    }

    SUCCEED();
}
