﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/image/image_ExifBuilder.h>
#include <nn/image/image_ExifExtractor.h>
#include <nn/image/image_JpegEncoder.h>
#include <nn/image/image_JpegDecoder.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_Result.h>
#include "../../Common/testCapsrv_TestFileUtility.h"

#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_ResultPrivate.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_ParseMakerNote.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_ReplaceJpegSignature.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_CalculateJpegHmac.h"


// MakerNote のバージョン
#define NNT_CAPSRV_VERSION_BYTES 0x00, 0x00, 0x00, 0x00

//--------------------------
// 署名
//--------------------------
#define NNT_CAPSRV_ENTRY_TAG_SIGNATURE_BYTES 0x00, 0x00

// テスト用の署名（適当）
#define NNT_CAPSRV_ENTRY_SIZE_SIGNATURE_BYTES 0x20, 0x00
#define NNT_CAPSRV_ENTRY_BODY_SIGNATURE_BYTES \
    0x19, 0x46, 0xBA, 0x4C, 0xB8, 0x40, 0x4C, 0x3C, 0xB7, 0x3E, 0x8A, 0xEE, 0x11, 0x8E, 0x68, 0x30, \
    0x9C, 0xDE, 0x28, 0x91, 0x70, 0x20, 0x43, 0x5D, 0xBB, 0x09, 0xBC, 0xBA, 0x82, 0x8D, 0x49, 0x09

// ゼロ埋めされた署名
#define NNT_CAPSRV_ENTRY_BODY_ZERO_SIGNATURE_BYTES \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, \
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00

// 不正に短い署名
#define NNT_CAPSRV_ENTRY_SIZE_BAD_SHORT_SIGNATURE_BYTES 0x10, 0x00
#define NNT_CAPSRV_ENTRY_BODY_BAD_SHORT_SIGNATURE_BYTES \
    0x19, 0x46, 0xBA, 0x4C, 0xB8, 0x40, 0x4C, 0x3C, 0xB7, 0x3E, 0x8A, 0xEE, 0x11, 0x8E, 0x68, 0x30

// 不正に長い署名
#define NNT_CAPSRV_ENTRY_SIZE_BAD_LONG_SIGNATURE_BYTES 0x40, 0x00
#define NNT_CAPSRV_ENTRY_BODY_BAD_LONG_SIGNATURE_BYTES \
    0x19, 0x46, 0xBA, 0x4C, 0xB8, 0x40, 0x4C, 0x3C, 0xB7, 0x3E, 0x8A, 0xEE, 0x11, 0x8E, 0x68, 0x30, \
    0x9C, 0xDE, 0x28, 0x91, 0x70, 0x20, 0x43, 0x5D, 0xBB, 0x09, 0xBC, 0xBA, 0x82, 0x8D, 0x49, 0x09, \
    0x19, 0x46, 0xBA, 0x4C, 0xB8, 0x40, 0x4C, 0x3C, 0xB7, 0x3E, 0x8A, 0xEE, 0x11, 0x8E, 0x68, 0x30, \
    0x9C, 0xDE, 0x28, 0x91, 0x70, 0x20, 0x43, 0x5D, 0xBB, 0x09, 0xBC, 0xBA, 0x82, 0x8D, 0x49, 0x09

//----------------------------
// アプリ ID
//----------------------------
#define NNT_CAPSRV_ENTRY_TAG_APPID_BYTES     0x01, 0x00

#define NNT_CAPSRV_ENTRY_SIZE_APPID_BYTES     0x10, 0x00
#define NNT_CAPSRV_APPID_BYTES     0x40, 0x46, 0x7B, 0x60, 0xDC, 0x27, 0x45, 0x99, 0x8E, 0xFA, 0xA8, 0x9D, 0xBE, 0x88, 0xCB, 0x28

//----------------------------
// エントリ
//----------------------------

#define NNT_CAPSRV_ENTRY_SIGNATURE_BYTES \
    NNT_CAPSRV_ENTRY_TAG_SIGNATURE_BYTES, \
    NNT_CAPSRV_ENTRY_SIZE_SIGNATURE_BYTES, \
    NNT_CAPSRV_ENTRY_BODY_SIGNATURE_BYTES

#define NNT_CAPSRV_ENTRY_APPID_BYTES \
    NNT_CAPSRV_ENTRY_TAG_APPID_BYTES, \
    NNT_CAPSRV_ENTRY_SIZE_APPID_BYTES, \
    NNT_CAPSRV_APPID_BYTES

namespace {
    static const size_t ExifWorkMemorySize = 1024;

    std::vector<uint8_t> CreateJpegData(const void* pMakerNote, size_t makerNoteSize, bool hasExif, bool hasMakerNote) NN_NOEXCEPT
    {
        std::mt19937 engine;
        engine.seed(12345);
        nnt::capsrv::TestRawImage raw = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(1280, 720, 4, engine);

        std::vector<char> exifWorkMemory(ExifWorkMemorySize);
        nn::image::ExifBuilder exifBuilder(exifWorkMemory.data(), exifWorkMemory.size());
        if(hasMakerNote)
        {
            exifBuilder.SetMakerNote(pMakerNote, makerNoteSize);
        }
        exifBuilder.Analyze();

        nn::image::JpegEncoder encoder;
        encoder.SetPixelData(raw.data.data(), raw.GetDimension(), 4);
        encoder.Analyze();

        std::vector<uint8_t> workMemory;
        workMemory.resize(encoder.GetAnalyzedWorkBufferSize());

        std::vector<uint8_t> jpegData;
        jpegData.resize(1280 * 720 * 3);
        size_t dataSize = 0;
        if(hasExif)
        {
            NN_ABORT_UNLESS(
                encoder.Encode(&dataSize, jpegData.data(), jpegData.size(), workMemory.data(), workMemory.size(), &exifBuilder) == nn::image::JpegStatus_Ok
            );
        }
        else
        {
            NN_ABORT_UNLESS(
                encoder.Encode(&dataSize, jpegData.data(), jpegData.size(), workMemory.data(), workMemory.size()) == nn::image::JpegStatus_Ok
            );
        }

        jpegData.resize(dataSize);
        return jpegData;
    }

    std::vector<uint8_t> ExtractMakerNote(const std::vector<uint8_t>& jpegData) NN_NOEXCEPT
    {
        const void* pExifData;
        size_t exifDataSize;
        NN_ABORT_UNLESS(
            nn::image::JpegDecoder::GetExifData(&pExifData, &exifDataSize, jpegData.data(), jpegData.size()) == nn::image::JpegStatus_Ok
        );

        std::vector<char> exifWorkMemory(ExifWorkMemorySize);
        nn::image::ExifExtractor exifExtractor(exifWorkMemory.data(), exifWorkMemory.size());
        exifExtractor.SetExifData(pExifData, exifDataSize);
        NN_ABORT_UNLESS(
            exifExtractor.Analyze() == nn::image::JpegStatus_Ok
        );

        const uint8_t* pMakerNote = nullptr;
        size_t makerNoteSize = 0;
        pMakerNote = reinterpret_cast<const uint8_t*>(exifExtractor.ExtractMakerNote(&makerNoteSize));
        NN_ABORT_UNLESS_NOT_NULL(pMakerNote);

        std::vector<uint8_t> result;
        result.assign(pMakerNote, pMakerNote + makerNoteSize);

        return result;
    }

}

TEST(UnitTest, CalculateJpegHmac)
{

    std::vector<char> workBuf;
    workBuf.resize(1024);
    // 正しく計算できるかチェック
    {
        // ゼロ署名でファイルデータを作成
        uint8_t makerNote[] =
        {
            NNT_CAPSRV_VERSION_BYTES,
            NNT_CAPSRV_ENTRY_TAG_SIGNATURE_BYTES,
            NNT_CAPSRV_ENTRY_SIZE_SIGNATURE_BYTES,
            NNT_CAPSRV_ENTRY_BODY_ZERO_SIGNATURE_BYTES,
            NNT_CAPSRV_ENTRY_APPID_BYTES,
        };
        auto jpegData = CreateJpegData(makerNote, sizeof(makerNote), true, true);

        // ゼロ署名のデータから計算
        nn::capsrv::server::detail::Signature signatureZero = {};
        nn::capsrv::server::detail::Signature signatureZeroRaw = {};
        {
            nn::Result result;
            result = nn::capsrv::server::detail::CalculateJpegHmac(&signatureZero, jpegData.data(), jpegData.size(), workBuf.data(), workBuf.size());
            EXPECT_TRUE(result.IsSuccess());
            result = nn::capsrv::server::detail::CalculateJpegHmacRaw(&signatureZeroRaw, jpegData.data(), jpegData.size());
            EXPECT_TRUE(result.IsSuccess());
        }

        auto jpegDataValidSignature = jpegData;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::capsrv::server::detail::ReplaceJpegSignature(jpegData.data(), jpegData.size(), &signatureZero, workBuf.data(), workBuf.size())
        );

        // 署名を置き換えたデータから計算
        nn::capsrv::server::detail::Signature signatureValid = {};
        nn::capsrv::server::detail::Signature signatureValidRaw = {};
        {
            nn::Result result;
            result = nn::capsrv::server::detail::CalculateJpegHmac(&signatureValid, jpegData.data(), jpegData.size(), workBuf.data(), workBuf.size());
            EXPECT_TRUE(result.IsSuccess());
            result = nn::capsrv::server::detail::CalculateJpegHmacRaw(&signatureValidRaw, jpegData.data(), jpegData.size());
            EXPECT_TRUE(result.IsSuccess());
        }

        // 一致すべきものをチェック
        EXPECT_EQ(0, std::memcmp(&signatureZero, &signatureZeroRaw, sizeof(nn::capsrv::server::detail::Signature)));
        EXPECT_EQ(0, std::memcmp(&signatureZero, &signatureValid, sizeof(nn::capsrv::server::detail::Signature)));
        // 一致すべきでないものをチェック
        EXPECT_NE(0, std::memcmp(&signatureZero, &signatureValidRaw, sizeof(nn::capsrv::server::detail::Signature)));
    }


}// NOLINT(impl/function_size)
