﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt.h>
#include <nn/os.h>

#include <nn/capsrv/capsrv_ScreenShotForApplication.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>
#include <nn/capsrv/capsrv_ConvertScreenShotOrientation.h>
#include <nn/util/util_Color.h>
#include "../../Common/testCapsrv_TestFileUtility.h"

namespace {

    class Rectangle
    {
    public:
        Rectangle() NN_NOEXCEPT
            : m_PositionX(0)
            , m_PositionY(0)
            , m_Width(0)
            , m_Height(0)
        {
        }

        // 面積が 0 のとき true
        bool IsEmpty() const { return m_Width == 0 || m_Height == 0; }

        int GetPositionX() const NN_NOEXCEPT { return m_PositionX; }
        int GetPositionY() const NN_NOEXCEPT { return m_PositionY; }
        int GetWidth() const NN_NOEXCEPT  { return m_Width;  }
        int GetHeight() const NN_NOEXCEPT { return m_Height; }

        int GetBeginX() const NN_NOEXCEPT { return m_PositionX; }
        int GetBeginY() const NN_NOEXCEPT { return m_PositionY; }
        int GetEndX() const NN_NOEXCEPT { return m_PositionX + m_Width;  }
        int GetEndY() const NN_NOEXCEPT { return m_PositionY + m_Height; }

        void SetPositionX(int value) NN_NOEXCEPT { m_PositionX = value; }
        void SetPositionY(int value) NN_NOEXCEPT { m_PositionY = value; }
        void SetWidth(int value) NN_NOEXCEPT { m_Width = value; }
        void SetHeight(int value) NN_NOEXCEPT { m_Height = value; }

        static Rectangle FromPositionAndSize(int x, int y, int w, int h) NN_NOEXCEPT
        {
            Rectangle rect;
            rect.m_PositionX = x;
            rect.m_PositionY = y;
            rect.m_Width = w;
            rect.m_Height = h;
            return rect;
        }

        Rectangle Intersect(const Rectangle& value) const NN_NOEXCEPT
        {
            int xBeg = std::max(m_PositionX, value.m_PositionX);
            int yBeg = std::max(m_PositionY, value.m_PositionY);
            int xEnd = std::max(std::min(m_PositionX + m_Width , value.m_PositionX + value.m_Width ), 0);
            int yEnd = std::max(std::min(m_PositionY + m_Height, value.m_PositionY + value.m_Height), 0);

            Rectangle rect;
            rect.m_PositionX = xBeg;
            rect.m_PositionY = yBeg;
            rect.m_Width  = xEnd - xBeg;
            rect.m_Height = yEnd - yBeg;
            return rect;
        }

    public:
        int m_PositionX;
        int m_PositionY;
        int m_Width;
        int m_Height;

    };


    void FillImage(nnt::capsrv::TestRawImage& target, nn::util::Color4u8 color) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(target.width, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(target.height, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(target.data.size(), static_cast<size_t>(4 * target.width * target.height));
        NN_SDK_REQUIRES_EQUAL(static_cast<int>(target.data.size() % 4), 0);

        uint8_t* p = target.data.data();
        for(int y = 0; y < target.height; y++)
        {
            for(int x = 0; x < target.width; x++)
            {
                p[0] = color.GetR();
                p[1] = color.GetG();
                p[2] = color.GetB();
                p[3] = color.GetA();
                p += 4;
            }
        }
    }

    void FillImage(nnt::capsrv::TestRawImage& target, nn::util::Color4u8 color, const Rectangle& rect) NN_NOEXCEPT
    {
        const Rectangle targetRect = Rectangle::FromPositionAndSize(0, 0, target.width, target.height);
        const Rectangle drawRect = targetRect.Intersect(rect);

        if(drawRect.IsEmpty())
        {
            return;
        }

        uint8_t* p = target.data.data();
        int stride = 4 * target.width;

        for(int y = drawRect.GetBeginY(); y < drawRect.GetEndY(); y++)
        {
            uint8_t* pY = p + y * stride;
            for(int x = drawRect.GetBeginX(); x < drawRect.GetEndX(); x++)
            {
                uint8_t* pXY = pY + x * 4;
                pXY[0] = color.GetR();
                pXY[1] = color.GetG();
                pXY[2] = color.GetB();
                pXY[3] = color.GetA();
            }
        }

    }

    nn::Result SaveScreenShot(nn::capsrv::ApplicationAlbumEntry* pOutEntry, bool requestOverlayNotification) NN_NOEXCEPT
    {
        nnt::capsrv::TestRawImage image;
        image.width = 1280;
        image.height = 720;
        image.data.resize(4 * image.width * image.height);

        FillImage(image, nn::util::Color4u8(128, 128, 128, 255));
        FillImage(image, nn::util::Color4u8(255,   0,   0, 255), Rectangle::FromPositionAndSize(        0,        0, 64, 64));
        FillImage(image, nn::util::Color4u8(  0, 255,   0, 255), Rectangle::FromPositionAndSize(1280 - 64,        0, 64, 64));
        FillImage(image, nn::util::Color4u8(  0,   0, 255, 255), Rectangle::FromPositionAndSize(        0, 720 - 64, 64, 64));
        FillImage(image, nn::util::Color4u8(  0,   0,   0, 255), Rectangle::FromPositionAndSize(1280 - 64, 720 - 64, 64, 64));
        NN_SDK_REQUIRES_EQUAL(image.width, 1280);
        NN_SDK_REQUIRES_EQUAL(image.height, 720);

        nn::capsrv::ApplicationAlbumEntry appEntry = {};
        nn::capsrv::OverlayNotificationRequestType ovlnReq = nn::capsrv::OverlayNotificationRequest_None;
        if(requestOverlayNotification)
        {
            ovlnReq |= nn::capsrv::OverlayNotificationRequest_All;
        }

        nn::capsrv::ScreenShotAttribute attribute;
        attribute.SetDefault();
        NN_RESULT_DO(nn::capsrv::SaveScreenShot(
            &appEntry,
            image.data.data(),
            image.data.size(),
            attribute,
            ovlnReq
        ));

        *pOutEntry = appEntry;
        NN_RESULT_SUCCESS;
    }

    nn::Result SaveScreenShotForApplication(nn::capsrv::ApplicationAlbumEntry* pOutEntry, bool requestOverlayNotification) NN_NOEXCEPT
    {
        nnt::capsrv::TestRawImage image;
        image.width = 1280;
        image.height = 720;
        image.data.resize(4 * image.width * image.height);

        FillImage(image, nn::util::Color4u8(128, 128, 128, 255));
        FillImage(image, nn::util::Color4u8(255,   0,   0, 255), Rectangle::FromPositionAndSize(        0,        0, 64, 64));
        FillImage(image, nn::util::Color4u8(  0, 255,   0, 255), Rectangle::FromPositionAndSize(1280 - 64,        0, 64, 64));
        FillImage(image, nn::util::Color4u8(  0,   0, 255, 255), Rectangle::FromPositionAndSize(        0, 720 - 64, 64, 64));
        FillImage(image, nn::util::Color4u8(  0,   0,   0, 255), Rectangle::FromPositionAndSize(1280 - 64, 720 - 64, 64, 64));
        NN_SDK_REQUIRES_EQUAL(image.width, 1280);
        NN_SDK_REQUIRES_EQUAL(image.height, 720);

        nn::capsrv::ApplicationAlbumEntry appEntry = {};
        nn::capsrv::OverlayNotificationRequestType ovlnReq = nn::capsrv::OverlayNotificationRequest_None;
        if(requestOverlayNotification)
        {
            ovlnReq |= nn::capsrv::OverlayNotificationRequest_All;
        }

        NN_RESULT_DO(nn::capsrv::SaveScreenShotForApplication(
            &appEntry,
            image.data.data(),
            image.data.size(),
            nn::capsrv::ScreenShotSize_1280x720,
            ovlnReq
        ));

        *pOutEntry = appEntry;
        NN_RESULT_SUCCESS;
    }

    nn::Result SaveScreenShotForApplicationOrientation(nn::capsrv::ApplicationAlbumEntry* pOutEntry, nn::capsrv::ScreenShotOrientation orientation) NN_NOEXCEPT
    {
        int w = 1280;
        int h = 720;
        int s = 64;

        nnt::capsrv::TestRawImage image;
        image.width = w;
        image.height = h;
        image.data.resize(4 * image.width * image.height);

        nn::capsrv::OverlayNotificationRequestType ovlnReq = nn::capsrv::OverlayNotificationRequest_All;

        nn::util::Color4u8 topColor(255, 0, 0, 255);
        nn::util::Color4u8 bottomColor(0, 0, 255, 255);

        FillImage(image, nn::util::Color4u8(128, 128, 128, 255));
        switch(orientation)
        {
        case nn::capsrv::ScreenShotOrientation_Default:
            {
                FillImage(image, topColor   , Rectangle::FromPositionAndSize(0,     0, w, s));
                FillImage(image, bottomColor, Rectangle::FromPositionAndSize(0, h - s, w, s));
                break;
            }
        case nn::capsrv::ScreenShotOrientation_Rotate90:
            {
                FillImage(image, topColor   , Rectangle::FromPositionAndSize(w - s, 0, s, h));
                FillImage(image, bottomColor, Rectangle::FromPositionAndSize(    0, 0, s, h));
                break;
            }
        case nn::capsrv::ScreenShotOrientation_Rotate180:
            {
                FillImage(image, topColor   , Rectangle::FromPositionAndSize(0, h - s, w, s));
                FillImage(image, bottomColor, Rectangle::FromPositionAndSize(0,     0, w, s));
                break;
            }
        case nn::capsrv::ScreenShotOrientation_Rotate270:
            {
                FillImage(image, topColor   , Rectangle::FromPositionAndSize(    0, 0, s, h));
                FillImage(image, bottomColor, Rectangle::FromPositionAndSize(w - s, 0, s, h));
                break;
            }
        default: NN_UNEXPECTED_DEFAULT;
        }
        NN_SDK_REQUIRES_EQUAL(image.width, 1280);
        NN_SDK_REQUIRES_EQUAL(image.height, 720);

        nn::capsrv::ScreenShotAttribute attribute;
        attribute.SetDefault();
        attribute.size = nn::capsrv::ScreenShotSize_1280x720;
        attribute.orientation = orientation;

        nn::capsrv::ApplicationAlbumEntry appEntry = {};
        NN_RESULT_DO(nn::capsrv::SaveScreenShotForApplication(
            &appEntry,
            image.data.data(),
            image.data.size(),
            attribute,
            ovlnReq
        ));

        *pOutEntry = appEntry;
        NN_RESULT_SUCCESS;
    }

    nn::Result SaveScreenShotForApplicationCopyright(nn::capsrv::ApplicationAlbumEntry* pOutEntry, bool isCopyrightImageComposited, bool isUneditableAreaSet) NN_NOEXCEPT
    {
        int w = 1280;
        int h = 720;
        int s = 64;

        nnt::capsrv::TestRawImage image;
        image.width = w;
        image.height = h;
        image.data.resize(4 * image.width * image.height);

        nn::capsrv::OverlayNotificationRequestType ovlnReq = nn::capsrv::OverlayNotificationRequest_All;

        nn::util::Color4u8 topColor(255, 0, 0, 255);
        nn::util::Color4u8 bottomColor(0, 0, 255, 255);

        FillImage(image, nn::util::Color4u8(128, 128, 128, 255));
        FillImage(image, topColor   , Rectangle::FromPositionAndSize(0,     0, w, s));
        FillImage(image, bottomColor, Rectangle::FromPositionAndSize(0, h - s, w, s));
        NN_SDK_REQUIRES_EQUAL(image.width, 1280);
        NN_SDK_REQUIRES_EQUAL(image.height, 720);

        nn::capsrv::ScreenShotAttribute attribute;
        attribute.SetDefault();
        attribute.size = nn::capsrv::ScreenShotSize_1280x720;
        attribute.flags |= (isCopyrightImageComposited ? nn::capsrv::detail::ScreenShotAttributeFlag_IsCopyrightImageComposited : 0);
        attribute.flags |= (isUneditableAreaSet ? nn::capsrv::detail::ScreenShotAttributeFlag_HasUneditableArea : 0);
        attribute.uneditableAreaCoordinateX  = isUneditableAreaSet ? 320 : 0;
        attribute.uneditableAreaCoordinateY  = isUneditableAreaSet ? 180 : 0;
        attribute.uneditableAreaWidth        = isUneditableAreaSet ? 640 : 0;
        attribute.uneditableAreaHeight       = isUneditableAreaSet ? 360 : 0;

        nn::capsrv::ApplicationAlbumEntry appEntry = {};
        NN_RESULT_DO(nn::capsrv::SaveScreenShotForApplication(
            &appEntry,
            image.data.data(),
            image.data.size(),
            attribute,
            ovlnReq
        ));

        *pOutEntry = appEntry;
        NN_RESULT_SUCCESS;
    }

}

TEST(SaveApplicationScreenShot, Save)
{
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));


    nn::capsrv::ApplicationAlbumEntry entry = {};
    {
        NN_LOG("InitializeForApplication()\n");
        nn::capsrv::InitializeForApplication();
        NN_UTIL_SCOPE_EXIT {
            NN_LOG("FinalizeForApplication()\n");
            nn::capsrv::FinalizeForApplication();
        };

        // 保存
        NN_LOG("SaveScreenShotForApplication()\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(SaveScreenShotForApplication(&entry, true));

        // オーバーレイ確認のため待ち
        NN_LOG("Wait for 3sec to see overlay disp\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

    {
        NN_LOG("InitializeAlbumAccess()\n");
        nn::capsrv::InitializeAlbumAccess();
        NN_UTIL_SCOPE_EXIT {
            NN_LOG("FinalizeAlbumAccess()\n");
            nn::capsrv::FinalizeAlbumAccess();
        };

        // AlbumEntry を取得
        nn::capsrv::AlbumEntry e1 ={};
        NN_LOG("GetAlbumEntryFromApplicationAlbumEntry(2)\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumEntryFromApplicationAlbumEntry(&e1, entry));

        nn::capsrv::AlbumEntry e2 = {};
        NN_LOG("GetAlbumEntryFromApplicationAlbumEntry(3)\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumEntryFromApplicationAlbumEntry(&e2, entry, e1.fileId.applicationId));

        // 同じ AlbumEntry がとれていることの確認
        NN_LOG("Checking 2 AlbumEntries are same\n");
        EXPECT_EQ(0, std::memcmp(&e1, &e2, sizeof(nn::capsrv::AlbumEntry)));

        // 読込んでみる
        NN_LOG("Checking the screenshot file can be read\n");
        std::vector<uint8_t> dataByEntry;
        {
            dataByEntry.resize(static_cast<size_t>(nn::capsrv::AlbumFileSizeLimit_ScreenShot));
            size_t fileSize = 0;
            EXPECT_TRUE(nn::capsrv::LoadAlbumScreenShotFile(&fileSize, dataByEntry.data(), dataByEntry.size(), &e2.fileId).IsSuccess());
            dataByEntry.resize(fileSize);
        }
    }
}

TEST(SaveApplicationScreenShot, Save_Orientation)
{
    {
        NN_LOG("InitializeForApplication()\n");
        nn::capsrv::InitializeForApplication();
        nn::capsrv::InitializeAlbumAccess();
        NN_UTIL_SCOPE_EXIT {
            NN_LOG("FinalizeForApplication()\n");
            nn::capsrv::FinalizeForApplication();
            nn::capsrv::FinalizeAlbumAccess();
        };

        nn::capsrv::ScreenShotOrientation orientationList[] = {
            nn::capsrv::ScreenShotOrientation_Default,
            nn::capsrv::ScreenShotOrientation_Rotate90,
            nn::capsrv::ScreenShotOrientation_Rotate180,
            nn::capsrv::ScreenShotOrientation_Rotate270,
        };
        int orientationCount = sizeof(orientationList) / sizeof(orientationList[0]);

        std::vector<uint8_t> loadBuf;
        std::vector<uint8_t> workBuf;
        for(int i = 0; i < orientationCount; i++)
        {
            auto ori = orientationList[i];
            nn::capsrv::ApplicationAlbumEntry entry = {};
            // 保存
            NN_LOG("SaveScreenShotForApplication() orientation = %d\n", ori);
            NN_ABORT_UNLESS_RESULT_SUCCESS(SaveScreenShotForApplicationOrientation(&entry, ori));

            // オーバーレイ確認のため待ち
            NN_LOG("Wait for 3sec to see overlay disp\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

            nn::capsrv::AlbumEntry e = {};
            NN_LOG("GetAlbumEntryFromApplicationAlbumEntry()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumEntryFromApplicationAlbumEntry(&e, entry));

            // ロードしてみる
            int width = 0;
            int height = 0;
            const nn::capsrv::AppletData zeroAppletData = {};
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData = {};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &width, &height, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &e.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, width);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, height);
            EXPECT_EQ(nn::capsrv::ScreenShotSize_1280x720, attribute.size);
            EXPECT_EQ(ori, attribute.orientation);
            EXPECT_EQ(nn::capsrv::AlbumFileDescription_ScreenShotSaved, attribute.description);
            EXPECT_EQ(1, attribute.frameCount);
            EXPECT_EQ(0, attribute.frameRateNumerator);
            EXPECT_EQ(0, attribute.frameRateDenominator);
            EXPECT_EQ(0, attribute.dataDurationMilliseconds);
            EXPECT_EQ(0, attribute.keyFrameInterval);
            EXPECT_EQ(0, std::memcmp(&zeroAppletData, &appletData, sizeof(nn::capsrv::AppletData)));

            nn::capsrv::ScreenShotAttribute attributeFull = {};
            nn::capsrv::AppletData appletDataFull = {};
            loadBuf.resize(4 * 1280 * 720);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &width, &height, &attributeFull, &appletDataFull, loadBuf.data(), loadBuf.size(), &e.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(1280, width);
            EXPECT_EQ(720, height);
            EXPECT_EQ(0, std::memcmp(&attribute, &attributeFull, sizeof(nn::capsrv::ScreenShotAttribute)));
            EXPECT_EQ(0, std::memcmp(&appletData, &appletDataFull, sizeof(nn::capsrv::AppletData)));
        }
    }
}

TEST(SaveApplicationScreenShot, Save_Copyright)
{
    {
        NN_LOG("InitializeForApplication()\n");
        nn::capsrv::InitializeForApplication();
        nn::capsrv::InitializeAlbumAccess();
        NN_UTIL_SCOPE_EXIT {
            NN_LOG("FinalizeForApplication()\n");
            nn::capsrv::FinalizeForApplication();
            nn::capsrv::FinalizeAlbumAccess();
        };

        std::vector<uint8_t> loadBuf;
        std::vector<uint8_t> workBuf;
        for(int i = 0; i < 4; i++)
        {
            bool isCopyrightImageComposited = (i >= 2);
            bool isUneditableAreaSet        = (i == 1 || i == 3);

            nn::capsrv::ApplicationAlbumEntry entry = {};
            // 保存
            NN_LOG("SaveScreenShotForApplication() isCopyright=%d isUneditableArea=%d\n", isCopyrightImageComposited, isUneditableAreaSet);
            NN_ABORT_UNLESS_RESULT_SUCCESS(SaveScreenShotForApplicationCopyright(&entry, isCopyrightImageComposited, isUneditableAreaSet));

            // オーバーレイ確認のため待ち
            NN_LOG("Wait for 3sec to see overlay disp\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

            nn::capsrv::AlbumEntry e = {};
            NN_LOG("GetAlbumEntryFromApplicationAlbumEntry()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumEntryFromApplicationAlbumEntry(&e, entry));

            // ロードしてみる
            int width = 0;
            int height = 0;
            const nn::capsrv::AppletData zeroAppletData = {};
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData = {};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &width, &height, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &e.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, width);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, height);
            EXPECT_EQ(nn::capsrv::ScreenShotSize_1280x720, attribute.size);
            EXPECT_EQ(nn::capsrv::AlbumFileDescription_ScreenShotSaved, attribute.description);
            EXPECT_EQ(1, attribute.frameCount);
            EXPECT_EQ(0, attribute.frameRateNumerator);
            EXPECT_EQ(0, attribute.frameRateDenominator);
            EXPECT_EQ(0, attribute.dataDurationMilliseconds);
            EXPECT_EQ(0, attribute.keyFrameInterval);
            EXPECT_EQ(0, std::memcmp(&zeroAppletData, &appletData, sizeof(nn::capsrv::AppletData)));
            EXPECT_EQ(isCopyrightImageComposited, attribute.IsCopyrightImageComposited());
            EXPECT_EQ(isUneditableAreaSet, attribute.HasUneditableArea());
            if (isUneditableAreaSet)
            {
                EXPECT_EQ(320, attribute.uneditableAreaCoordinateX);
                EXPECT_EQ(180, attribute.uneditableAreaCoordinateY);
                EXPECT_EQ(640, attribute.uneditableAreaWidth);
                EXPECT_EQ(360, attribute.uneditableAreaHeight);
            }

            nn::capsrv::ScreenShotAttribute attributeFull = {};
            nn::capsrv::AppletData appletDataFull = {};
            loadBuf.resize(4 * 1280 * 720);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &width, &height, &attributeFull, &appletDataFull, loadBuf.data(), loadBuf.size(), &e.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(1280, width);
            EXPECT_EQ(720, height);
            EXPECT_EQ(0, std::memcmp(&attribute, &attributeFull, sizeof(nn::capsrv::ScreenShotAttribute)));
            EXPECT_EQ(0, std::memcmp(&appletData, &appletDataFull, sizeof(nn::capsrv::AppletData)));
            EXPECT_EQ(isCopyrightImageComposited, attribute.IsCopyrightImageComposited());
            EXPECT_EQ(isUneditableAreaSet, attribute.HasUneditableArea());
            if (isUneditableAreaSet)
            {
                EXPECT_EQ(320, attribute.uneditableAreaCoordinateX);
                EXPECT_EQ(180, attribute.uneditableAreaCoordinateY);
                EXPECT_EQ(640, attribute.uneditableAreaWidth);
                EXPECT_EQ(360, attribute.uneditableAreaHeight);
            }
        }
    }
}

// 2NUP ファーム用の暫定 API のチェック
TEST(SaveApplicationScreenShot, Save_Dev)
{
    nn::capsrv::InitializeScreenShotControl();
    NN_UTIL_SCOPE_EXIT { nn::capsrv::FinalizeScreenShotControl(); };
    nn::capsrv::InitializeAlbumControl();
    NN_UTIL_SCOPE_EXIT { nn::capsrv::FinalizeAlbumControl(); };
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT { nn::capsrv::FinalizeAlbumAccess(); };

    // 保存
    NN_LOG("Checking SaveScreenShot()\n");
    nn::capsrv::ApplicationAlbumEntry appEntry = {};
    EXPECT_TRUE(SaveScreenShot(&appEntry, true).IsSuccess());

    // オーバーレイ確認のため待ち
    NN_LOG("Wait for 3sec to see overlay disp\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // AlbumEntry を取得
    NN_LOG("Checking GetAlbumEntryFromApplicationAlbumEntry(aruid)\n");
    nn::capsrv::AlbumEntry entry = {};
    EXPECT_TRUE(nn::capsrv::GetAlbumEntryFromApplicationAlbumEntry(&entry, appEntry).IsSuccess());

    // AlbumEntry を取得（その2）
    NN_LOG("Checking GetAlbumEntryFromApplicationAlbumEntry(applicationId)\n");
    nn::capsrv::AlbumEntry entry2 = {};
    EXPECT_TRUE(nn::capsrv::GetAlbumEntryFromApplicationAlbumEntry(&entry2, appEntry, entry.fileId.applicationId).IsSuccess());

    // 同じ AlbumEntry がとれていることの確認
    NN_LOG("Checking 2 AlbumEntries are same\n");
    EXPECT_EQ(0, std::memcmp(&entry, &entry2, sizeof(nn::capsrv::AlbumEntry)));

    // 読込んでみる
    NN_LOG("Checking the screenshot file can be read\n");
    std::vector<uint8_t> dataByEntry;
    {
        dataByEntry.resize(static_cast<size_t>(nn::capsrv::AlbumFileSizeLimit_ScreenShot));
        size_t fileSize = 0;
        EXPECT_TRUE(nn::capsrv::LoadAlbumScreenShotFile(&fileSize, dataByEntry.data(), dataByEntry.size(), &entry.fileId).IsSuccess());
        dataByEntry.resize(fileSize);
    }
}

// Enum の変換が間違っていないことの確認
TEST(SaveApplicationScreenShot, Convert_OrientationEnum)
{
    // ScreenShotOrientation -> ImageOrientation
    EXPECT_EQ(nn::album::ImageOrientation_None     , nn::capsrv::ConvertScreenShotOrientationToImageOrientation(nn::capsrv::ScreenShotOrientation_Default  ));
    EXPECT_EQ(nn::album::ImageOrientation_Rotate90 , nn::capsrv::ConvertScreenShotOrientationToImageOrientation(nn::capsrv::ScreenShotOrientation_Rotate90 ));
    EXPECT_EQ(nn::album::ImageOrientation_Rotate180, nn::capsrv::ConvertScreenShotOrientationToImageOrientation(nn::capsrv::ScreenShotOrientation_Rotate180));
    EXPECT_EQ(nn::album::ImageOrientation_Rotate270, nn::capsrv::ConvertScreenShotOrientationToImageOrientation(nn::capsrv::ScreenShotOrientation_Rotate270));

    // ImageOrientation -> ScreenShotOrientation
    EXPECT_EQ(nn::capsrv::ScreenShotOrientation_Default  , nn::capsrv::ConvertImageOrientationToScreenShotOrientation(nn::album::ImageOrientation_None     ));
    EXPECT_EQ(nn::capsrv::ScreenShotOrientation_Rotate90 , nn::capsrv::ConvertImageOrientationToScreenShotOrientation(nn::album::ImageOrientation_Rotate90 ));
    EXPECT_EQ(nn::capsrv::ScreenShotOrientation_Rotate180, nn::capsrv::ConvertImageOrientationToScreenShotOrientation(nn::album::ImageOrientation_Rotate180));
    EXPECT_EQ(nn::capsrv::ScreenShotOrientation_Rotate270, nn::capsrv::ConvertImageOrientationToScreenShotOrientation(nn::album::ImageOrientation_Rotate270));
}
