﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/mem/mem_StandardAllocator.h>

#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_CaptureModule.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_MemoryPool.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_ImageBuffer.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_DisplayBuffer.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_DisplayCapture.h"
#include <nn/gfx/util/gfx_TransientMemoryPoolAllocatorHolder.h>

#include "testCapsrv_Static.h"

static const int RepeatCount = 1000;

TEST(DisplayCapture, InitializeFinalizeRgba8)
{
    nn::capsrv::capture::InitializeCaptureModule();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::capture::FinalizeCaptureModule(); };

    auto pModule = nn::capsrv::capture::GetCaptureModule();
    nn::capsrv::capture::MemoryPool pool;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pool.Initialize(pModule, g_Storage, sizeof(g_Storage)));
    NN_UTIL_SCOPE_EXIT{ pool.Finalize(); };

    NN_LOG("setup info\n");
    nn::capsrv::capture::ImageBufferInfo info;
    info.width = 1280;
    info.height = 720;
    info.format = nn::capsrv::capture::ImageFormat_Rgba8;
    size_t size = nn::capsrv::capture::ImageBuffer::GetRequiredMemorySize(info);
    size_t alignment = nn::capsrv::capture::ImageBuffer::GetRequiredMemoryAlignment(pModule, info);
    NN_LOG("  size  : %llu\n", size);
    NN_LOG("  align : %llu\n", alignment);

    nn::gfx::MemoryPool memoryPool; // ダミー
    nn::gfx::util::TransientMemoryPoolAllocatorHolder<2> poolAllocatorHolder;
    poolAllocatorHolder.Initialize(&memoryPool, 0, sizeof(g_Storage), nn::gfx::util::TransientMemoryPoolAllocator::AlignmentMax);
    NN_UTIL_SCOPE_EXIT{ poolAllocatorHolder.Finalize(); };
    auto pPoolAllocator = poolAllocatorHolder.Get();
    pPoolAllocator->FillHistory();

    nn::capsrv::capture::DisplayCapture cap;
    cap.Initialize(pModule, "tPhoto", nn::vi::LayerStack_Screenshot, info.width, info.height, nn::capsrv::capture::ImageFormat_Rgba8);
    NN_UTIL_SCOPE_EXIT { cap.Finalize(); };

    auto initializeImageBuffer = [&](nn::capsrv::capture::ImageBuffer* p){
        auto offset = pPoolAllocator->Allocate(size, alignment);
        NN_ASSERT_GREATER_EQUAL(offset, 0);
        p->Initialize(pModule, info, &pool, offset, size);
    };
    auto finalizeImageBuffer = [&](nn::capsrv::capture::ImageBuffer* p){
        p->Finalize();
    };

    auto initializeDisplayBuffer = [&](nn::capsrv::capture::DisplayBuffer* p, nn::capsrv::capture::ImageBuffer* pImg){
        p->InitializeRgba(pModule, nn::capsrv::capture::ImageFormat_Rgba8, pImg);
    };
    auto finalizeDisplayBuffer = [&](nn::capsrv::capture::DisplayBuffer* p){
        p->Finalize();
    };

    auto f = [&](){
        pPoolAllocator->Free();
        pPoolAllocator->Begin();
        nn::capsrv::capture::ImageBuffer img;
        nn::capsrv::capture::DisplayBuffer dsp;
        initializeImageBuffer(&img);
        initializeDisplayBuffer(&dsp, &img);

        cap.SetCaptureBuffer(0, &dsp);

        cap.UnsetCaptureBuffer(0);

        finalizeDisplayBuffer(&dsp);
        finalizeImageBuffer(&img);
        pPoolAllocator->End();
    };

    NN_LOG("first initializing-finalizing\n");
    f();

    size_t freeMemoryAfterFirst = g_GraphicsAllocator.GetTotalFreeSize();

    NN_LOG("repeat initializing-finalizing\n");
    for(int i = 0; i < RepeatCount; i++)
    {
        if(i % 100 == 0)
        {
            NN_LOG("  %d-th loop\n", i);
        }
        f();
    }


    size_t freeMemoryAfterRepeat = g_GraphicsAllocator.GetTotalFreeSize();

    NN_LOG("free memory:\n  after first %llu\n  after repeat %llu\n", freeMemoryAfterFirst, freeMemoryAfterRepeat);
    // 多分同じになる
    EXPECT_EQ(freeMemoryAfterFirst, freeMemoryAfterRepeat);
}
