﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/mem/mem_StandardAllocator.h>

#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_CaptureModule.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_MemoryPool.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_ImageBuffer.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_DisplayBuffer.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_DisplayCapture.h"
#include <nn/gfx/util/gfx_TransientMemoryPoolAllocatorHolder.h>
#include <nn/capsrv/capsrv_ThumbnailFormat.h>
#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>

#include "testCapsrv_Utility.h"

#include "testCapsrv_Static.h"

TEST(Capture, SimpleRgba8888)
{
    const int Width = nnt::capsrv::DefaultWidth;
    const int Height = nnt::capsrv::DefaultHeight;
    const nn::capsrv::capture::ImageFormat Format = nn::capsrv::capture::ImageFormat_Rgba8;
    const nn::vi::LayerStack LayerStack = nn::vi::LayerStack_Default;

    nn::capsrv::capture::InitializeCaptureModule();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::capture::FinalizeCaptureModule(); };

    nnt::capsrv::CaptureMemoryPool pool(g_Storage, sizeof(g_Storage));
    nnt::capsrv::CaptureDisplayBuffer dsp(Width, Height, Format, &pool);
    nnt::capsrv::CaptureDisplayCapture cap(Width, Height, Format, LayerStack);

    cap->SetCaptureBuffer(0, dsp.Get());

    // 描画の初期化
    g_Renderer.Initialize();
    NN_UTIL_SCOPE_EXIT { g_Renderer.Finalize(); };

    std::vector<uint8_t> frameData;
    frameData.assign(4 * Width * Height, 255);
    std::vector<uint8_t> captureData;
    captureData.assign(4 * Width * Height, 255);

    {
        // 出画
        g_Renderer.Present(frameData.data(), frameData.size());

        // キャプチャ
        dsp->FlushCache();
        auto p = cap->AcquireCaptureBuffer(nn::TimeSpan::FromMilliSeconds(100));
        size_t writtenSize = 0;
        dsp->GetImageBuffer()->WriteToMemory(&writtenSize, captureData.data(), captureData.size(), nn::capsrv::capture::ImageFormat_Rgba8, nullptr);

        // RGBA8 なら完全に一致するはず
        EXPECT_EQ(frameData.size(), writtenSize);
        EXPECT_EQ(0, std::memcmp(frameData.data(), captureData.data(), writtenSize));

        cap->ReleaseCaptureBuffer(p);
    }
}

TEST(Capture, SimpleYuv440)
{
    const int Width = nnt::capsrv::DefaultWidth;
    const int Height = nnt::capsrv::DefaultHeight;
    const nn::capsrv::capture::ImageFormat Format = nn::capsrv::capture::ImageFormat_Yuv440_Bt601_Planar;
    const nn::vi::LayerStack LayerStack = nn::vi::LayerStack_Default;

    auto pModule = nn::capsrv::capture::GetCaptureModule();
    nn::capsrv::capture::InitializeCaptureModule();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::capture::FinalizeCaptureModule(); };

    nnt::capsrv::CaptureMemoryPool pool(g_Storage, sizeof(g_Storage));
    nnt::capsrv::CaptureDisplayBuffer dsp(Width, Height, Format, &pool);
    nnt::capsrv::CaptureDisplayCapture cap(Width, Height, Format, LayerStack);
    nnt::capsrv::CaptureImageBuffer tmp(Width, Height, nn::capsrv::capture::ImageFormat_Rgba8, &pool);

    // オーバーレイサイズ
    nnt::capsrv::CaptureImageBuffer ovlThumb(nn::capsrv::ThumbnailImageSize_Width, nn::capsrv::ThumbnailImageSize_Height, nn::capsrv::capture::ImageFormat_Rgba8, &pool);
    // Viewer サイズ
    nnt::capsrv::CaptureImageBuffer viwerThumb(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, nn::capsrv::capture::ImageFormat_Rgba8, &pool);

    cap->SetCaptureBuffer(0, dsp.Get());

    // 描画の初期化
    g_Renderer.Initialize();
    NN_UTIL_SCOPE_EXIT { g_Renderer.Finalize(); };

    std::vector<uint8_t> frameData;
    frameData.assign(4 * Width * Height, 255);
    std::vector<uint8_t> captureData;
    captureData.assign(4 * Width * Height, 255);

    {
        // 出画
        g_Renderer.Present(frameData.data(), frameData.size());

        // キャプチャ
        dsp->FlushCache();
        auto p = cap->AcquireCaptureBuffer(nn::TimeSpan::FromMilliSeconds(100));

        g_Renderer.Present(captureData.data(), captureData.size());

        tmp->FlushCache();
        p->CopyToImageBuffer(tmp.Get(), pModule, nullptr, nn::capsrv::capture::FilterMode_Nicest);
        p->CopyToImageBuffer(ovlThumb.Get(), pModule, nullptr, nn::capsrv::capture::FilterMode_Nicest);
        p->CopyToImageBuffer(viwerThumb.Get(), pModule, nullptr, nn::capsrv::capture::FilterMode_Nicest);

        size_t writtenSize = 0;
        tmp->WriteToMemory(&writtenSize, captureData.data(), captureData.size(), nn::capsrv::capture::ImageFormat_Rgba8, nullptr);

        cap->ReleaseCaptureBuffer(p);

        g_Renderer.Present(captureData.data(), captureData.size());
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    }
}
