﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"

// libnn_capsrvServer の関数
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/album/capsrvServer_AlbumPathUtility.h"

namespace {

    class RandomFileIdGenerator
    {
        static const uint32_t RandomSeed = 0xD94F4858;

    public:
        RandomFileIdGenerator()
        {
            m_Engine.seed(RandomSeed);
        }

        nn::capsrv::AlbumFileId GenerateFileId(nn::capsrv::AlbumStorageType storage)
        {
            nn::capsrv::AlbumFileId id = {};
            id.storage = storage;
            id.contents = nn::capsrv::AlbumFileContents_ScreenShot;
            id.time.year   = static_cast<uint16_t>(2000 + (m_Dist(m_Engine)) % 100);
            id.time.month  = static_cast<uint8_t>(1 + m_Dist(m_Engine) % 12);
            id.time.day    = static_cast<uint8_t>(1 + m_Dist(m_Engine) % 28);
            id.time.hour   = static_cast<uint8_t>(m_Dist(m_Engine) % 24);
            id.time.minute = static_cast<uint8_t>(m_Dist(m_Engine) % 60);
            id.time.second = static_cast<uint8_t>(m_Dist(m_Engine) % 60);
            id.time.id     = static_cast<uint8_t>(m_Dist(m_Engine) % 100);
            id.applicationId.value = (static_cast<uint64_t>(m_Dist(m_Engine)) << 32) + static_cast<uint64_t>(m_Count);
            m_Count++;
            return id;
        }

    private:
        std::mt19937 m_Engine;
        std::uniform_int_distribution<uint32_t> m_Dist;
        int m_Count;
    };

    void CreateFile(const nn::capsrv::AlbumFileId& id)
    {
        auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(id);
        nnt::capsrv::DirectAlbumAccessor::CreateParentDirectories(path.c_str());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateFile(path.c_str(), 0));
    }

    void Browse(int64_t* totalCount, char* path, int pathLength)
    {
        {
            nn::fs::DirectoryHandle h;
            int64_t fileCount = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&h, path, nn::fs::OpenDirectoryMode_File));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(h);
            };
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&fileCount, h));
            *totalCount += fileCount;
        }

        {
            nn::fs::DirectoryHandle h;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&h, path, nn::fs::OpenDirectoryMode_Directory));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(h);
            };

            static const size_t DirEntryCapacity = 32;
            nn::fs::DirectoryEntry entries[DirEntryCapacity];
            for (;;)
            {
                int64_t entryCount = 0;
                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    nn::fs::ReadDirectory(&entryCount, entries, h, DirEntryCapacity));

                for (int64_t i = 0; i < entryCount; i++)
                {
                    int additionalLength = nn::util::SNPrintf(path + pathLength, nn::fs::EntryNameLengthMax - pathLength, "%s/", entries[i].name);
                    Browse(totalCount, path, pathLength + additionalLength);
                }

                if (entryCount < DirEntryCapacity)
                {
                    break;
                }
            }
        }
    }

    void RunBenchmark(nn::capsrv::AlbumStorageType storage, int fileCount, RandomFileIdGenerator& idgen)
    {
        int n = 0;

        // 足りない分のファイルを追加で作成
        nn::capsrv::GetAlbumFileCount(&n, storage);
        for(int i = 0; i < fileCount - n; i++)
        {
            auto id = idgen.GenerateFileId(storage);
            CreateFile(id);
        }

        {
            auto path = nnt::capsrv::DirectAlbumAccessor::GetRootPath(storage);
            int64_t count = 0;
            nn::os::FenceMemoryAnyAny();
            auto startTick = nn::os::GetSystemTick();
            nn::os::FenceMemoryAnyAny();
            {
                static char s_PathBuffer[nn::fs::EntryNameLengthMax];
                int pathLen = nn::util::Strlcpy(s_PathBuffer, path, static_cast<int>(std::strlen(path)) + 1);
                Browse(&count, s_PathBuffer, pathLen);
            }
            nn::os::FenceMemoryAnyAny();
            auto endTick = nn::os::GetSystemTick();
            nn::os::FenceMemoryAnyAny();

            auto time = (endTick - startTick).ToTimeSpan();
            auto time_microsec = time.GetMicroSeconds();

            NN_LOG("[benckmark][count] storage %d, count %d, time %lld microsec.\n", static_cast<int>(storage), fileCount, time_microsec);

            EXPECT_EQ(fileCount, count);
        }

        {
            auto path = nnt::capsrv::DirectAlbumAccessor::GetRootPath(storage);
            std::vector<nn::fs::DirectoryEntry> list;
            list.resize(fileCount);
            int64_t count = 0;

            nn::os::FenceMemoryAnyAny();
            auto startTick = nn::os::GetSystemTick();
            nn::os::FenceMemoryAnyAny();
            {
                nn::fs::DirectoryHandle h;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&h, path, nn::fs::OpenDirectoryMode_File));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseDirectory(h);
                };
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadDirectory(&count, list.data(), h, static_cast<int64_t>(list.size())));
            }
            nn::os::FenceMemoryAnyAny();
            auto endTick = nn::os::GetSystemTick();
            nn::os::FenceMemoryAnyAny();

            auto time = (endTick - startTick).ToTimeSpan();
            auto time_microsec = time.GetMicroSeconds();

            NN_LOG("[benckmark][list] storage %d, count %d, time %lld microsec.\n", static_cast<int>(storage), fileCount, time_microsec);
        }
    }
}

TEST(Benchmark, PlaneFileList)
{
    nn::capsrv::InitializeAlbumAccess();

    NN_LOG("Cleaning up album\n");
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_LOG("benchmark start.\n");
    {
        RandomFileIdGenerator idgen;
        RunBenchmark(nn::capsrv::AlbumStorage_Nand, 1000, idgen);
    }

    if(nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        RandomFileIdGenerator idgen;
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  1000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  2000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  3000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  4000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  5000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  6000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  7000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  8000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  9000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd, 10000, idgen);
    }
    NN_LOG("benckmark complete.\n");

    NN_LOG("Cleaning up album\n");
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
}
