﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_TestFileUtility.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_Config.h"

namespace {

    class RandomFileIdGenerator
    {
        static const uint32_t RandomSeed = 0xD94F4858;

    public:
        RandomFileIdGenerator()
        {
            m_Engine.seed(RandomSeed);
        }

        nn::capsrv::AlbumFileId GenerateFileId(nn::capsrv::AlbumStorageType storage)
        {
            nn::capsrv::AlbumFileId id = {};
            id.storage = storage;
            id.contents = nn::capsrv::AlbumFileContents_ScreenShot;
            id.time.year   = static_cast<uint16_t>(2000 + (m_Dist(m_Engine)) % 100);
            id.time.month  = static_cast<uint8_t>(1 + m_Dist(m_Engine) % 12);
            id.time.day    = static_cast<uint8_t>(1 + m_Dist(m_Engine) % 28);
            id.time.hour   = static_cast<uint8_t>(m_Dist(m_Engine) % 24);
            id.time.minute = static_cast<uint8_t>(m_Dist(m_Engine) % 60);
            id.time.second = static_cast<uint8_t>(m_Dist(m_Engine) % 60);
            id.time.id     = static_cast<uint8_t>(m_Dist(m_Engine) % 100);
            id.applicationId.value = (static_cast<uint64_t>(m_Dist(m_Engine)) << 32) + static_cast<uint64_t>(m_Count);
            m_Count++;
            return id;
        }

    private:
        std::mt19937 m_Engine;
        std::uniform_int_distribution<uint32_t> m_Dist;
        int m_Count;
    };

    void CreateFile(std::mt19937& engine, const nn::capsrv::AlbumFileId& id)
    {
        using namespace nnt::capsrv;
        TestScreenShotFileMetaData meta;
        meta.time = id.time;
        meta.applicationId = id.applicationId;
        meta.orientation = nn::capsrv::ScreenShotOrientation_Default;

        static const size_t ImageBlockSize = 16;
        auto raw = TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::server::ScreenShotWidth, nn::capsrv::server::ScreenShotHeight, ImageBlockSize, engine);
        auto jpeg = TestFileUtility::CreateScreenShotFileData(raw, meta, nnt::capsrv::TestFileSignaturePattern_Valid);

        auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(id);
        DirectAlbumAccessor::SaveFile(jpeg, path.c_str());
    }

    void RunBenchmark(nn::capsrv::AlbumStorageType storage, int fileCount, RandomFileIdGenerator& idgen)
    {
        int n = 0;

        uint32_t seed = 0xc83982c9;
        std::mt19937 engine (seed);

        // 足りない分のファイルを追加で作成
        nn::capsrv::GetAlbumFileCount(&n, storage);
        for(int i = 0; i < fileCount - n; i++)
        {
            auto id = idgen.GenerateFileId(storage);
            CreateFile(engine, id);
        }

        NN_LOG("[benchmark][load] %d files created\n", fileCount);

        int actualFileCount = 0;
        std::vector<nn::capsrv::AlbumEntry> albumEntries;
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::capsrv::GetAlbumFileCount(&actualFileCount, storage));

            EXPECT_EQ(fileCount, actualFileCount);

            albumEntries.resize(fileCount);
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::capsrv::GetAlbumFileList(&n, albumEntries.data(), fileCount, storage));
        }

        {
            int64_t minTimeUs = INT64_MAX;
            int64_t maxTimeUs = INT64_MIN;
            int64_t avgTimeUs = 0;

            for (size_t idx = 0; idx < static_cast<size_t>(actualFileCount); idx++)
            {
                nn::os::FenceMemoryAnyAny();
                auto startTick = nn::os::GetSystemTick();
                nn::os::FenceMemoryAnyAny();
                {
                    static uint8_t s_LoadBuffer[nn::capsrv::AlbumFileSizeLimit_ScreenShot];
                    size_t size;
                    NN_ABORT_UNLESS_RESULT_SUCCESS(
                        nn::capsrv::LoadAlbumScreenShotFile(&size, s_LoadBuffer, sizeof(s_LoadBuffer), &albumEntries[idx].fileId));
                }
                nn::os::FenceMemoryAnyAny();
                auto endTick = nn::os::GetSystemTick();
                nn::os::FenceMemoryAnyAny();

                int64_t timeUs = (endTick - startTick).ToTimeSpan().GetMicroSeconds();

                if (timeUs < minTimeUs)
                {
                    minTimeUs = timeUs;
                }
                if (timeUs > maxTimeUs)
                {
                    maxTimeUs = timeUs;
                }
                avgTimeUs += timeUs;
            }

            avgTimeUs /= actualFileCount;

            NN_LOG("[benchmark][load] storage %d, count %d, avg time %lld microsec, min time %lld microsec, max time %lld microsec.\n", static_cast<int>(storage), fileCount, avgTimeUs, minTimeUs, maxTimeUs);
        }
    }
}

TEST(Benchmark, LoadAlbumScreenShotFile)
{
    nn::capsrv::InitializeAlbumAccess();

    NN_LOG("Cleaning up album\n");
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_LOG("benchmark start.\n");
    {
        RandomFileIdGenerator idgen;
        RunBenchmark(nn::capsrv::AlbumStorage_Nand,   10, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Nand,  100, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Nand, 1000, idgen);
    }

    if(nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        RandomFileIdGenerator idgen;
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,    10, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,   100, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  1000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  2000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd,  3000, idgen);
        RunBenchmark(nn::capsrv::AlbumStorage_Sd, 10000, idgen);
    }
    NN_LOG("benckmark complete.\n");

    NN_LOG("Cleaning up album\n");
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
}
