﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nnt.h>
#include "../../Common/testCapsrv_Macro.h"
#include "testCapsrv_StartupTestCase.h"

namespace {

    void TestGetAlbumMountResult(nn::capsrv::AlbumStorageType storage, bool isStorageAvailable) NN_NOEXCEPT
    {
        NN_LOG("Checking storage %d\n", storage);
        NN_LOG("  Force-unmounting\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::ForceAlbumUnmounted(storage));

        NN_LOG("  Checking mount result\n");
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumIsNotMounted, nn::capsrv::GetAlbumMountResult(storage));
        EXPECT_FALSE(nn::capsrv::IsAlbumMounted(storage));

        if(isStorageAvailable)
        {
            NN_LOG("  Notifying storage available\n");
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::NotifyAlbumStorageIsAvailable(storage));

            NN_LOG("  Checking mount result\n");
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumMountResult(storage));
            EXPECT_TRUE(nn::capsrv::IsAlbumMounted(storage));
        }
        else
        {
            NN_LOG("  Notifying storage available\n");
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumIsNotMounted, nn::capsrv::NotifyAlbumStorageIsAvailable(storage));

            NN_LOG("  Checking mount result\n");
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumIsNotMounted, nn::capsrv::GetAlbumMountResult(storage));
            EXPECT_FALSE(nn::capsrv::IsAlbumMounted(storage));
        }

        NN_LOG("  Notifying storage unavailable\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::NotifyAlbumStorageIsUnavailable(storage));

        NN_LOG("  Checking mount result\n");
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumIsNotMounted, nn::capsrv::GetAlbumMountResult(storage));
        EXPECT_FALSE(nn::capsrv::IsAlbumMounted(storage));

        NN_LOG("  Resetting mount status\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::ResetAlbumMountStatus(storage));
    }

}

TEST(AlbumAccessApi, GetAlbumMountResult)
{
    nnt::capsrv::StartupTestCase();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());

    TestGetAlbumMountResult(nn::capsrv::AlbumStorage_Nand, true);
    TestGetAlbumMountResult(nn::capsrv::AlbumStorage_Sd, nn::fs::IsSdCardInserted());

    nn::capsrv::FinalizeAlbumAccess();
    nn::capsrv::FinalizeAlbumControl();
}

// TORIAEZU: CaptureControlApi だけど AlbumAccessApi のテストに入れておく。
TEST(CaptureControlApi, StorageManagement)
{
    nnt::capsrv::StartupTestCase();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());

    // TORIAEZU: 一通り呼べることを確認
    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::NotifyAlbumStorageIsAvailable(nn::capsrv::AlbumStorage_Nand));
    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::NotifyAlbumStorageIsUnavailable(nn::capsrv::AlbumStorage_Nand));

    if(nn::fs::IsSdCardInserted())
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::NotifyAlbumStorageIsAvailable(nn::capsrv::AlbumStorage_Sd));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::NotifyAlbumStorageIsUnavailable(nn::capsrv::AlbumStorage_Sd));
    }

    nn::capsrv::FinalizeAlbumControl();
}

