﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// 以下の API のテストを行います
// - StorageCopyAlbumFile

#include <vector>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nn/fs.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "../../Common/testCapsrv_Macro.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_Config.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_ResultPrivate.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/album/capsrvServer_AlbumPathUtility.h"
#include "testCapsrv_StartupTestCase.h"

namespace {
    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, 1024, 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .jpg, 1024, 12346),
        NNT_CAPSRV_FILEINFO(NA, 2016, 05, 03, 18, 31, 58, 02, 0xBBBBBBBBBBBBBBBB, .jpg, 1024, 12347),
        NNT_CAPSRV_FILEINFO(SD, 2016, 07, 04, 15, 20, 57, 03, 0xCCCCCCCCCCCCCCCC, .jpg, 1024, 12348),
        NNT_CAPSRV_FILEINFO(SD, 2016, 08, 06, 14, 10, 56, 04, 0xDDDDDDDDDDDDDDDD, .jpg, 1024, 12349),
    };
    static const int TotalFileCount = sizeof(FileList) / sizeof(FileList[0]);

    const nn::capsrv::AlbumStorageType StorageList[] = { nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumStorage_Sd };
    static const int StorageCount = sizeof(StorageList) / sizeof(StorageList[0]);

    void CheckStorageCopyAlbumFile(
        int sourceStorage,
        int sourceFileIndex,
        int destStorage
        )
    {
        // 既にあるファイルをクリア
        EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());


        // ファイル作成
        for(int i = 0; i < TotalFileCount; i++)
        {
            FileList[i].Create();
        }

        // ファイルリスト取得
        std::vector<nn::capsrv::AlbumEntry> albumEntriyLists[StorageCount];
        for(int s = 0; s < StorageCount; s++)
        {
            auto& list = albumEntriyLists[s];
            list = nnt::capsrv::GetAlbumEntryList(StorageList[s]);
            nnt::capsrv::SortAlbumEntryListStorageTime(list);
        }

        if(albumEntriyLists[sourceStorage].size() <= static_cast<size_t>(sourceFileIndex))
        {
            NN_LOG("Check storage-copy file skipped: requested index %d, exists %d files.\n", sourceFileIndex, albumEntriyLists[sourceStorage].size());
            return;
        }

        // コピー
        const nn::capsrv::AlbumEntry& srcEntry = albumEntriyLists[sourceStorage][sourceFileIndex];
        NN_LOG("StorageCopy file:\n  source: storage %d, index %d\n  destination: storage %d\n", StorageList[sourceStorage], sourceFileIndex, StorageList[destStorage]);
        nn::Result copyResult = nn::capsrv::StorageCopyAlbumFile(&srcEntry.fileId, StorageList[destStorage]);
        EXPECT_TRUE(copyResult.IsSuccess());
        const nnt::capsrv::FileInfo& srcFileInfo = *std::find_if(FileList, FileList + TotalFileCount, [&srcEntry](const nnt::capsrv::FileInfo& e){ return e.IsMatchTime(srcEntry); });
        EXPECT_NE(FileList + TotalFileCount, &srcFileInfo);

        // コピー後のファイルリスト取得
        std::vector<nn::capsrv::AlbumEntry> albumEntriyLists2[StorageCount];
        for(int s = 0; s < StorageCount; s++)
        {
            auto& list = albumEntriyLists2[s];
            list = nnt::capsrv::GetAlbumEntryList(StorageList[s]);
            nnt::capsrv::SortAlbumEntryListStorageTime(list);
        }

        // チェック
        {
            std::vector<nn::capsrv::AlbumEntry> entryListPre;
            std::vector<nn::capsrv::AlbumEntry> entryListPost;
            for(int s = 0; s < StorageCount; s++)
            {
                entryListPre.insert(entryListPre.end(), albumEntriyLists[s].begin(), albumEntriyLists[s].end());
                entryListPost.insert(entryListPost.end(), albumEntriyLists2[s].begin(), albumEntriyLists2[s].end());
            }

            // 1 つ増えているはず
            EXPECT_EQ(entryListPre.size() + 1, entryListPost.size());

            // 狙ったものが増えていることを確認
            nnt::capsrv::AlbumEntryListDifference difference;
            nnt::capsrv::SubtractAlbumEntryList(&difference, entryListPre, entryListPost);
            EXPECT_EQ(difference.positive.size(), 0);
            EXPECT_EQ(difference.negative.size(), 1);
            if(difference.negative.size() > 0)
            {
                const nn::capsrv::AlbumEntry& dstEntry = difference.negative[0];
                // 時刻を検査
                EXPECT_EQ(nnt::capsrv::CompareTime(srcEntry.fileId.time, dstEntry.fileId.time), 0);
                // ファイルサイズを検査
                EXPECT_EQ(srcEntry.size, dstEntry.size);
                // ファイルの中身を検査
                std::vector<uint8_t> data = nnt::capsrv::DirectAlbumAccessor::LoadFile(
                    nnt::capsrv::DirectAlbumAccessor::GetFilePath(dstEntry.fileId).c_str()
                );
                EXPECT_EQ(dstEntry.size, static_cast<int64_t>(data.size()));
                EXPECT_TRUE(srcFileInfo.CheckData(data.data(), dstEntry.size));
            }
        }
    }

}// anonymous namespace

TEST(AlbumAccessApi, StorageCopyAlbumFile)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    if(!nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Nand) || !nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        NN_LOG("Skipping test case. Both NAND and SD should be mounted\n");
    }

    for(int ss = 0; ss < StorageCount; ss++)
    {
        if(!nn::capsrv::IsAlbumMounted(StorageList[ss]))
        {
            // src storage がマウントされていなければ無視
            continue;
        }
        for(int ds = 0; ds < StorageCount; ds++)
        {
            if(ss == ds)
            {
                // src storage と dst storage が同じなら無視
                continue;
            }
            if(!nn::capsrv::IsAlbumMounted(StorageList[ds]))
            {
                // dst storage がマウントされていなければ無視
                continue;
            }
            for(int i = 0; i < TotalFileCount; i++)
            {
                CheckStorageCopyAlbumFile(ss, i, ds);
            }
        }
    }

    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    SUCCEED();
}

TEST(AlbumAccessApi, StorageCopyAlbumFile_CheckCache)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    if(!nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Nand) || !nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        NN_LOG("Skipping test case. Both NAND and SD should be mounted\n");
        return;
    }

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileListSizeLimit[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, 12345, 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, 56789, 56789),
    };

    // ファイル作成
    int createdFileCount = 0;
    for(int i = 0; i < sizeof(FileListSizeLimit) / sizeof(FileListSizeLimit[0]); i++)
    {
        if(FileListSizeLimit[i].Create())
        {
            createdFileCount++;
        }
    }

    // キャッシュを更新させる
    NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE(storage)
    {
        nn::capsrv::RefreshAlbumCache(storage);
    }

    nn::capsrv::AlbumCacheData origCache;
    EXPECT_TRUE(nn::capsrv::GetAlbumCache(&origCache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot).IsSuccess());

    NN_LOG("Check copy storage repeatation\n");

    for(int i = 0; i < sizeof(FileListSizeLimit) / sizeof(FileListSizeLimit[0]); i++)
    {
        auto& fileInfo = FileListSizeLimit[i];
        auto fileId = fileInfo.GetAlbumFileId();
        const int RepeatCount = 10;
        for(int j = 0; j < RepeatCount; j++)
        {
            NN_LOG("  Copy NAND -> SD\n");
            {
                fileId.storage = nn::capsrv::AlbumStorage_Nand;
                auto result =nn::capsrv::StorageCopyAlbumFile(&fileId, nn::capsrv::AlbumStorage_Sd);
                EXPECT_TRUE(result.IsSuccess());
                if(result.IsFailure())
                {
                    NN_LOG("  result = %08X\n", result.GetInnerValueForDebug());
                }
            }
            NN_LOG("  Copy SD -> NAND\n");
            {
                fileId.storage = nn::capsrv::AlbumStorage_Sd;
                auto result =nn::capsrv::StorageCopyAlbumFile(&fileId, nn::capsrv::AlbumStorage_Nand);
                EXPECT_TRUE(result.IsSuccess());
                if(result.IsFailure())
                {
                    NN_LOG("  result = %08X\n", result.GetInnerValueForDebug());
                }
            }
        }
    }

    nn::capsrv::AlbumCacheData cache0;
    nn::capsrv::AlbumCacheData cache1;
    EXPECT_TRUE(nn::capsrv::GetAlbumCache(&cache0, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot).IsSuccess());
    EXPECT_TRUE(nn::capsrv::GetAlbumCache(&cache1, nn::capsrv::AlbumStorage_Sd, nn::capsrv::AlbumFileContents_ScreenShot).IsSuccess());

    // キャッシュが正しいことを確認
    {
        auto& cache = cache0;
        EXPECT_EQ(origCache.fileCount, cache.fileCount);
        NN_LOG(" cache(NAND) count=%lld\n", cache.fileCount);
    }
    {
        auto& cache = cache1;
        EXPECT_EQ(origCache.fileCount, cache.fileCount);
        NN_LOG(" cache(SD  ) count=%lld\n", cache.fileCount);
    }


    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

}// NOLINT(impl/function_size)


TEST(AlbumAccessApi, StorageCopyAlbumFile_FileSizeLimit)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    if(!nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Nand) || !nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        NN_LOG("Skipping test case. Both NAND and SD should be mounted\n");
        return;
    }

    nn::capsrv::AlbumStorageType dstStorage = nn::capsrv::AlbumStorage_Sd;

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileListSizeLimit[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_ScreenShot    , 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_ScreenShot + 1, 12345),
    };

    // ファイル作成
    int createdFileCount = 0;
    for(int i = 0; i < sizeof(FileListSizeLimit) / sizeof(FileListSizeLimit[0]); i++)
    {
        if(FileListSizeLimit[i].Create())
        {
            createdFileCount++;
        }
    }

    NN_LOG("Check file size limitation for StorageCopyAlbumFile()\n");
    {
        {
            auto& fileInfo = FileListSizeLimit[0];
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId = fileInfo.appId;
            fileId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
            fileId.storage = fileInfo.storage;
            fileId.time = fileInfo.time;

            auto result =nn::capsrv::StorageCopyAlbumFile(&fileId, dstStorage);
            EXPECT_TRUE(result.IsSuccess() || nn::capsrv::ResultAlbumIsNotMounted::Includes(result));
        }
        {
            auto& fileInfo = FileListSizeLimit[1];
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId = fileInfo.appId;
            fileId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
            fileId.storage = fileInfo.storage;
            fileId.time = fileInfo.time;

            auto result =nn::capsrv::StorageCopyAlbumFile(&fileId, dstStorage);
            EXPECT_TRUE(nn::capsrv::ResultAlbumFileNotFound::Includes(result));
        }
    }


    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    SUCCEED();
}// NOLINT(impl/function_size)

// アルバムの使用制限のテスト
TEST(AlbumAccessApi, StorageCopyAlbumFile_AlbumLimitation)
{
    nnt::capsrv::StartupTestCase();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SetInternalConversionEnabled(false));

    if(!nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Nand) || !nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        NN_LOG("Skipping test case. Both NAND and SD should be mounted\n");
        return;
    }

    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    static const int64_t TestFileSize = 100 * 1024;
    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileListAlbumLimit[] = {
        NNT_CAPSRV_FILEINFO(SD, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, TestFileSize, 12345),
        NNT_CAPSRV_FILEINFO(SD, 2016, 01, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, TestFileSize, 12345),
    };

    // SD 側にファイル作成
    for(int i = 0; i < sizeof(FileListAlbumLimit) / sizeof(FileListAlbumLimit[0]); i++)
    {
        NN_ABORT_UNLESS(FileListAlbumLimit[i].Create());
    }

    std::mt19937 engine(0xD2CAA641);
    // ファイル数のテスト
    {
        NN_LOG("Creating empty files...\n");
        {
            int i;
            for(i = 0; i < nn::capsrv::AlbumFileCountLimit_NandScreenShot - 1; i++)
            {
                nn::capsrv::AlbumFileId fileId = {};
                fileId.applicationId.value = 1;   // 0 以外
                fileId.storage = nn::capsrv::AlbumStorage_Nand;
                // NOTE: 年月日をランダムにするとサブディレクトリ掘るのに時間がかかる
                fileId.time.year  = 2017;
                fileId.time.month = 8;
                fileId.time.day   = 5;
                fileId.time.hour   = engine() % 24;
                fileId.time.minute = engine() % 60;
                fileId.time.second = engine() % 60;
                fileId.applicationId.value = i + 1; // 0 以外にする
                auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
                std::vector<uint8_t> data;
                nnt::capsrv::DirectAlbumAccessor::SaveFile(data, path.c_str());
                if((i + 1) % 100 == 0)
                {
                    NN_LOG("  Created %d files\n", i + 1);
                }
            }
            NN_LOG("  Created %d files ... complete\n", i + 1);
        }
        // 不正なパスのファイルを作成
        {
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId.value = 1;   // 0 以外
            fileId.storage = nn::capsrv::AlbumStorage_Nand;
            fileId.time.year  = 2222;
            fileId.time.month = 6;
            fileId.time.day   = 20;
            auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
            path[7] = '3'; // 年を書き換え 2222->2232
            std::vector<uint8_t> data;
            nnt::capsrv::DirectAlbumAccessor::CreateFile(nn::capsrv::AlbumFileSizeLimit_ScreenShot, path.c_str());
            NN_LOG("  Created file at invalid filepath %s\n", path.c_str());
        }

        // キャッシュを更新させる
        NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE(storage)
        {
            nn::capsrv::RefreshAlbumCache(storage);
        }

        nn::capsrv::AlbumCacheData expectedCacheData = {};
        // キャッシュを取得
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::capsrv::GetAlbumCache(&expectedCacheData, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot)
            );
        }

        // 1 枚はコピーできる
        {
            auto& fileInfo = FileListAlbumLimit[0];
            auto id = fileInfo.GetAlbumFileId();
            auto result = nn::capsrv::StorageCopyAlbumFile(&id, nn::capsrv::AlbumStorage_Nand);
            EXPECT_TRUE(result.IsSuccess());
            // キャッシュ確認
            expectedCacheData.fileCount++;
            nn::capsrv::AlbumCacheData cacheData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot));
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
        }
        // もう 1 枚はコピーできない
        {
            auto& fileInfo = FileListAlbumLimit[1];
            auto id = fileInfo.GetAlbumFileId();
            auto result = nn::capsrv::StorageCopyAlbumFile(&id, nn::capsrv::AlbumStorage_Nand);
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalAlbumLimitationFileCountLimit, result);
            // キャッシュ確認
            nn::capsrv::AlbumCacheData cacheData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot));
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
        }
    }

    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 合計ファイルサイズのテスト
    // NOTE: Scoop 管理外のファイルは制限に考慮されない。
#if 0
    {
        auto& srcFileInfo = FileListAlbumLimit[0];
        srcFileInfo.Create();

        nn::capsrv::AlbumCacheData expectedCacheData = {};
        NN_LOG("Creating large files...\n");
        // 1 枚コピーできる容量を残して大きいファイルを作る
        {
            int64_t unitSize = nn::capsrv::AlbumFileSizeLimit_ScreenShot;
            int64_t size = nn::capsrv::server::TotalFileSizeLimitNand - TestFileSize;
            int64_t createdSize = 0;
            int i = 0;
            while(size > 0)
            {
                nn::capsrv::AlbumFileId fileId = {};
                fileId.storage = nn::capsrv::AlbumStorage_Nand;
                fileId.time.year  = 2010 + (engine() % 10);
                fileId.time.month = 1 + (engine() % 12);
                fileId.time.day   = 1 + (engine() % 31);
                fileId.applicationId.value = i + 1; // 0 以外にする
                auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
                int64_t fileSize = std::min(unitSize, size);
                nnt::capsrv::DirectAlbumAccessor::CreateFile(fileSize, path.c_str());
                size -= fileSize;
                createdSize += fileSize;
                i++;
                NN_LOG("  Created %lld bytes\n", createdSize);
            }
            NN_LOG("  Created %lld bytes ... complete\n", createdSize);
            expectedCacheData.totalFileSize += createdSize;
            expectedCacheData.fileCount += static_cast<int64_t>(i);
        }

        // 不正なパスのファイルを作成
        {
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId.value = 1;   // 0 以外
            fileId.storage = nn::capsrv::AlbumStorage_Nand;
            fileId.time.year  = 2222;
            fileId.time.month = 6;
            fileId.time.day   = 20;
            auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
            path[7] = '3'; // 年を書き換え 2222->2223
            std::vector<uint8_t> data;
            nnt::capsrv::DirectAlbumAccessor::CreateFile(nn::capsrv::AlbumFileSizeLimit_ScreenShot, path.c_str());
            NN_LOG("  Created file at invalid filepath %s\n", path.c_str());
        }

        NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE(storage)
        {
            nn::capsrv::RefreshAlbumCache(storage);
        }

        // キャッシュを確認
        {
            nn::capsrv::AlbumCacheData cacheData = {};
            EXPECT_TRUE(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand).IsSuccess());
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
            EXPECT_EQ(expectedCacheData.totalFileSize, cacheData.totalFileSize);
        }

        // コピーできることを確認
        {
            auto& fileInfo = srcFileInfo;
            auto id = fileInfo.GetAlbumFileId();
            auto result = nn::capsrv::StorageCopyAlbumFile(&id, nn::capsrv::AlbumStorage_Nand);
            EXPECT_TRUE(result.IsSuccess());
            // キャッシュ確認
            expectedCacheData.fileCount++;
            expectedCacheData.totalFileSize += fileInfo.filesize;
            nn::capsrv::AlbumCacheData cacheData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand));
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
            EXPECT_EQ(expectedCacheData.totalFileSize, cacheData.totalFileSize);
        }

        // コピーできないようにファイルを調整
        {
            // コピーしたファイルを消す
            auto id = srcFileInfo.GetAlbumFileId();
            id.storage = nn::capsrv::AlbumStorage_Nand;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::DeleteAlbumFile(&id));
            // 1 バイト作る
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId.value = 1;   // 0 以外
            fileId.storage = nn::capsrv::AlbumStorage_Nand;
            fileId.time.year  = 3016;
            fileId.time.month = 1;
            fileId.time.day   = 1;
            auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
            nnt::capsrv::DirectAlbumAccessor::CreateFile(1, path.c_str());
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(nn::capsrv::AlbumStorage_Nand));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&expectedCacheData, nn::capsrv::AlbumStorage_Nand));

        // コピーできないことを確認
        {
            auto& fileInfo = srcFileInfo;
            auto id = fileInfo.GetAlbumFileId();
            auto result = nn::capsrv::StorageCopyAlbumFile(&id, nn::capsrv::AlbumStorage_Nand);
            EXPECT_TRUE(nn::capsrv::server::ResultInternalAlbumLimitationTotalFileSizeLimit::Includes(result));
            // キャッシュ確認
            nn::capsrv::AlbumCacheData cacheData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand));
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
            EXPECT_EQ(expectedCacheData.totalFileSize, cacheData.totalFileSize);
        }

        // キャッシュが変わっていないことを確認
        {
            nn::capsrv::AlbumCacheData cacheData = {};
            EXPECT_TRUE(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand).IsSuccess());
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
            EXPECT_EQ(expectedCacheData.totalFileSize, cacheData.totalFileSize);
        }
    }
#endif


    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    // キャッシュを更新させる
    NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE(storage)
    {
        nn::capsrv::RefreshAlbumCache(storage);
    }

    SUCCEED();
}// NOLINT(impl/function_size)

// アルバムの使用制限のテスト
TEST(AlbumAccessApi, GetRequiredStorageSpaceToCopyAll)
{
    nnt::capsrv::StartupTestCase();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SetInternalConversionEnabled(false));

    if(!nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Nand) || !nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        NN_LOG("Skipping test case. Both NAND and SD should be mounted\n");
        return;
    }

    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    static const int64_t TestFileSize = 100 * 1024;
    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileListStorageCopy[] = {
        NNT_CAPSRV_FILEINFO(SD, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, TestFileSize, 12345),
        NNT_CAPSRV_FILEINFO(SD, 2016, 01, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, TestFileSize, 12345),
        NNT_CAPSRV_FILEINFO(SD, 2016, 01, 04, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, TestFileSize, 12345),
        NNT_CAPSRV_FILEINFO(SD, 2016, 02, 04, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, TestFileSize, 12345),
        NNT_CAPSRV_FILEINFO(SD, 2017, 02, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, TestFileSize, 12345),
        NNT_CAPSRV_FILEINFO(SD, 2018, 01, 03, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, TestFileSize, 12345),
    };

    size_t requiredSizeToNand = 0;
    size_t requiredSizeToSd = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetRequiredStorageFreeSpaceSizeToCopyAll(&requiredSizeToNand, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumStorage_Sd));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetRequiredStorageFreeSpaceSizeToCopyAll(&requiredSizeToSd, nn::capsrv::AlbumStorage_Sd, nn::capsrv::AlbumStorage_Nand));

    EXPECT_EQ(requiredSizeToNand, requiredSizeToSd);

    // SD 側にファイル作成
    for(int i = 0; i < sizeof(FileListStorageCopy) / sizeof(FileListStorageCopy[0]); i++)
    {
        NN_ABORT_UNLESS(FileListStorageCopy[i].Create());
        nn::capsrv::RefreshAlbumCache(nn::capsrv::AlbumStorage_Sd);

        size_t currentRequiredSizeToNand = 0;
        size_t currentRequiredSizeToSd = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetRequiredStorageFreeSpaceSizeToCopyAll(&currentRequiredSizeToNand, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumStorage_Sd));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetRequiredStorageFreeSpaceSizeToCopyAll(&currentRequiredSizeToSd, nn::capsrv::AlbumStorage_Sd, nn::capsrv::AlbumStorage_Nand));

        EXPECT_GT(currentRequiredSizeToNand, requiredSizeToNand);
        EXPECT_EQ(currentRequiredSizeToSd, requiredSizeToSd);
        requiredSizeToNand = currentRequiredSizeToNand;
        requiredSizeToSd   = currentRequiredSizeToSd;
    }

    // NAND にコピー
    for(int i = 0; i < sizeof(FileListStorageCopy) / sizeof(FileListStorageCopy[0]); i++)
    {
        auto srcFileId = FileListStorageCopy[i].GetAlbumFileId();
        nn::capsrv::AlbumStorageType dstStorage = nn::capsrv::AlbumStorage_Nand;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::StorageCopyAlbumFile(&srcFileId, dstStorage));

        size_t currentRequiredSizeToNand = 0;
        size_t currentRequiredSizeToSd = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetRequiredStorageFreeSpaceSizeToCopyAll(&currentRequiredSizeToNand, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumStorage_Sd));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetRequiredStorageFreeSpaceSizeToCopyAll(&currentRequiredSizeToSd, nn::capsrv::AlbumStorage_Sd, nn::capsrv::AlbumStorage_Nand));

        EXPECT_LT(currentRequiredSizeToNand, requiredSizeToNand);
        EXPECT_EQ(currentRequiredSizeToSd, requiredSizeToSd);
        requiredSizeToNand = currentRequiredSizeToNand;
        requiredSizeToSd   = currentRequiredSizeToSd;
    }

    EXPECT_EQ(requiredSizeToNand, requiredSizeToSd);

    SUCCEED();
}// NOLINT(impl/function_size)
