﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/image/image_ExifExtractor.h>
#include <nn/settings/system/settings_Capture.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>

#include <nnt.h>

#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_TestFileUtility.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_Config.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_ResultPrivate.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/album/capsrvServer_AlbumPathUtility.h"
#include "testCapsrv_StartupTestCase.h"

TEST(AlbumAccessApi, SaveScreenShotOfMovie_Orientation)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    nn::capsrv::ScreenShotOrientation orientationList[] = {
        nn::capsrv::ScreenShotOrientation_Default,
        nn::capsrv::ScreenShotOrientation_Rotate90,
        nn::capsrv::ScreenShotOrientation_Rotate180,
        nn::capsrv::ScreenShotOrientation_Rotate270,
    };
    int orientationCount = sizeof(orientationList) / sizeof(orientationList[0]);

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    for(int i = 0; i < orientationCount; i++)
    {
        auto ori = orientationList[i];


        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            attribute.size = size;
            attribute.orientation = ori;
            nn::capsrv::AppletData appletData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveScreenShotOfMovie(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                attribute,
                appletData,
                &origId
            ));
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData ={};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(size, attribute.size);
            EXPECT_EQ(ori, attribute.orientation);

            attribute = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(size, attribute.size);
            EXPECT_EQ(ori, attribute.orientation);
        }
    }
}

TEST(AlbumAccessApi, SaveScreenShotOfMovie_Copyright)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    for(int i = 0; i < 4; i++)
    {
        bool isCopyrightImageComposited = (i >= 2);
        bool isUneditableAreaSet        = (i == 1 || i == 3);

        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            attribute.size = size;
            attribute.flags |= (isCopyrightImageComposited ? nn::capsrv::detail::ScreenShotAttributeFlag_IsCopyrightImageComposited : 0);
            attribute.flags |= (isUneditableAreaSet ? nn::capsrv::detail::ScreenShotAttributeFlag_HasUneditableArea : 0);
            attribute.uneditableAreaCoordinateX  = isUneditableAreaSet ? 320 : 0;
            attribute.uneditableAreaCoordinateY  = isUneditableAreaSet ? 180 : 0;
            attribute.uneditableAreaWidth        = isUneditableAreaSet ? 640 : 0;
            attribute.uneditableAreaHeight       = isUneditableAreaSet ? 360 : 0;
            nn::capsrv::AppletData appletData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveScreenShotOfMovie(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                attribute,
                appletData,
                &origId
            ));
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData ={};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(size, attribute.size);
            EXPECT_EQ(isCopyrightImageComposited, attribute.IsCopyrightImageComposited());
            EXPECT_EQ(isUneditableAreaSet, attribute.HasUneditableArea());
            if (isUneditableAreaSet)
            {
                EXPECT_EQ(320, attribute.uneditableAreaCoordinateX);
                EXPECT_EQ(180, attribute.uneditableAreaCoordinateY);
                EXPECT_EQ(640, attribute.uneditableAreaWidth);
                EXPECT_EQ(360, attribute.uneditableAreaHeight);
            }

            attribute = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(size, attribute.size);
            EXPECT_EQ(isCopyrightImageComposited, attribute.IsCopyrightImageComposited());
            EXPECT_EQ(isUneditableAreaSet, attribute.HasUneditableArea());
            if (isUneditableAreaSet)
            {
                EXPECT_EQ(320, attribute.uneditableAreaCoordinateX);
                EXPECT_EQ(180, attribute.uneditableAreaCoordinateY);
                EXPECT_EQ(640, attribute.uneditableAreaWidth);
                EXPECT_EQ(360, attribute.uneditableAreaHeight);
            }
        }
    }
}

TEST(AlbumAccessApi, SaveScreenShotOfMovie_FileContents)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    nn::capsrv::AlbumFileContentsType contentsList[] = {
        nn::capsrv::AlbumFileContents_ScreenShot,
        nn::capsrv::AlbumFileContents_Movie,
    };
    int contentsCount = sizeof(contentsList) / sizeof(contentsList[0]);

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    for(int i = 0; i < contentsCount; i++)
    {
        auto cont = contentsList[i];
        origId.contents = cont;

        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            attribute.size = size;
            nn::capsrv::AppletData appletData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveScreenShotOfMovie(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                attribute,
                appletData,
                &origId
            ));
            // 保存されたものは常に ScreenShot になる。
            EXPECT_EQ(entry.fileId.contents, nn::capsrv::AlbumFileContents_ScreenShot);
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData ={};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(size, attribute.size);

            attribute = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(size, attribute.size);
        }
    }
}

TEST(AlbumAccessApi, SaveScreenShotOfMovie_AppletData)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    int trialCount = 8;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    nn::capsrv::ScreenShotOrientation orientationList[] = {
        nn::capsrv::ScreenShotOrientation_Default,
        nn::capsrv::ScreenShotOrientation_Rotate90,
        nn::capsrv::ScreenShotOrientation_Rotate180,
        nn::capsrv::ScreenShotOrientation_Rotate270,
    };
    int orientationCount = sizeof(orientationList) / sizeof(orientationList[0]);

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::uniform_int_distribution<int> dist;
    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    for(int i = 0; i < trialCount; i++)
    {
        nn::capsrv::AppletData srcAppletData = {};
        for(size_t pos = 0; pos < sizeof(srcAppletData); pos++)
        {
            reinterpret_cast<uint8_t&>(srcAppletData.value[pos]) = static_cast<uint8_t>(dist(rand) % 256);
        }

        nn::capsrv::ScreenShotAttribute srcAttribute;
        srcAttribute.SetDefault();
        srcAttribute.size = size;
        srcAttribute.orientation = orientationList[dist(rand) % orientationCount];

        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveScreenShotOfMovie(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                srcAttribute,
                srcAppletData,
                &origId
            ));
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attributeThumb = {};
            nn::capsrv::AppletData appletDataThumb = {};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attributeThumb, &appletDataThumb, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(nn::capsrv::ScreenShotSize_1280x720, attributeThumb.size); // srcAttribute とは無関係に決まる
            EXPECT_EQ(srcAttribute.orientation, attributeThumb.orientation);     // srcAttribute の値を引き継ぐ
            EXPECT_EQ(nn::capsrv::AlbumFileDescription_ScreenShotEdited, attributeThumb.description);
            EXPECT_EQ(1, attributeThumb.frameCount);
            EXPECT_EQ(0, attributeThumb.frameRateNumerator);
            EXPECT_EQ(0, attributeThumb.frameRateDenominator);
            EXPECT_EQ(0, attributeThumb.dataDurationMilliseconds);
            EXPECT_EQ(0, attributeThumb.keyFrameInterval);
            EXPECT_EQ(0, std::memcmp(&srcAppletData, &appletDataThumb, sizeof(nn::capsrv::AppletData)));

            nn::capsrv::ScreenShotAttribute attributeFull = {};
            nn::capsrv::AppletData appletDataFull = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attributeFull, &appletDataFull, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(0, std::memcmp(&attributeThumb, &attributeFull, sizeof(nn::capsrv::ScreenShotAttribute)));
            EXPECT_EQ(0, std::memcmp(&appletDataThumb, &appletDataFull, sizeof(nn::capsrv::AppletData)));
        }
    }
}
