﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/image/image_ExifExtractor.h>
#include <nn/settings/system/settings_Capture.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>

#include <nnt.h>

#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_TestFileUtility.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_Config.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_ResultPrivate.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/album/capsrvServer_AlbumPathUtility.h"
#include "testCapsrv_StartupTestCase.h"

TEST(AlbumAccessApi, SaveEditedScreenShot)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    const nn::settings::system::PrimaryAlbumStorage PrimaryAlbumStorageList[] = {
        nn::settings::system::PrimaryAlbumStorage_Nand,
        nn::settings::system::PrimaryAlbumStorage_SdCard,
    };

    NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE_I(idx,primaryStorage)
    {
        // 保存先を変更
        NN_LOG("Set primary storage %d\n", idx);
        nn::settings::system::SetPrimaryAlbumStorage(PrimaryAlbumStorageList[idx]);

        const int width = 1280;
        const int height = 720;
        const int thumbWidth  = nn::capsrv::ViewerThumbnailImageSize_Width;
        const int thumbHeight = nn::capsrv::ViewerThumbnailImageSize_Height;
        std::vector<uint8_t> srcData;
        srcData.resize(4 * width * height);
        std::vector<uint8_t> srcThumb;
        srcThumb.resize(4 * thumbWidth * thumbHeight);

        // 書き込むデータを生成
        {
            uint8_t* p = srcData.data();
            for(int y = 0; y < height; y++)
            {
                for(int x = 0; x < width; x++)
                {
                    *(p++) = (x > width / 2) ? 255 : 0;
                    *(p++) = (y > height / 2) ? 255 : 0;
                    *(p++) = ((x + y) < (width + height) / 2) ? 255 : 0;
                    *(p++) = 255;
                }
            }
            uint8_t* q = srcThumb.data();
            for(int y = 0; y < thumbHeight; y++)
            {
                for(int x = 0; x < thumbWidth; x++)
                {
                    *(q++) = (x > thumbWidth / 2) ? 255 : 0;
                    *(q++) = (y > thumbHeight / 2) ? 255 : 0;
                    *(q++) = ((x + y) < (thumbWidth + thumbHeight) / 2) ? 255 : 0;
                    *(q++) = 255;
                }
            }
        }

        // 適当に編集元の ID をでっちあげる
        nn::capsrv::AlbumFileId origId = {};
        origId.applicationId.value = 0x1234567890123456;
        origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
        origId.storage = nn::capsrv::AlbumStorage_Sd;
        origId.time.year = 2016;
        origId.time.month = 4;
        origId.time.day = 11;
        origId.time.hour = 18;
        origId.time.minute = 49;
        origId.time.second = 20;
        origId.time.id = 0;

        nn::capsrv::AlbumEntry newEntry = {};

        nn::capsrv::ScreenShotAttribute attribute;
        attribute.SetDefault();
        nn::capsrv::AppletData appletData = {};
        NNT_EXPECT_RESULT_SUCCESS(
            nn::capsrv::SaveEditedScreenShot(
                &newEntry,
                srcData.data(),
                srcData.size(),
                width,
                height,
                srcThumb.data(),
                srcThumb.size(),
                thumbWidth,
                thumbHeight,
                attribute,
                appletData,
                &origId
            )
        );
        EXPECT_EQ(primaryStorage, newEntry.fileId.storage);

        std::vector<char> writtenData;
        writtenData.resize(500 * 1024);

        size_t fileSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::capsrv::LoadAlbumScreenShotFile(&fileSize, writtenData.data(), writtenData.size(), &newEntry.fileId)
        );
        EXPECT_EQ(newEntry.size, static_cast<int64_t>(fileSize));
        writtenData.resize(fileSize);

        std::vector<char> work;
        std::vector<uint8_t> decodedData;
        decodedData.resize(4 * width * height);

        // 保存した画像を展開
        {
            nn::image::JpegDecoder decoder;
            decoder.SetImageData(writtenData.data(), writtenData.size());

            auto analyzeResult = decoder.Analyze();
            EXPECT_EQ(nn::image::JpegStatus_Ok, analyzeResult);

            auto dim = decoder.GetAnalyzedDimension();
            EXPECT_EQ(width, static_cast<int>(dim.width));
            EXPECT_EQ(height, static_cast<int>(dim.height));

            work.resize(decoder.GetAnalyzedWorkBufferSize());

            auto decodeResult = decoder.Decode(decodedData.data(), decodedData.size(), 4, work.data(), work.size());
            EXPECT_EQ(nn::image::JpegStatus_Ok, decodeResult);
        }

        // 元の画像とほぼ一致するかを検査
        {
            uint8_t* p = srcData.data();
            uint8_t* q = decodedData.data();
            int diffSum = 0;
            for(int i = 0; i < 4 * width * height; i++)
            {
                int diff = static_cast<int>(*(p++)) - static_cast<int>(*(q++));
                diffSum += std::abs(diff);
            }
            double d = static_cast<double>(diffSum) / static_cast<double>(4 * width * height);
            NN_LOG("diffSum = %d, d = %lf\n", diffSum, d);
            EXPECT_LE(d, 0.25); // やってみたら 0.247... だった。
        }

        // サムネイルを展開
        std::vector<uint8_t> writtenThumb;
        std::vector<uint8_t> decodedThumb;
        decodedThumb.resize(4 * thumbWidth * thumbHeight);
        {
            {
                const void* pExif = nullptr;
                size_t exifSize = 0;
                EXPECT_EQ(nn::image::JpegStatus_Ok, nn::image::JpegDecoder::GetExifData(&pExif, &exifSize, writtenData.data(), writtenData.size()));

                work.resize(nn::image::ExifExtractor::GetWorkBufferSize());
                nn::image::ExifExtractor extractor(work.data(), work.size());
                extractor.SetExifData(pExif, exifSize);
                EXPECT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());

                size_t thumbSize = 0;
                const uint8_t* pThumb = reinterpret_cast<const uint8_t*>(extractor.ExtractThumbnail(&thumbSize));

                writtenThumb.assign(pThumb, pThumb + thumbSize);
            }
            {
                nn::image::JpegDecoder decoder;
                decoder.SetImageData(writtenThumb.data(), writtenThumb.size());

                auto analyzeResult = decoder.Analyze();
                EXPECT_EQ(nn::image::JpegStatus_Ok, analyzeResult);

                auto dim = decoder.GetAnalyzedDimension();
                EXPECT_EQ(thumbWidth, static_cast<int>(dim.width));
                EXPECT_EQ(thumbHeight, static_cast<int>(dim.height));

                work.resize(decoder.GetAnalyzedWorkBufferSize());

                auto decodeResult = decoder.Decode(decodedThumb.data(), decodedThumb.size(), 4, work.data(), work.size());
                EXPECT_EQ(nn::image::JpegStatus_Ok, decodeResult);
            }
        }

        // 元のサムネイル画像とほぼ一致するかを検査
        {
            uint8_t* p = srcThumb.data();
            uint8_t* q = decodedThumb.data();
            int diffSum = 0;
            for(int i = 0; i < 4 * thumbWidth * thumbHeight; i++)
            {
                int diff = static_cast<int>(*(p++)) - static_cast<int>(*(q++));
                diffSum += std::abs(diff);
            }
            double d = static_cast<double>(diffSum) / static_cast<double>(4 * thumbWidth * thumbHeight);
            NN_LOG("diffSum = %d, d = %lf\n", diffSum, d);
            EXPECT_LE(d, 0.6); // やってみたら 0.581... だった。
        }

        // サムネイルを取得
        std::vector<uint8_t> extractedThumb;
        {
            extractedThumb.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            size_t thumbSize = 0;
            EXPECT_TRUE(
                nn::capsrv::LoadAlbumFileThumbnail(&thumbSize, extractedThumb.data(), extractedThumb.size(), &newEntry.fileId).IsSuccess()
            );
            extractedThumb.resize(thumbSize);
            EXPECT_EQ(writtenThumb.size(), thumbSize);
            EXPECT_EQ(writtenThumb, extractedThumb);
        }

    }

    nn::capsrv::FinalizeAlbumAccess();
} // NOLINT(impl/function_size)

TEST(AlbumAccessApi, SaveEditedScreenShot_FileSizeLimitCheck)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    const nn::settings::system::PrimaryAlbumStorage PrimaryAlbumStorageList[] = {
        nn::settings::system::PrimaryAlbumStorage_Nand,
        nn::settings::system::PrimaryAlbumStorage_SdCard,
    };

    NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE_I(idx,primaryStorage)
    {
        // 保存先を変更
        NN_LOG("Set primary storage %d\n", idx);
        nn::settings::system::SetPrimaryAlbumStorage(PrimaryAlbumStorageList[idx]);

        const int width = 1280;
        const int height = 720;
        const int thumbWidth  = nn::capsrv::ViewerThumbnailImageSize_Width;
        const int thumbHeight = nn::capsrv::ViewerThumbnailImageSize_Height;
        std::vector<uint8_t> srcData;
        srcData.resize(4 * width * height);
        std::vector<uint8_t> srcThumb;
        srcThumb.resize(4 * thumbWidth * thumbHeight);

        // 書き込むデータを生成
        {
            std::mt19937 rand;
            rand.seed(0x987654AB);
            std::uniform_int_distribution<int> dist;

            uint8_t* p = srcData.data();
            for(int y = 0; y < height; y++)
            {
                // 適当に 500 ～ 512 KB の JPEG ファイルになるように調整
                for(int x = 0; x < width; x++)
                {
                    *(p++) = static_cast<uint8_t>(dist(rand));
                    *(p++) = (static_cast<uint8_t>(dist(rand)) / 4) * 4;
                    *(p++) = (static_cast<uint8_t>(dist(rand)) / 128) * 128;
                    *(p++) = 255;
                }
            }
            uint8_t* q = srcThumb.data();
            for(int y = 0; y < thumbHeight; y++)
            {
                for(int x = 0; x < thumbWidth; x++)
                {
                    *(q++) = static_cast<uint8_t>(dist(rand));
                    *(q++) = static_cast<uint8_t>(dist(rand));
                    *(q++) = static_cast<uint8_t>(dist(rand));
                    *(q++) = 255;
                }
            }
        }

        // 適当に編集元の ID をでっちあげる
        nn::capsrv::AlbumFileId origId = {};
        origId.applicationId.value = 0x1234567890123456;
        origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
        origId.storage = nn::capsrv::AlbumStorage_Sd;
        origId.time.year = 2016;
        origId.time.month = 4;
        origId.time.day = 11;
        origId.time.hour = 18;
        origId.time.minute = 49;
        origId.time.second = 20;
        origId.time.id = 0;

        nn::capsrv::AlbumEntry newEntry = {};

        nn::capsrv::ScreenShotAttribute attribute;
        attribute.SetDefault();
        nn::capsrv::AppletData appletData = {};
        auto result = nn::capsrv::SaveEditedScreenShot(
                &newEntry,
                srcData.data(),
                srcData.size(),
                width,
                height,
                srcThumb.data(),
                srcThumb.size(),
                thumbWidth,
                thumbHeight,
                attribute,
                appletData,
                &origId
            );

        if(result.IsSuccess())
        {
            EXPECT_LE(newEntry.size, nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("ComplexImage -> %lld bytes\n", static_cast<int64_t>(newEntry.size));
        }
        else
        {
            NN_LOG("ComplexImage encode error (%d-%d)\n", result.GetModule(), result.GetDescription());
            EXPECT_TRUE(nn::capsrv::ResultScreenShotEncodingFailed::Includes(result));
        }
    }

    nn::capsrv::FinalizeAlbumAccess();
} // NOLINT(impl/function_size)

TEST(AlbumAccessApi, SaveEditedScreenShot_Orientation)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    nn::capsrv::ScreenShotOrientation orientationList[] = {
        nn::capsrv::ScreenShotOrientation_Default,
        nn::capsrv::ScreenShotOrientation_Rotate90,
        nn::capsrv::ScreenShotOrientation_Rotate180,
        nn::capsrv::ScreenShotOrientation_Rotate270,
    };
    int orientationCount = sizeof(orientationList) / sizeof(orientationList[0]);

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    for(int i = 0; i < orientationCount; i++)
    {
        auto ori = orientationList[i];


        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            attribute.size = size;
            attribute.orientation = ori;
            nn::capsrv::AppletData appletData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveEditedScreenShot(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                attribute,
                appletData,
                &origId
            ));
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData ={};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(size, attribute.size);
            EXPECT_EQ(ori, attribute.orientation);

            attribute = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(size, attribute.size);
            EXPECT_EQ(ori, attribute.orientation);
        }
    }
}

TEST(AlbumAccessApi, SaveEditedScreenShot_Copyright)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    for(int i = 0; i < 4; i++)
    {
        bool isCopyrightImageComposited = (i >= 2);
        bool isUneditableAreaSet        = (i == 1 || i == 3);

        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            attribute.size = size;
            attribute.flags |= (isCopyrightImageComposited ? nn::capsrv::detail::ScreenShotAttributeFlag_IsCopyrightImageComposited : 0);
            attribute.flags |= (isUneditableAreaSet ? nn::capsrv::detail::ScreenShotAttributeFlag_HasUneditableArea : 0);
            attribute.uneditableAreaCoordinateX  = isUneditableAreaSet ? 320 : 0;
            attribute.uneditableAreaCoordinateY  = isUneditableAreaSet ? 180 : 0;
            attribute.uneditableAreaWidth        = isUneditableAreaSet ? 640 : 0;
            attribute.uneditableAreaHeight       = isUneditableAreaSet ? 360 : 0;
            nn::capsrv::AppletData appletData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveEditedScreenShot(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                attribute,
                appletData,
                &origId
            ));
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData ={};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(size, attribute.size);
            EXPECT_EQ(isCopyrightImageComposited, attribute.IsCopyrightImageComposited());
            EXPECT_EQ(isUneditableAreaSet, attribute.HasUneditableArea());
            if (isUneditableAreaSet)
            {
                EXPECT_EQ(320, attribute.uneditableAreaCoordinateX);
                EXPECT_EQ(180, attribute.uneditableAreaCoordinateY);
                EXPECT_EQ(640, attribute.uneditableAreaWidth);
                EXPECT_EQ(360, attribute.uneditableAreaHeight);
            }

            attribute = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(size, attribute.size);
            EXPECT_EQ(isCopyrightImageComposited, attribute.IsCopyrightImageComposited());
            EXPECT_EQ(isUneditableAreaSet, attribute.HasUneditableArea());
            if (isUneditableAreaSet)
            {
                EXPECT_EQ(320, attribute.uneditableAreaCoordinateX);
                EXPECT_EQ(180, attribute.uneditableAreaCoordinateY);
                EXPECT_EQ(640, attribute.uneditableAreaWidth);
                EXPECT_EQ(360, attribute.uneditableAreaHeight);
            }
        }
    }
}

TEST(AlbumAccessApi, SaveEditedScreenShot_FileContents)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    nn::capsrv::AlbumFileContentsType contentsList[] = {
        nn::capsrv::AlbumFileContents_ScreenShot,
        nn::capsrv::AlbumFileContents_Movie,
    };
    int contentsCount = sizeof(contentsList) / sizeof(contentsList[0]);

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    for(int i = 0; i < contentsCount; i++)
    {
        auto cont = contentsList[i];
        origId.contents = cont;

        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            attribute.size = size;
            nn::capsrv::AppletData appletData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveEditedScreenShot(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                attribute,
                appletData,
                &origId
            ));
            // 保存されたものは常に ScreenShot になる。
            EXPECT_EQ(entry.fileId.contents, nn::capsrv::AlbumFileContents_ScreenShot);
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData ={};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(size, attribute.size);

            attribute = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attribute, &appletData, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(size, attribute.size);
        }
    }
}

TEST(AlbumAccessApi, SaveEditedScreenShot_MovieAttributeForScreenShot)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::uniform_int_distribution<int> dist;
    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    {
        nn::capsrv::AppletData srcAppletData = {};
        for(size_t pos = 0; pos < sizeof(srcAppletData); pos++)
        {
            reinterpret_cast<uint8_t&>(srcAppletData.value[pos]) = static_cast<uint8_t>(dist(rand) % 256);
        }

        nn::capsrv::ScreenShotAttribute srcAttribute;
        srcAttribute.SetDefault();
        srcAttribute.size = size;

        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveEditedScreenShot(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                srcAttribute,
                srcAppletData,
                &origId
            ));
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attributeThumb = {};
            nn::capsrv::AppletData appletDataThumb = {};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attributeThumb, &appletDataThumb, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(nn::capsrv::ScreenShotSize_1280x720, attributeThumb.size); // srcAttribute とは無関係に決まる
            EXPECT_EQ(srcAttribute.orientation, attributeThumb.orientation);     // srcAttribute の値を引き継ぐ
            EXPECT_EQ(nn::capsrv::AlbumFileDescription_ScreenShotEdited, attributeThumb.description);
            EXPECT_EQ(1, attributeThumb.frameCount);
            EXPECT_EQ(0, attributeThumb.frameRateNumerator);
            EXPECT_EQ(0, attributeThumb.frameRateDenominator);
            EXPECT_EQ(0, attributeThumb.dataDurationMilliseconds);
            EXPECT_EQ(0, attributeThumb.keyFrameInterval);
            EXPECT_EQ(0, std::memcmp(&srcAppletData, &appletDataThumb, sizeof(nn::capsrv::AppletData)));
            EXPECT_EQ(nn::capsrv::ScreenShotSize_1280x720, attributeThumb.movieSize);

            nn::capsrv::ScreenShotAttribute attributeFull = {};
            nn::capsrv::AppletData appletDataFull = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attributeFull, &appletDataFull, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(0, std::memcmp(&attributeThumb, &attributeFull, sizeof(nn::capsrv::ScreenShotAttribute)));
            EXPECT_EQ(0, std::memcmp(&appletDataThumb, &appletDataFull, sizeof(nn::capsrv::AppletData)));
        }
    }
}


TEST(AlbumAccessApi, SaveEditedScreenShot_AppletData)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    auto size = nn::capsrv::ScreenShotSize_1280x720;
    int width = 1280;
    int height = 720;

    int trialCount = 8;

    // 適当に編集元の ID をでっちあげる
    nn::capsrv::AlbumFileId origId = {};
    origId.applicationId.value = 0x1234567890123456;
    origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
    origId.storage = nn::capsrv::AlbumStorage_Sd;
    origId.time.year = 2016;
    origId.time.month = 4;
    origId.time.day = 11;
    origId.time.hour = 18;
    origId.time.minute = 49;
    origId.time.second = 20;
    origId.time.id = 0;

    nn::capsrv::ScreenShotOrientation orientationList[] = {
        nn::capsrv::ScreenShotOrientation_Default,
        nn::capsrv::ScreenShotOrientation_Rotate90,
        nn::capsrv::ScreenShotOrientation_Rotate180,
        nn::capsrv::ScreenShotOrientation_Rotate270,
    };
    int orientationCount = sizeof(orientationList) / sizeof(orientationList[0]);

    std::mt19937 rand;
    rand.seed(0x987654AB);
    auto imageData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, 2, rand);
    auto thumbData = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height, 2, rand);

    std::uniform_int_distribution<int> dist;
    std::vector<uint8_t> loadBuf;
    std::vector<uint8_t> workBuf;
    for(int i = 0; i < trialCount; i++)
    {
        nn::capsrv::AppletData srcAppletData = {};
        for(size_t pos = 0; pos < sizeof(srcAppletData); pos++)
        {
            reinterpret_cast<uint8_t&>(srcAppletData.value[pos]) = static_cast<uint8_t>(dist(rand) % 256);
        }

        nn::capsrv::ScreenShotAttribute srcAttribute;
        srcAttribute.SetDefault();
        srcAttribute.size = size;
        srcAttribute.orientation = orientationList[dist(rand) % orientationCount];

        // 保存
        nn::capsrv::AlbumEntry entry = {};
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SaveEditedScreenShot(
                &entry,
                imageData.data.data(),
                imageData.data.size(),
                width,
                height,
                thumbData.data.data(),
                thumbData.data.size(),
                nn::capsrv::ViewerThumbnailImageSize_Width,
                nn::capsrv::ViewerThumbnailImageSize_Height,
                srcAttribute,
                srcAppletData,
                &origId
            ));
        }

        // ロードしてみる
        {
            int loadWidth = 0;
            int loadHeight = 0;
            nn::capsrv::ScreenShotAttribute attributeThumb = {};
            nn::capsrv::AppletData appletDataThumb = {};
            loadBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
            workBuf.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
            NN_LOG("LoadAlbumScreenShotThumbnailImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotThumbnailImage(
                &loadWidth, &loadHeight, &attributeThumb, &appletDataThumb, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, loadWidth);
            EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, loadHeight);
            EXPECT_EQ(nn::capsrv::ScreenShotSize_1280x720, attributeThumb.size); // srcAttribute とは無関係に決まる
            EXPECT_EQ(srcAttribute.orientation, attributeThumb.orientation);     // srcAttribute の値を引き継ぐ
            EXPECT_EQ(nn::capsrv::AlbumFileDescription_ScreenShotEdited, attributeThumb.description);
            EXPECT_EQ(1, attributeThumb.frameCount);
            EXPECT_EQ(0, attributeThumb.frameRateNumerator);
            EXPECT_EQ(0, attributeThumb.frameRateDenominator);
            EXPECT_EQ(0, attributeThumb.dataDurationMilliseconds);
            EXPECT_EQ(0, attributeThumb.keyFrameInterval);
            EXPECT_EQ(0, std::memcmp(&srcAppletData, &appletDataThumb, sizeof(nn::capsrv::AppletData)));

            nn::capsrv::ScreenShotAttribute attributeFull = {};
            nn::capsrv::AppletData appletDataFull = {};
            loadBuf.resize(4 * width * height);
            workBuf.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("LoadAlbumScreenShotImage()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(
                &loadWidth, &loadHeight, &attributeFull, &appletDataFull, loadBuf.data(), loadBuf.size(), &entry.fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), workBuf.data(), workBuf.size()
            ));
            EXPECT_EQ(width, loadWidth);
            EXPECT_EQ(height, loadHeight);
            EXPECT_EQ(0, std::memcmp(&attributeThumb, &attributeFull, sizeof(nn::capsrv::ScreenShotAttribute)));
            EXPECT_EQ(0, std::memcmp(&appletDataThumb, &appletDataFull, sizeof(nn::capsrv::AppletData)));
        }
    }
}


TEST(AlbumAccessApi, SaveEditedScreenShot_ComplexImage)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    const nn::settings::system::PrimaryAlbumStorage PrimaryAlbumStorageList[] = {
        nn::settings::system::PrimaryAlbumStorage_Nand,
        nn::settings::system::PrimaryAlbumStorage_SdCard,
    };

    NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE_I(idx,primaryStorage)
    {
        // 保存先を変更
        NN_LOG("Set primary storage %d\n", idx);
        nn::settings::system::SetPrimaryAlbumStorage(PrimaryAlbumStorageList[idx]);

        const int width = 1280;
        const int height = 720;
        const int thumbWidth  = nn::capsrv::ViewerThumbnailImageSize_Width;
        const int thumbHeight = nn::capsrv::ViewerThumbnailImageSize_Height;
        std::vector<uint8_t> srcData;
        srcData.resize(4 * width * height);
        std::vector<uint8_t> srcThumb;
        srcThumb.resize(4 * thumbWidth * thumbHeight);

        // 書き込むデータを生成
        // とても品質を落とさないと保存できない
        {
            std::mt19937 rand;
            rand.seed(0x125690CD);
            std::uniform_int_distribution<int> dist;

            uint8_t* p = srcData.data();
            for(int y = 0; y < height; y++)
            {
                for(int x = 0; x < width; x++)
                {
                    *(p++) = static_cast<uint8_t>(dist(rand));
                    *(p++) = static_cast<uint8_t>(dist(rand));
                    *(p++) = static_cast<uint8_t>(dist(rand));
                    *(p++) = 255;
                }
            }
            uint8_t* q = srcThumb.data();
            for(int y = 0; y < thumbHeight; y++)
            {
                for(int x = 0; x < thumbWidth; x++)
                {
                    *(q++) = static_cast<uint8_t>(dist(rand));
                    *(q++) = static_cast<uint8_t>(dist(rand));
                    *(q++) = static_cast<uint8_t>(dist(rand));
                    *(q++) = 255;
                }
            }
        }

        // 適当に編集元の ID をでっちあげる
        nn::capsrv::AlbumFileId origId = {};
        origId.applicationId.value = 0x1234567890123456;
        origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
        origId.storage = nn::capsrv::AlbumStorage_Sd;
        origId.time.year = 2017;
        origId.time.month = 8;
        origId.time.day = 12;
        origId.time.hour = 10;
        origId.time.minute = 12;
        origId.time.second = 22;
        origId.time.id = 0;

        nn::capsrv::AlbumEntry newEntry = {};

        nn::capsrv::ScreenShotAttribute attribute;
        attribute.SetDefault();
        nn::capsrv::AppletData appletData = {};
        auto result = nn::capsrv::SaveEditedScreenShot(
                &newEntry,
                srcData.data(),
                srcData.size(),
                width,
                height,
                srcThumb.data(),
                srcThumb.size(),
                thumbWidth,
                thumbHeight,
                attribute,
                appletData,
                &origId
            );

        if(result.IsSuccess())
        {
            EXPECT_LE(newEntry.size, nn::capsrv::AlbumFileSizeLimit_ScreenShot);
            NN_LOG("ComplexImage -> %lld bytes\n", static_cast<int64_t>(newEntry.size));
        }
        else
        {
            NN_LOG("ComplexImage encode error (%d-%d)\n", result.GetModule(), result.GetDescription());
            FAIL();
        }
    }

    nn::capsrv::FinalizeAlbumAccess();
} // NOLINT(impl/function_size)

// アルバムの使用制限のテスト
TEST(AlbumAccessApi, SaveEditedScreenShot_AlbumLimitation)
{
    nnt::capsrv::StartupTestCase();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SetInternalConversionEnabled(false));
    // NAND だけ使う
    NNT_CAPSRV_FOREACH_STORAGE(storage)
    {
        if(storage != nn::capsrv::AlbumStorage_Nand)
        {
            nn::capsrv::ForceAlbumUnmounted(storage);
        }
    }

    const int width = 1280;
    const int height = 720;
    const int blockSize = 16;
    const int thumbWidth  = 320;
    const int thumbHeight = 180;
    const int thumbBlockSize = 4;
    std::mt19937 engine(0xD2CAA640);
    nn::capsrv::AlbumFileId origId = {};
    {
        origId.applicationId.value = 0x1234567890123456;
        origId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
        origId.storage = nn::capsrv::AlbumStorage_Nand;
        origId.time.year = 2016;
        origId.time.month = 4;
        origId.time.day = 11;
        origId.time.hour = 18;
        origId.time.minute = 49;
        origId.time.second = 20;
        origId.time.id = 0;
    }

    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // ファイル数のテスト
    {
        NN_LOG("Creating empty files...\n");
        {
            int i;
            std::mt19937 engineDate(0x82919391);
            for(i = 0; i < nn::capsrv::AlbumFileCountLimit_NandScreenShot - 1; i++)
            {
                nn::capsrv::AlbumFileId fileId = {};
                fileId.storage = nn::capsrv::AlbumStorage_Nand;
                // NOTE: 年月日をランダムにするとサブディレクトリ掘るのに時間がかかる
                fileId.time.year  = 2019;
                fileId.time.month = 10;
                fileId.time.day   = 22;
                fileId.time.hour   = engine() % 24;
                fileId.time.minute = engine() % 60;
                fileId.time.second = engine() % 60;
                fileId.applicationId.value = i + 1; // 0 だけは避ける
                auto path =nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
                std::vector<uint8_t> data;
                nnt::capsrv::DirectAlbumAccessor::SaveFile(data, path.c_str());
                if((i + 1) % 100 == 0)
                {
                    NN_LOG("  Created %d files\n", i + 1);
                }
            }
            NN_LOG("  Created %d files ... complete\n", i + 1);
        }

        // 不正なパスのファイルを作成
        {
            nn::capsrv::AlbumFileId fileId = {};
            fileId.storage = nn::capsrv::AlbumStorage_Nand;
            fileId.time.year  = 2222;
            fileId.time.month = 3;
            fileId.time.day   = 15;
            fileId.applicationId.value = 1; // 0 以外なら何でもよい
            auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
            path[7] = '3'; // 年を書き換え 2222->2232
            std::vector<uint8_t> data;
            nnt::capsrv::DirectAlbumAccessor::SaveFile(data, path.c_str());
            NN_LOG("  Created empty file at invalid filepath %s\n", path.c_str());
        }

        // キャッシュを更新させる
        NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE(storage)
        {
            nn::capsrv::RefreshAlbumCache(storage);
        }

        {
            nn::capsrv::AlbumEntry entry = {};
            auto raw = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, blockSize, engine);
            auto rawThumb = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(thumbWidth, thumbHeight, thumbBlockSize, engine);
            // もう 1 枚は撮れる
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            nn::capsrv::AppletData appletData = {};
            EXPECT_TRUE(nn::capsrv::SaveEditedScreenShot(
                &entry, raw.data.data(), raw.data.size(), width, height, rawThumb.data.data(), rawThumb.data.size(), thumbWidth, thumbHeight, attribute, appletData, &origId
            ).IsSuccess());
            // 更に 1 枚は撮れない
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SetInternalConversionEnabled(true));
            EXPECT_TRUE(nn::capsrv::ResultAlbumFileCountLimit::Includes( nn::capsrv::SaveEditedScreenShot(
                    &entry, raw.data.data(), raw.data.size(), width, height, rawThumb.data.data(), rawThumb.data.size(), thumbWidth, thumbHeight, attribute, appletData, &origId
            )));
            //NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SetInternalConversionEnabled(false));
            //EXPECT_TRUE(nn::capsrv::server::ResultInternalAlbumLimitationFileCountLimit::Includes( nn::capsrv::SaveEditedScreenShot(
            //        &entry, raw.data.data(), raw.data.size(), width, height, rawThumb.data.data(), rawThumb.data.size(), thumbWidth, thumbHeight, attribute, appletData, &origId
            //)));
        }
    }

    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 合計ファイルサイズのテスト
    // NOTE: Scoop 管理外のファイルは制限に考慮されない。
#if 0
    {
        nn::capsrv::AlbumCacheData expectedCacheData = {};
        NN_LOG("Creating large files...\n");
        // 1 枚は入る容量を残して大きいファイルを作る
        {
            int64_t unitSize = nn::capsrv::AlbumFileSizeLimit_ScreenShot;
            int64_t size = nn::capsrv::server::TotalFileSizeLimitNand - nn::capsrv::AlbumFileSizeLimit_ScreenShot;
            int64_t createdSize = 0;
            int i = 0;
            while(size > 0)
            {
                nn::capsrv::AlbumFileId fileId = {};
                fileId.storage = nn::capsrv::AlbumStorage_Nand;
                fileId.time.year  = 2010 + (engine() % 10);
                fileId.time.month = 1 + (engine() % 12);
                fileId.time.day   = 1 + (engine() % 31);
                fileId.applicationId.value = i + 1; // 0 だけは避ける
                auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
                int64_t fileSize = std::min(unitSize, size);
                nnt::capsrv::DirectAlbumAccessor::CreateFile(fileSize, path.c_str());
                size -= fileSize;
                createdSize += fileSize;
                i++;
                NN_LOG("  Created %lld bytes\n", createdSize);
            }
            NN_LOG("  Created %lld bytes ... complete\n", createdSize);
            expectedCacheData.totalFileSize += createdSize;
            expectedCacheData.fileCount += static_cast<int64_t>(i);
        }
        // 不正なパスのファイルを作成
        {
            nn::capsrv::AlbumFileId fileId = {};
            fileId.storage = nn::capsrv::AlbumStorage_Nand;
            fileId.time.year  = 2222;
            fileId.time.month = 6;
            fileId.time.day   = 20;
            fileId.applicationId.value = 1; // 0 以外なら何でもよい
            auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
            path[7] = '3'; // 年を書き換え 2222->2223
            std::vector<uint8_t> data;
            nnt::capsrv::DirectAlbumAccessor::CreateFile(nn::capsrv::AlbumFileSizeLimit_ScreenShot, path.c_str());
            NN_LOG("  Created file at invalid filepath %s\n", path.c_str());
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(nn::capsrv::AlbumStorage_Nand));

        // キャッシュを確認
        {
            nn::capsrv::AlbumCacheData cacheData = {};
            EXPECT_TRUE(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand).IsSuccess());
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
            EXPECT_EQ(expectedCacheData.totalFileSize, cacheData.totalFileSize);
        }

        // 撮れることを確認
        {
            nn::capsrv::AlbumEntry entry = {};
            auto raw = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, blockSize, engine);
            auto rawThumb = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(thumbWidth, thumbHeight, thumbBlockSize, engine);
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            nn::capsrv::AppletData appletData = {};
            EXPECT_TRUE(nn::capsrv::SaveEditedScreenShot(
                &entry, raw.data.data(), raw.data.size(), width, height, rawThumb.data.data(), rawThumb.data.size(), thumbWidth, thumbHeight, attribute, appletData, &origId
            ).IsSuccess());
            size_t size = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumFileSize(&size, &entry.fileId));
            expectedCacheData.totalFileSize += static_cast<int64_t>(size);
            expectedCacheData.fileCount++;
        }

        // キャッシュを確認
        {
            nn::capsrv::AlbumCacheData cacheData = {};
            EXPECT_TRUE(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand).IsSuccess());
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
            EXPECT_EQ(expectedCacheData.totalFileSize, cacheData.totalFileSize);
        }

        // 撮れないようにファイルを追加
        {
            nn::capsrv::AlbumFileId fileId = {};
            fileId.storage = nn::capsrv::AlbumStorage_Nand;
            fileId.time.year  = 2222;
            fileId.time.month = 6;
            fileId.time.day   = 20;
            fileId.applicationId.value = 1; // 0 以外なら何でもよい
            auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
            std::vector<uint8_t> data;
            nnt::capsrv::DirectAlbumAccessor::CreateFile(nn::capsrv::AlbumFileSizeLimit_ScreenShot, path.c_str());
            NN_LOG("  Created empty file at invalid filepath %s\n", path.c_str());
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(nn::capsrv::AlbumStorage_Nand));

        // キャッシュを確認
        {
            nn::capsrv::AlbumCacheData cacheData = {};
            EXPECT_TRUE(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand).IsSuccess());
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
            EXPECT_EQ(expectedCacheData.totalFileSize, cacheData.totalFileSize);
        }

        // 撮れないことを確認
        {
            nn::capsrv::AlbumEntry entry = {};
            auto raw = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(width, height, blockSize, engine);
            auto rawThumb = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(thumbWidth, thumbHeight, thumbBlockSize, engine);
            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            nn::capsrv::AppletData appletData = {};
            EXPECT_TRUE(nn::capsrv::server::ResultInternalAlbumLimitationTotalFileSizeLimit::Includes( nn::capsrv::SaveEditedScreenShot(
                    &entry, raw.data.data(), raw.data.size(), width, height, rawThumb.data.data(), rawThumb.data.size(), thumbWidth, thumbHeight, attribute, appletData, &origId
            )));
        }

        // キャッシュが変わっていないことを確認
        {
            nn::capsrv::AlbumCacheData cacheData = {};
            EXPECT_TRUE(nn::capsrv::GetAlbumCache(&cacheData, nn::capsrv::AlbumStorage_Nand).IsSuccess());
            EXPECT_EQ(expectedCacheData.fileCount, cacheData.fileCount);
            EXPECT_EQ(expectedCacheData.totalFileSize, cacheData.totalFileSize);
        }
    }
#endif


    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    // キャッシュを更新させる
    NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE(storage)
    {
        nn::capsrv::RefreshAlbumCache(storage);
    }

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, SaveEditedScreenShot_DirectoryStructureCorruption)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();
    nnt::capsrv::DirectAlbumAccessor::CleanupAlbum(nn::capsrv::AlbumStorage_Nand);
    nn::capsrv::RefreshAlbumCache(nn::capsrv::AlbumStorage_Nand);

    const int Width = 1280;
    const int Height = 720;
    const int ThumbWidth = nn::capsrv::ViewerThumbnailImageSize_Width;
    const int ThumbHeight = nn::capsrv::ViewerThumbnailImageSize_Height;

    // 書込み先を NAND に固定
    nn::settings::system::SetPrimaryAlbumStorage(nn::settings::system::PrimaryAlbumStorage_Nand);

    auto saveScreenShot = [&](nn::capsrv::AlbumEntry* pOutEntry) -> nn::Result
    {
        std::vector<uint8_t> srcData(4 * Width * Height);
        std::vector<uint8_t> srcThumb(4 * ThumbWidth * ThumbHeight);

        // 適当に編集元の ID をでっちあげる
        nn::capsrv::AlbumFileId origId = {};
        origId.applicationId.value = 0x1234567890123456;
        origId.contents    = nn::capsrv::AlbumFileContents_ScreenShot;
        origId.storage     = nn::capsrv::AlbumStorage_Sd;
        origId.time.year   = 2016;
        origId.time.month  = 4;
        origId.time.day    = 11;
        origId.time.hour   = 18;
        origId.time.minute = 49;
        origId.time.second = 20;
        origId.time.id     = 0;

        nn::capsrv::ScreenShotAttribute attribute;
        attribute.SetDefault();
        nn::capsrv::AppletData appletData = {};
        return nn::capsrv::SaveEditedScreenShot(pOutEntry, srcData.data(), srcData.size(), Width, Height, srcThumb.data(), srcThumb.size(), ThumbWidth, ThumbHeight, attribute, appletData, &origId);
    };


    // 現在の日付を取得
    // 取得するために、一度ダミーのファイルを保存する
    nn::capsrv::AlbumEntry entry = {};
    for(;;)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(saveScreenShot(&entry));

        // 日付が変わる直前だと危ないのでやり直す
        if(entry.fileId.time.hour == 23 &&
            entry.fileId.time.minute == 59 &&
            entry.fileId.time.second >= 50
            )
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            continue;
        }

        break;
    }

    // アルバムをクリーン
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAlbum(nn::capsrv::AlbumStorage_Nand));

    // サブディレクトリの名前に（ディレクトリではなく）ファイルが存在している場合に返される Result をテスト
    for(int depth = 0; depth < nnt::capsrv::DirectAlbumAccessor::GetSubDirectoryDepth(); depth++)
    {
        int length = 0;
        char path[1024] = {};
        // サブディレクトリのパスを作成
        nnt::capsrv::DirectAlbumAccessor::GetSubDirectoryPath(&length, path, sizeof(path), entry.fileId, depth);
        NN_LOG("Subdirectory path[%d]: %s\n", depth, path);

        // 末尾の '/' を除外
        if(path[length - 1] == '/')
        {
            path[length - 1] = '\0';
        }

        // サブディレクトリのパスにファイルを作成
        nnt::capsrv::DirectAlbumAccessor::CreateFile(0, path);

        // 「ディレクトリ構造が壊れている」のエラーが返ることの確認
        nn::capsrv::AlbumEntry e = {};
        auto result = saveScreenShot(&e);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDestinationStructureCorrupted, result);

        // 作ったファイルを削除
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteFile(path));
    }

    nn::capsrv::FinalizeAlbumAccess();
} // NOLINT(impl/function_size)
