﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <vector>
#include <list>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/capsrv/movie/capsrv_MovieReaderFileSystem.h>
#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "testCapsrv_StartupTestCase.h"
#include "detail/testCapsrv_TestMovieStreamWrapperBase.h"
#include "detail/testCapsrv_TestMovieStreamRandomAccessImpl.h"

//static int RepeatCount = 100;

#define NNT_CAPSRV_IO_LOG(...) NN_LOG(__VA_ARGS__)

#ifndef NNT_CAPSRV_IO_LOG
#define NNT_CAPSRV_IO_LOG(...)
#endif

#define NNT_CAPSRV_FILELIST(varList, varCount)  \
    const nnt::capsrv::FileInfo varList[] = {  \
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .mp4, 10000000, 12345),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .mp4, 10000000, 12346),   \
    };  \
    static const int varCount = sizeof(varList) / sizeof(varList[0]);   \
    NN_UNUSED(varCount);    \
    NN_UNUSED(varList);

namespace {

    std::vector<std::vector<nn::capsrv::AlbumEntry>> SetupTestMovieFiles() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());

        EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

        NNT_CAPSRV_FILELIST(fileList, fileCount);
        // ファイル作成
        for(int i = 0; i < fileCount; i++){ fileList[i].CreateMovie(); }

        std::vector<std::vector<nn::capsrv::AlbumEntry>> entryListList;
        entryListList.resize(nn::capsrv::AlbumStorageCount);

        // ファイルリスト取得
        NNT_CAPSRV_FOREACH_STORAGE_I(s, storage)
        {
            auto& list = entryListList[s];
            list = nnt::capsrv::GetAlbumEntryList(storage);
            nnt::capsrv::SortAlbumEntryListStorageTime(list);
        }
        nn::capsrv::FinalizeAlbumAccess();

        return entryListList;
    }
}


namespace {
    class TestFile
        : public nnt::capsrv::detail::TestMovieStreamWrapperBase
    {
    public:
        void Initialize(nn::fs::FileHandle h) NN_NOEXCEPT
        {
            m_FileHandle = h;
        }

        void Finalize() NN_NOEXCEPT
        {
            m_FileHandle = {};
        }

    protected:
        nn::Result WriteImpl(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::WriteFile(m_FileHandle, offset, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        }

        nn::Result ReadImpl(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::ReadFile(pOutReadSize, m_FileHandle, offset, buffer, size);
        }

        nn::Result GetSizeImpl(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::GetFileSize(pOutValue, m_FileHandle);
        }

        nn::Result SetSizeImpl(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::SetFileSize(m_FileHandle, size);
        }

        nn::Result FlushImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::FlushFile(m_FileHandle);
        }

        const uint8_t* GetInternalStorageImpl(size_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(pOutSize);
            return nullptr;
        }

    private:
        nn::fs::FileHandle m_FileHandle;
    };
}

namespace {

    void TestMovieReaderFileSystemImpl(
        size_t sizeToReadMax,
        int64_t chunkSize,
        int64_t cacheCount
    )
    {
        NN_LOG("sizeToReadMax   = %llu\n", sizeToReadMax);
        NN_LOG("cacheChunkSize  = %llu\n", chunkSize);
        NN_LOG("cacheChunkCount = %llu\n", cacheCount);

        // ワークメモリ
        size_t mwfsSize = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemorySizeForFile(chunkSize, cacheCount);
        size_t mwfsAlignment = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemoryAlignmentForFile();
        NN_LOG("memorySize  = %llu\n", mwfsSize);
        NN_LOG("memoryAlign = %llu\n", mwfsAlignment);
        std::vector<char> mem(mwfsSize);

        // ファイルシステム
        nn::capsrv::movie::MovieReaderFileSystem mrfs;
        mrfs.Initialize("Movie");
        NN_UTIL_SCOPE_EXIT{ mrfs.Finalize(); };


        NNT_CAPSRV_FILELIST(fileList, fileCount);
        for(int i = 0; i < fileCount; i++)
        {
            auto fileId = fileList[i].GetAlbumFileId();
            if(!nn::capsrv::IsAlbumMounted(fileId.storage))
            {
                continue;
            }
            auto fileData = fileList[i].GetData();

            // 動画読込みストリームを作成
            nn::capsrv::AlbumMovieReadStreamHandle h = {};
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&h, fileId));
            NN_UTIL_SCOPE_EXIT{ NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CloseAlbumMovieReadStreamImpl(h)); };

            // データ部分の読込み
            {
                const char* filename = "movie123456.mp4";
                const char* filepath = "Movie:/movie123456.mp4";

                // ファイルシステムにアタッチ
                // 失敗するのはアタッチ可能な数の上限に到達したときだけ。
                NN_ABORT_UNLESS_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection(filename, h, chunkSize, cacheCount, mem.data(), mem.size()));
                NN_UTIL_SCOPE_EXIT{ mrfs.UnregisterMovieReadStreamDataSection(h); };

                nn::fs::FileHandle handle = {};
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filepath, nn::fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

                int64_t fileSize = 0;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));
                EXPECT_EQ(static_cast<int64_t>(fileData.size()), fileSize);

                TestFile tf;
                tf.Initialize(handle);
                {
                    int64_t remain = fileSize;
                    int64_t pos = 0;
                    while(remain > 0)
                    {
                        size_t sizeToRead = std::min(static_cast<size_t>(remain), sizeToReadMax);
                        size_t readSize = 0;
                        std::vector<uint8_t> readData(sizeToRead);
                        NN_LOG("reading [%llu - %llu)\n", pos, pos + sizeToRead);
                        NNT_EXPECT_RESULT_SUCCESS(tf.Read(&readSize, readData.data(), sizeToRead, pos));
                        EXPECT_EQ(sizeToRead, readSize);
                        EXPECT_EQ(0, std::memcmp(fileData.data() + pos, readData.data(), sizeToRead));
                        remain -= static_cast<int64_t>(sizeToRead);
                        pos    += static_cast<int64_t>(sizeToRead);
                    }
                }
                tf.Finalize();
            }
        }
    }

}



TEST(AlbumAccessApi, MovieReaderFileSystem_Read)
{
    const size_t SizeToReadMax = 64 * 1024;
    const int64_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    const int64_t CacheCount = 4;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    (void)SetupTestMovieFiles();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    //NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    //NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    TestMovieReaderFileSystemImpl(SizeToReadMax, ChunkSize, CacheCount);

    TestMovieReaderFileSystemImpl(SizeToReadMax, ChunkSize, CacheCount);
    TestMovieReaderFileSystemImpl(SizeToReadMax, ChunkSize, CacheCount);

    TestMovieReaderFileSystemImpl(       200, ChunkSize, CacheCount);
    TestMovieReaderFileSystemImpl(250 * 1024, ChunkSize, CacheCount);

    TestMovieReaderFileSystemImpl(SizeToReadMax, 2 * ChunkSize, CacheCount);
    TestMovieReaderFileSystemImpl(SizeToReadMax, 3 * ChunkSize, CacheCount);

    TestMovieReaderFileSystemImpl(SizeToReadMax, ChunkSize, 1);
    TestMovieReaderFileSystemImpl(SizeToReadMax, ChunkSize, 2);
}



TEST(AlbumAccessApi, MovieReaderFileSystem_Random)
{
    const int64_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    const int64_t CacheCount = 4;
    std::mt19937 rand(0x24572);

    nnt::capsrv::detail::TestMovieStreamRandomAccess::Config config = {};
    config.loopCount = 300;
    config.chunkSize = ChunkSize;
    config.offsetMax = 4 * 1024 * 1024;
    config.sizeMax   = 4 * 1024;
    config.offsetCutoffThreshold   = 256 * 1024;
    config.sizeCutoffThreshold     = 256;
    config.probabilityWrite  = 0;
    config.probabilityRead   = 1;
    config.probabilityResize = 0;
    config.probabilityFlush  = 0;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    SetupTestMovieFiles();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    // ワークメモリ
    size_t mwfsSize = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemorySizeForFile(ChunkSize, CacheCount);
    size_t mwfsAlignment = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemoryAlignmentForFile();
    NN_LOG("memorySize  = %llu\n", mwfsSize);
    NN_LOG("memoryAlign = %llu\n", mwfsAlignment);
    std::vector<char> mem(mwfsSize);

    // ファイルシステム
    nn::capsrv::movie::MovieReaderFileSystem mrfs;
    mrfs.Initialize("Movie");
    NN_UTIL_SCOPE_EXIT{ mrfs.Finalize(); };

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    for(int i = 0; i < fileCount; i++)
    {
        NN_LOG("file: %d\n", i);
        auto fileId = fileList[i].GetAlbumFileId();
        if(!nn::capsrv::IsAlbumMounted(fileId.storage))
        {
            continue;
        }

        nn::capsrv::AlbumMovieReadStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&h, fileId));
        NN_UTIL_SCOPE_EXIT{ nn::capsrv::CloseAlbumMovieReadStream(h); };

        const char* filename = "a.mp4";
        const char* filepath = "Movie:/a.mp4";

        NN_ABORT_UNLESS_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection(filename, h, ChunkSize, CacheCount, mem.data(), mem.size()));
        NN_UTIL_SCOPE_EXIT{ mrfs.UnregisterMovieReadStreamDataSection(h); };

        nn::fs::FileHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filepath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

        TestFile tf;
        tf.Initialize(handle);
        NN_UTIL_SCOPE_EXIT{ tf.Finalize(); };

        tf.shadow = fileList[i].GetData();

        nnt::capsrv::detail::TestMovieStreamRandomAccess::Test(tf, config, rand);
    }
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, MovieReaderFileSystem_RegisterMultipleFile)
{
    const int64_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    const int64_t CacheCount = 1;

    static const int SlotCount = 4;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    SetupTestMovieFiles();

    NNT_CAPSRV_FILELIST(fileList, fileCount);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    // ストリーム
    nn::capsrv::AlbumMovieReadStreamHandle handles[2] = {};
    for(int i = 0; i < 2; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&handles[i], fileList[i].GetAlbumFileId()));
    }
    NN_UTIL_SCOPE_EXIT{
        for(int i = 0; i < 2; i++)
        {
            nn::capsrv::CloseAlbumMovieReadStream(handles[i]);
        }
    };

    // ワークメモリ
    size_t mrfsSize = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemorySizeForFile(ChunkSize, CacheCount);
    size_t mrfsAlignment = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemoryAlignmentForFile();
    void* work[SlotCount] = {};
    for(int i = 0; i < SlotCount; i++)
    {
        work[i] = aligned_alloc(mrfsAlignment, mrfsSize);
        NN_ABORT_UNLESS_NOT_NULL(work[i]);
    }
    NN_UTIL_SCOPE_EXIT{
        for(int i = 0; i < SlotCount; i++)
        {
            free(work[i]);
            work[i] = nullptr;
        }
    };

    auto checkOpenable = [&](const char* filepath) -> nn::Result
    {
        nn::fs::FileHandle hFile = {};
        auto result = nn::fs::OpenFile(&hFile, filepath, nn::fs::OpenMode_Read);
        if(result.IsSuccess())
        {
            nn::fs::CloseFile(hFile);
        }
        return result;
    };

    // ファイルシステム
    nn::capsrv::movie::MovieReaderFileSystem mrfs;
    mrfs.Initialize("Movie");
    NN_UTIL_SCOPE_EXIT{ mrfs.Finalize(); };

    // 異なるストリームが同じファイル名で登録できないこと
    NN_LOG("checking filename duplication\n");
    {
        NNT_EXPECT_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection("movie0.mp4", handles[0], ChunkSize, CacheCount, work[0], mrfsSize));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultAlreadyExists, mrfs.RegisterMovieReadStreamDataSection("movie0.mp4", handles[1], ChunkSize, CacheCount, work[1], mrfsSize));
        mrfs.UnregisterMovieReadStreamDataSection("movie0.mp4");
    }

    // 同じストリームが異なるファイル名で登録できること
    NN_LOG("checking stream duplication\n");
    {
        NNT_EXPECT_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection("movie0.mp4", handles[0], ChunkSize, CacheCount, work[0], mrfsSize));
        NNT_EXPECT_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection("movie1.mp4", handles[0], ChunkSize, CacheCount, work[1], mrfsSize));
        NNT_EXPECT_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection("movie2.mp4", handles[1], ChunkSize, CacheCount, work[2], mrfsSize));
        NNT_EXPECT_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection("movie3.mp4", handles[1], ChunkSize, CacheCount, work[3], mrfsSize));
        NNT_EXPECT_RESULT_SUCCESS(checkOpenable("Movie:/movie0.mp4"));
        NNT_EXPECT_RESULT_SUCCESS(checkOpenable("Movie:/movie1.mp4"));
        NNT_EXPECT_RESULT_SUCCESS(checkOpenable("Movie:/movie2.mp4"));
        NNT_EXPECT_RESULT_SUCCESS(checkOpenable("Movie:/movie3.mp4"));
    }

    // ファイル名指定で登録解除できること
    NN_LOG("checking stream duplication\n");
    {
        mrfs.UnregisterMovieReadStreamDataSection("movie1.mp4");
        NNT_EXPECT_RESULT_SUCCESS(checkOpenable("Movie:/movie0.mp4"));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, checkOpenable("Movie:/movie1.mp4"));
        NNT_EXPECT_RESULT_SUCCESS(checkOpenable("Movie:/movie2.mp4"));
        NNT_EXPECT_RESULT_SUCCESS(checkOpenable("Movie:/movie3.mp4"));
    }

    // ストリーム指定で登録解除できること
    NN_LOG("checking stream duplication\n");
    {
        mrfs.UnregisterMovieReadStreamDataSection(handles[1]);
        NNT_EXPECT_RESULT_SUCCESS(checkOpenable("Movie:/movie0.mp4"));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, checkOpenable("Movie:/movie1.mp4"));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, checkOpenable("Movie:/movie2.mp4"));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, checkOpenable("Movie:/movie3.mp4"));
    }

    {
        mrfs.UnregisterMovieReadStreamDataSection("movie0.mp4");
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, checkOpenable("Movie:/movie0.mp4"));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, checkOpenable("Movie:/movie1.mp4"));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, checkOpenable("Movie:/movie2.mp4"));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, checkOpenable("Movie:/movie3.mp4"));
    }

}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, MovieReaderFileSystem_ByControl_Random)
{
    const int64_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    const int64_t CacheCount = 4;
    std::mt19937 rand(0x24572);

    nnt::capsrv::detail::TestMovieStreamRandomAccess::Config config = {};
    config.loopCount = 100;
    config.chunkSize = ChunkSize;
    config.offsetMax = 4 * 1024 * 1024;
    config.sizeMax   = 4 * 1024;
    config.offsetCutoffThreshold   = 256 * 1024;
    config.sizeCutoffThreshold     = 256;
    config.probabilityWrite  = 0;
    config.probabilityRead   = 1;
    config.probabilityResize = 0;
    config.probabilityFlush  = 0;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
    SetupTestMovieFiles();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    // ワークメモリ
    size_t mwfsSize = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemorySizeForFile(ChunkSize, CacheCount);
    size_t mwfsAlignment = nn::capsrv::movie::MovieReaderFileSystem::GetRequiredMemoryAlignmentForFile();
    NN_LOG("memorySize  = %llu\n", mwfsSize);
    NN_LOG("memoryAlign = %llu\n", mwfsAlignment);
    std::vector<char> mem(mwfsSize);

    // ファイルシステム
    nn::capsrv::movie::MovieReaderFileSystem mrfs;
    mrfs.Initialize("Movie");
    NN_UTIL_SCOPE_EXIT{ mrfs.Finalize(); };

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    for(int i = 0; i < fileCount; i++)
    {
        NN_LOG("file: %d\n", i);
        auto fileId = fileList[i].GetAlbumFileId();
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
            NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
            if(!nn::capsrv::IsAlbumMounted(fileId.storage))
            {
                continue;
            }
        }

        nn::capsrv::AlbumMovieReadStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&h, fileId));
        NN_UTIL_SCOPE_EXIT{ nn::capsrv::CloseAlbumMovieReadStream(h); };

        const char* filename = "a.mp4";
        const char* filepath = "Movie:/a.mp4";

        NN_ABORT_UNLESS_RESULT_SUCCESS(mrfs.RegisterMovieReadStreamDataSection(filename, h, ChunkSize, CacheCount, mem.data(), mem.size()));
        NN_UTIL_SCOPE_EXIT{ mrfs.UnregisterMovieReadStreamDataSection(h); };

        nn::fs::FileHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filepath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

        TestFile tf;
        tf.Initialize(handle);
        NN_UTIL_SCOPE_EXIT{ tf.Finalize(); };

        tf.shadow = fileList[i].GetData();

        nnt::capsrv::detail::TestMovieStreamRandomAccess::Test(tf, config, rand);
    }
}// NOLINT(impl/function_size)
