﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nn/fs.h>

#include <nnt.h>

#include <nn/settings/system/settings_Capture.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_Macro.h"
#include "testCapsrv_StartupTestCase.h"


namespace {

    struct TestCaseInfo
    {
        nn::settings::system::PrimaryAlbumStorage primaryStorage;
        bool isNandMounted;
        bool isSdMounted;
        void (*expectedFunction)();
    };

    void PrintTestCase(const TestCaseInfo& testCase)
    {
        NN_LOG(
            "TestCase: Primary=%d / NAND %s / SD %s\n",
            static_cast<int>(testCase.primaryStorage),
            testCase.isNandMounted ? "o" : "x",
            testCase.isSdMounted ? "o" : "x"
            );
    }

    void ExpectNand()
    {
        nn::capsrv::AlbumStorageType s;
        EXPECT_TRUE(nn::capsrv::GetAutoSavingStorage(&s).IsSuccess());
        EXPECT_EQ(nn::capsrv::AlbumStorage_Nand, s);
    }

    void ExpectSd()
    {
        nn::capsrv::AlbumStorageType s;
        EXPECT_TRUE(nn::capsrv::GetAutoSavingStorage(&s).IsSuccess());
        EXPECT_EQ(nn::capsrv::AlbumStorage_Sd, s);
    }

    void ExpectError()
    {
        nn::capsrv::AlbumStorageType s;
        EXPECT_TRUE(nn::capsrv::ResultAlbumIsNotMounted::Includes(nn::capsrv::GetAutoSavingStorage(&s)));
    }

}

TEST(AlbumAccessApi, GetAutoSavingStorage)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 取得されるべき結果
    //
    //   | 優先ストレージ | NAND | SD    |
    //   +----------------+------+-------+
    //   | NAND有／SD有   | NAND | SD    |
    //   | NAND有／SD無   | NAND | NAND  |
    //   | NAND無／SD有   | err  | SD    |
    //   | NAND無／SD無   | err  | err   |
    //
    TestCaseInfo TestCases[] = {
        { nn::settings::system::PrimaryAlbumStorage_Nand  , true , true , ExpectNand  },
        { nn::settings::system::PrimaryAlbumStorage_Nand  , true , false, ExpectNand  },
        { nn::settings::system::PrimaryAlbumStorage_Nand  , false, true , ExpectError },
        { nn::settings::system::PrimaryAlbumStorage_Nand  , false, false, ExpectError },

        { nn::settings::system::PrimaryAlbumStorage_SdCard, true , true , ExpectSd    },
        { nn::settings::system::PrimaryAlbumStorage_SdCard, true , false, ExpectNand  },
        { nn::settings::system::PrimaryAlbumStorage_SdCard, false, true , ExpectSd    },
        { nn::settings::system::PrimaryAlbumStorage_SdCard, false, false, ExpectError },
    };
    const int TestCaseCount = static_cast<int>(sizeof(TestCases) / sizeof(TestCases[0]));

    for(int i = 0; i < TestCaseCount; i++)
    {
        auto& testCase = TestCases[i];
        PrintTestCase(testCase);
        // 物理的にストレージが差さっておらず条件が満たせない場合スキップ
        if(testCase.isNandMounted && !nnt::capsrv::DirectAlbumAccessor::IsMounted(nn::capsrv::AlbumStorage_Nand))
        {
            NN_LOG("  skipped\n");
            continue;
        }
        if(testCase.isSdMounted && !nnt::capsrv::DirectAlbumAccessor::IsMounted(nn::capsrv::AlbumStorage_Sd))
        {
            NN_LOG("  skipped\n");
            continue;
        }

        // 条件をセットアップ
        nn::settings::system::SetPrimaryAlbumStorage(testCase.primaryStorage);

        NNT_CAPSRV_FOREACH_STORAGE(storage)
        {
            nn::capsrv::ResetAlbumMountStatus(storage);
        }
        if(!testCase.isNandMounted)
        {
            nn::capsrv::ForceAlbumUnmounted(nn::capsrv::AlbumStorage_Nand);
        }
        if(!testCase.isSdMounted)
        {
            nn::capsrv::ForceAlbumUnmounted(nn::capsrv::AlbumStorage_Sd);
        }

        // 取得される値が期待通りかチェック
        testCase.expectedFunction();
    }

    nn::capsrv::FinalizeAlbumAccess();

    SUCCEED();
}// NOLINT(impl/function_size)
