﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nn/fs.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_Macro.h"
#include "testCapsrv_StartupTestCase.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/album/capsrvServer_AlbumPathUtility.h"

TEST(AlbumAccessApi, GetAlbumUsage)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, 1024, 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .jpg, 1024, 12346),
        NNT_CAPSRV_FILEINFO(NA, 2016, 05, 03, 18, 31, 58, 02, 0xBBBBBBBBBBBBBBBB, .jpg, 1024, 12347),
        NNT_CAPSRV_FILEINFO(SD, 2016, 07, 04, 15, 20, 57, 03, 0xCCCCCCCCCCCCCCCC, .jpg, 1024, 12348),
        NNT_CAPSRV_FILEINFO(SD, 2016, 08, 06, 14, 10, 56, 04, 0xDDDDDDDDDDDDDDDD, .jpg, 1024, 12349),
        NNT_CAPSRV_FILEINFO(NA, 2016, 02, 20, 08, 02, 13, 05, 0xABCABCABCABCABCA, .mp4, 100000, 12350),
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 21, 07, 03, 17, 06, 0xAABBCCAABBCCAABB, .mp4, 100000, 12351),
        NNT_CAPSRV_FILEINFO(NA, 2016, 06, 22, 06, 05, 19, 07, 0xABCDDCBAABCDDCBA, .mp4, 100000, 12352),
        NNT_CAPSRV_FILEINFO(SD, 2016, 08, 23, 05, 07, 23, 08, 0xFEDCBAFEDCBAFEDC, .mp4, 100000, 12353),
        NNT_CAPSRV_FILEINFO(SD, 2016, 10, 24, 04, 11, 29, 09, 0x0001112223334445, .mp4, 100000, 12354),
    };
    static const int TotalFileCount = sizeof(FileList) / sizeof(FileList[0]);

    nn::capsrv::AlbumFileContentsFlag contentsFlagAll;
    contentsFlagAll.Set();

    // 全使用量が 0 になっていることを確認
    NN_LOG("checking usage is zero\n");
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(storage)
    {
        nn::capsrv::AlbumUsage u = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumUsage(&u, storage, contentsFlagAll));
        for(nn::capsrv::AlbumFileContentsType contents = 0; contents < nn::capsrv::AlbumUsage::ContentsCount; contents++)
        {
            auto expectedContents = 0;
            if(contents < nn::capsrv::AlbumFileContentsCount)
            {
                expectedContents = contents;
            }
            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetContentsUsage(contents);
            EXPECT_EQ(0, pContentsUsage->count);
            EXPECT_EQ(0, pContentsUsage->size);
            EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
            EXPECT_FALSE(pContentsUsage->IsUnknownContents());
            EXPECT_EQ(expectedContents, pContentsUsage->contents);
        }
        {
            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetUnknownUsage();
            EXPECT_EQ(0, pContentsUsage->count);
            EXPECT_EQ(0, pContentsUsage->size);
            EXPECT_EQ(0, pContentsUsage->contents);
            EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
            EXPECT_TRUE(pContentsUsage->IsUnknownContents());
        }
    }

    // ファイル作成
    int createdFileCountList[nn::capsrv::AlbumStorageCount][nn::capsrv::AlbumFileContentsCount] = {};
    uint64_t createdFileSizeList[nn::capsrv::AlbumStorageCount][nn::capsrv::AlbumFileContentsCount] = {};
    NNT_CAPSRV_FOREACH_FILEINFO(finfo, FileList, TotalFileCount)
    {
        if(finfo.Create())
        {
            createdFileCountList[finfo.storage][finfo.GetAlbumFileId().contents]++;
            createdFileSizeList[finfo.storage][finfo.GetAlbumFileId().contents] += finfo.filesize;
        }
    }

    // 再び使用量を確認
    NN_LOG("checking usage(all)\n");
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE_I(s, storage)
    {
        nn::capsrv::AlbumUsage u = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumUsage(&u, storage, contentsFlagAll));

        nn::capsrv::AlbumContentsUsage expectedUnknownUsage = {};
        expectedUnknownUsage.flags |= nn::capsrv::AlbumContentsUsageFlag_IsUnknownContents;

        for(nn::capsrv::AlbumFileContentsType contents = 0; contents < nn::capsrv::AlbumFileContentsCount; contents++)
        {
            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetContentsUsage(contents);
            EXPECT_EQ(createdFileCountList[s][contents], pContentsUsage->count);
            EXPECT_EQ(createdFileSizeList[s][contents], pContentsUsage->size);
            EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
            EXPECT_FALSE(pContentsUsage->IsUnknownContents());
            EXPECT_EQ(contents, pContentsUsage->contents);
        }
        for(nn::capsrv::AlbumFileContentsType contents = nn::capsrv::AlbumFileContentsCount; contents < nn::capsrv::AlbumUsage::ContentsCount; contents++)
        {
            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetContentsUsage(contents);
            EXPECT_EQ(0, pContentsUsage->count);
            EXPECT_EQ(0, pContentsUsage->size);
            EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
            EXPECT_FALSE(pContentsUsage->IsUnknownContents());
            EXPECT_EQ(0, pContentsUsage->contents);
        }
        {
            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetUnknownUsage();
            EXPECT_EQ(expectedUnknownUsage.count, pContentsUsage->count);
            EXPECT_EQ(expectedUnknownUsage.size, pContentsUsage->size);
            EXPECT_EQ(expectedUnknownUsage.contents, pContentsUsage->contents);
            EXPECT_EQ(expectedUnknownUsage.HasGreaterUsage(), pContentsUsage->HasGreaterUsage());
            EXPECT_EQ(expectedUnknownUsage.IsUnknownContents(), pContentsUsage->IsUnknownContents());
        }
    }

    // Contents を制限して使用量を確認
    auto checkLimitedContentsUsage = [&](nn::capsrv::AlbumFileContentsType targetContents) -> void
    {
        NNT_CAPSRV_FOREACH_MOUNTED_STORAGE_I(s, storage)
        {
            nn::capsrv::AlbumFileContentsFlag contentsMask = {};
            if(targetContents < nn::capsrv::AlbumFileContentsCount)
            {
                contentsMask.Set(targetContents);
            }

            nn::capsrv::AlbumUsage u = {};
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumUsage(&u, storage, contentsMask));

            nn::capsrv::AlbumContentsUsage expectedUnknownUsage = {};
            expectedUnknownUsage.flags |= nn::capsrv::AlbumContentsUsageFlag_IsUnknownContents;
            for(nn::capsrv::AlbumFileContentsType contents = 0; contents < nn::capsrv::AlbumFileContentsCount; contents++)
            {
                // target 以外を unknown に加算
                if(contents == targetContents)
                {
                    continue;
                }
                expectedUnknownUsage.count += createdFileCountList[s][contents];
                expectedUnknownUsage.size  += createdFileSizeList[s][contents];
            }

            for(nn::capsrv::AlbumFileContentsType contents = 0; contents < nn::capsrv::AlbumUsage::ContentsCount; contents++)
            {
                if(targetContents < nn::capsrv::AlbumFileContentsCount && contents == targetContents)
                {
                    const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetContentsUsage(targetContents);
                    EXPECT_EQ(createdFileCountList[s][targetContents], pContentsUsage->count);
                    EXPECT_EQ(createdFileSizeList[s][targetContents], pContentsUsage->size);
                    EXPECT_EQ(targetContents, pContentsUsage->contents);
                    EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
                    EXPECT_FALSE(pContentsUsage->IsUnknownContents());
                }
                else
                {
                    const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetContentsUsage(contents);
                    EXPECT_EQ(0, pContentsUsage->count);
                    EXPECT_EQ(0, pContentsUsage->size);
                    EXPECT_EQ(0, pContentsUsage->contents);
                    EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
                    EXPECT_FALSE(pContentsUsage->IsUnknownContents());
                }
            }
            {
                const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetUnknownUsage();
                EXPECT_EQ(expectedUnknownUsage.count, pContentsUsage->count);
                EXPECT_EQ(expectedUnknownUsage.size, pContentsUsage->size);
                EXPECT_EQ(expectedUnknownUsage.contents, pContentsUsage->contents);
                EXPECT_EQ(expectedUnknownUsage.HasGreaterUsage(), pContentsUsage->HasGreaterUsage());
                EXPECT_EQ(expectedUnknownUsage.IsUnknownContents(), pContentsUsage->IsUnknownContents());
            }
        }
    };

    NN_LOG("checking usage(screenshot)\n");
    checkLimitedContentsUsage(nn::capsrv::AlbumFileContents_ScreenShot);
    NN_LOG("checking usage(movie)\n");
    checkLimitedContentsUsage(nn::capsrv::AlbumFileContents_Movie);
    NN_LOG("checking usage(none)\n");
    checkLimitedContentsUsage(nn::capsrv::AlbumFileContentsCount);

    // 旧 API の互換性確認
    NN_LOG("checking compatibility\n");
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        nn::capsrv::AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set(nn::capsrv::AlbumFileContents_ScreenShot);
        nn::capsrv::AlbumUsage uOld = {};
        nn::capsrv::AlbumUsage uNew = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumUsage(&uOld, s));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumUsage(&uNew, s, contentsMask));
        EXPECT_EQ(0, std::memcmp(&uNew, &uOld, sizeof(nn::capsrv::AlbumUsage)));
    }

    // SD のみテスト
    NN_LOG("checking unknown file/directory\n");
    if(nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        // 不明なファイルの作成
        const char basepath[] = "tSD:/2016/01/01/";
        nnt::capsrv::DirectAlbumAccessor::CreateDirectories(basepath);

        nn::Result createFileResult = nn::fs::CreateFile("tSD:/2016/01/01/unknown_file.dat", 12345);
        EXPECT_TRUE(createFileResult.IsSuccess());

        {
            nn::capsrv::AlbumUsage u = {};
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumUsage(&u, nn::capsrv::AlbumStorage_Sd, contentsFlagAll));

            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetUnknownUsage();
            EXPECT_EQ(1, pContentsUsage->count);
            EXPECT_EQ(12345, pContentsUsage->size);
            EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
            EXPECT_TRUE(pContentsUsage->IsUnknownContents());
        }

        // 不明なサブディレクトリの作成
        nn::Result createDirResult = nn::fs::CreateDirectory("tSD:/2016/01/01/unknown_dir");
        EXPECT_TRUE(createDirResult.IsSuccess());

        {
            nn::capsrv::AlbumUsage u = {};
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumUsage(&u, nn::capsrv::AlbumStorage_Sd, contentsFlagAll));

            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetUnknownUsage();
            EXPECT_EQ(1, pContentsUsage->count);
            EXPECT_EQ(12345, pContentsUsage->size);
            EXPECT_TRUE(pContentsUsage->HasGreaterUsage());
            EXPECT_TRUE(pContentsUsage->IsUnknownContents());
        }
    }


    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, GetAlbumUsage_ManyFiles)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // ファイル作成
    int createdFileCountList[nn::capsrv::AlbumStorageCount][nn::capsrv::AlbumFileContentsCount] = {};
    uint64_t createdFileSizeList[nn::capsrv::AlbumStorageCount][nn::capsrv::AlbumFileContentsCount] = {};
    static const int FileCount = 200;

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE_I(s, storage)
    {
        // Create many files to stress the end of the entry buffer
        for (int fidx = 0; fidx < FileCount; fidx++)
        {
            nnt::capsrv::FileInfo finfo =
                NNT_CAPSRV_FILEINFO(NA, 2031, 12, 31, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, 1024, 12345);

            // Override file info members with what we need
            finfo.storage = storage;
            finfo.time.id = static_cast<uint8_t> (fidx % 100);
            finfo.time.minute = static_cast<uint8_t> ((fidx / 100) % 60);

            EXPECT_TRUE(finfo.Create());
            createdFileCountList[finfo.storage][finfo.GetAlbumFileId().contents]++;
            createdFileSizeList[finfo.storage][finfo.GetAlbumFileId().contents] += finfo.filesize;
        }
    }

    if (nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        // Create many folders stress the beginning of the entry buffer
        for (int year = 2020; year < 2032; year++)
        {
            char path[256];
            int yearLength = 0;
            yearLength = nn::util::SNPrintf(path, sizeof(path), "%s%04d", nnt::capsrv::DirectAlbumAccessor::GetRootPath(NNT_CAPSRV_STORAGE_VALUES_SD), year);
            nn::Result result = nn::fs::CreateDirectory(path);
            NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));

            for (int month = 0; month < 14; month++)
            {
                int monthLength = yearLength + nn::util::SNPrintf(path + yearLength, sizeof(path) - static_cast<size_t> (yearLength), "/%02d", month);
                result = nn::fs::CreateDirectory(path);
                NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));
                for (int day = 0; day < 35; day++)
                {
                    // Only handle corner cases (else, the test would take too long)
                    if (!(
                        year == 2020 || year == 2031 ||
                        month == 0 || month == 13 ||
                        day == 0 || day == 34
                        ))
                    {
                        continue;
                    }

                    nn::util::SNPrintf(path + monthLength, sizeof(path) - static_cast<size_t> (monthLength), "/%02d", day);
                    result = nn::fs::CreateDirectory(path);
                    NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));
                }
            }
        }

        const char BadFolder1[] = "2016/01/bad/test";
        const char BadFolder2[] = "2016/bad/01/test";
        const char BadFolder3[] = "bad/01/01/test";
        const char* (BadFolders[]) = { BadFolder1, BadFolder2, BadFolder3 };
        for (int badFolderIdx = 0; badFolderIdx < 3; badFolderIdx++)
        {
            char path[256];
            nn::util::SNPrintf(path, sizeof(path), "%s%s/test", nnt::capsrv::DirectAlbumAccessor::GetRootPath(NNT_CAPSRV_STORAGE_VALUES_SD), BadFolders[badFolderIdx]);
            nnt::capsrv::DirectAlbumAccessor::CreateParentDirectories(path);
        }
    }

    // 再び使用量を確認
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE_I(s, storage)
    {
        nn::capsrv::AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set();

        nn::capsrv::AlbumUsage u = {};
        nn::Result getAlbumUsageResult = nn::capsrv::GetAlbumUsage(&u, storage, contentsMask);
        EXPECT_TRUE(getAlbumUsageResult.IsSuccess());
        NNT_CAPSRV_FOREACH_CONTENTS(contents)
        {
            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetContentsUsage(contents);
            EXPECT_EQ(createdFileCountList[s][contents], pContentsUsage->count);
            EXPECT_EQ(createdFileSizeList[s][contents], pContentsUsage->size);
            EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
            EXPECT_FALSE(pContentsUsage->IsUnknownContents());
            EXPECT_EQ(contents, pContentsUsage->contents);
        }
        {
            const nn::capsrv::AlbumContentsUsage* pContentsUsage = u.GetUnknownUsage();
            EXPECT_EQ(0, pContentsUsage->count);
            EXPECT_EQ(0, pContentsUsage->size);
            if(s == nn::capsrv::AlbumStorage_Sd)
            {
                EXPECT_TRUE(pContentsUsage->HasGreaterUsage());
            }
            else
            {
                EXPECT_FALSE(pContentsUsage->HasGreaterUsage());
            }
            EXPECT_TRUE(pContentsUsage->IsUnknownContents());
        }
    }

    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}

TEST(AlbumAccessApi, GetAlbumUsage_FileSizeLimit)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_ScreenShot    , 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_ScreenShot + 1, 12345),
    };
    static const int TotalFileCount = sizeof(FileList) / sizeof(FileList[0]);

    // ファイル作成
    int createdFileCount = 0;
    for(int i = 0; i < TotalFileCount; i++)
    {
        if(FileList[i].Create())
        {
            createdFileCount++;
        }
    }

    NN_LOG("Check file size limitation for GetAlbumUsage()\n");
    {
        nn::capsrv::AlbumUsage usage;
        EXPECT_TRUE(nn::capsrv::GetAlbumUsage(&usage, nn::capsrv::AlbumStorage_Nand).IsSuccess());
        {
            auto& u = *usage.GetContentsUsage(nn::capsrv::AlbumFileContents_ScreenShot);
            EXPECT_EQ(1, u.count);
            EXPECT_EQ(FileList[0].filesize, u.size);
        }
        {
            auto& u = *usage.GetUnknownUsage();
            EXPECT_EQ(1, u.count);
            EXPECT_EQ(FileList[1].filesize, u.size);
        }
    }


    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}// NOLINT(impl/function_size)
