﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/capsrv/movie/capsrv_MovieMetaData.h>
#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "../../Common/testCapsrv_MovieFileCreator.h"
#include "testCapsrv_StartupTestCase.h"

static int RepeatCount = 100;

#define NNT_CAPSRV_FILELIST(varList, varCount)  \
    const nnt::capsrv::FileInfo varList[] = {  \
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .mp4, 10000000, 12345),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .mp4, 10000000, 12346),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 05, 03, 18, 31, 58, 02, 0xBBBBBBBBBBBBBBBB, .mp4, 10000000, 12347),   \
        NNT_CAPSRV_FILEINFO(SD, 2016, 07, 04, 15, 20, 57, 03, 0xCCCCCCCCCCCCCCCC, .mp4, 10000000, 12348),   \
        NNT_CAPSRV_FILEINFO(SD, 2016, 08, 06, 14, 10, 56, 04, 0xDDDDDDDDDDDDDDDD, .mp4, 10000000, 12349),   \
    };  \
    static const int varCount = sizeof(varList) / sizeof(varList[0]);   \
    NN_UNUSED(varCount);    \
    NN_UNUSED(varList);

//namespace {
//
//    std::vector<std::vector<nn::capsrv::AlbumEntry>> SetupTestMovieFiles() NN_NOEXCEPT
//    {
//        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
//
//        EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
//
//        NNT_CAPSRV_FILELIST(fileList, fileCount);
//        // ファイル作成
//        for(int i = 0; i < fileCount; i++){ fileList[i].Create(); }
//
//        std::vector<std::vector<nn::capsrv::AlbumEntry>> entryListList;
//        entryListList.resize(nn::capsrv::AlbumStorageCount);
//
//        // ファイルリスト取得
//        NNT_CAPSRV_FOREACH_STORAGE_I(s, storage)
//        {
//            auto& list = entryListList[s];
//            list = nnt::capsrv::GetAlbumEntryList(storage);
//            nnt::capsrv::SortAlbumEntryListStorageTime(list);
//        }
//        nn::capsrv::FinalizeAlbumAccess();
//
//        return entryListList;
//    }
//
//}


TEST(AlbumAccessApi, AlbumMovieWriteStream_StateChange)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    std::mt19937 rand(0x111111);
    auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, 0, rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);

    NN_LOG("Open -> ... -> Commit\n");
    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("t=%d:\n", i);
        //NN_LOG("  rep%d\n", i);
        fileId.applicationId.value++;
        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NN_LOG("  Open() -> Empty\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        {
            //NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::FinishAlbumMovieWriteStream(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::CommitAlbumMovieWriteStream(h));
        }
        NN_LOG("  StartDataSection() -> WritingData\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumAlreadyOpened, nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            //NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::FinishAlbumMovieWriteStream(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::CommitAlbumMovieWriteStream(h));
        }
        NN_LOG("  EndDataSection() -> DataComplete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumAlreadyOpened, nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
            //NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::FinishAlbumMovieWriteStream(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::CommitAlbumMovieWriteStream(h));
        }
        NN_LOG("  StartMetaSection() -> WritingMeta\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumAlreadyOpened, nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
            //NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::FinishAlbumMovieWriteStream(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::CommitAlbumMovieWriteStream(h));
        }

        NN_LOG("  Writing Meta\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize));

        NN_LOG("  EndMetaSection() -> MetaComplete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumAlreadyOpened, nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
            //NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::FinishAlbumMovieWriteStream(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::CommitAlbumMovieWriteStream(h));
        }
        NN_LOG("  Finish() -> Complete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumAlreadyOpened, nn::capsrv::FinishAlbumMovieWriteStream(h));
            //NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::CommitAlbumMovieWriteStream(h));
        }
        NN_LOG("  Commit() -> Complete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CommitAlbumMovieWriteStream(h));
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::FinishAlbumMovieWriteStream(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::CommitAlbumMovieWriteStream(h));
        }

        // ファイルが存在する == ファイルサイズが取れる
        size_t size = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumFileSize(&size, &fileId));
    }

    // キャッシュを確認
    {
        nn::capsrv::AlbumCacheData cache = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot));
        EXPECT_EQ(0, cache.fileCount);
    }
    {
        nn::capsrv::AlbumCacheData cache = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_Movie));
        EXPECT_EQ(static_cast<int64_t>(RepeatCount), cache.fileCount);
    }
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, AlbumMovieWriteStream_Discard)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    std::mt19937 rand(0x222222);
    auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, 0, rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);

    NN_LOG("Open -> Discard\n");
    for(int i = 0; i < RepeatCount; i++)
    {
        const int StageCount = 6;
        fileId.applicationId.value++;
        int stage = 0;
        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NN_LOG("  Open() -> Empty\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;
        NN_LOG("  StartDataSection() -> WritingData\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;
        NN_LOG("  EndDataSection() -> DataComplete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;
        NN_LOG("  StartMetaSection() -> WritingMeta\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;

        NN_LOG("  Writing Meta\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize));

        NN_LOG("  EndMetaSection() -> MetaComplete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;
        NN_LOG("  Finish() -> Complete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
        //if(i % StageCount == stage)
        {
            goto DISCARD;
        }


    DISCARD:
        NN_LOG("  Discard()!!!!\n");
        nn::capsrv::DiscardAlbumMovieWriteStream(h);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::DiscardAlbumMovieWriteStreamImpl(h));
        // ファイルが存在しないこと。
        size_t size = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound, nn::capsrv::GetAlbumFileSize(&size, &fileId));
    }

    // キャッシュを確認
    {
        nn::capsrv::AlbumCacheData cache = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot));
        EXPECT_EQ(0, cache.fileCount);
    }
    {
        nn::capsrv::AlbumCacheData cache = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_Movie));
        EXPECT_EQ(0, cache.fileCount);
    }
}

TEST(AlbumAccessApi, AlbumMovieWriteStream_DiscardNoDelete)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    std::mt19937 rand(0x333333);
    auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, 0, rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);

    NN_LOG("Open -> DiscardNoDelete\n");
    for(int i = 0; i < RepeatCount; i++)
    {
        const int StageCount = 6;
        fileId.applicationId.value++;
        int stage = 0;
        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NN_LOG("  Open() -> Empty\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;
        NN_LOG("  StartDataSection() -> WritingData\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;
        NN_LOG("  EndDataSection() -> DataComplete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;
        NN_LOG("  StartMetaSection() -> WritingMeta\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;

        NN_LOG("  Writing Meta\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize));

        NN_LOG("  EndMetaSection() -> MetaComplete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
        if(i % StageCount == stage)
        {
            goto DISCARD;
        }
        stage++;
        NN_LOG("  Finish() -> Complete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
        //if(i % StageCount == stage)
        {
            goto DISCARD;
        }


    DISCARD:
        NN_LOG("  Discard(NoDelete)!!!!\n");
        nn::capsrv::DiscardAlbumMovieWriteStreamNoDelete(h);
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::DiscardAlbumMovieWriteStreamNoDeleteImpl(h));
        //// ファイルが存在しないこと。
        //size_t size = 0;
        //NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound, nn::capsrv::GetAlbumFileSize(&size, &fileId));
    }

    // キャッシュを確認
    {
        nn::capsrv::AlbumCacheData cache = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot));
        EXPECT_EQ(0, cache.fileCount);
    }
    {
        nn::capsrv::AlbumCacheData cache = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_Movie));
        EXPECT_EQ(static_cast<int64_t>(RepeatCount), cache.fileCount);
    }

}

TEST(AlbumAccessApi, AlbumMovieWriteStream_OpenFinalize)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    std::mt19937 rand(0x444444);
    auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, 0, rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);

    NN_LOG("Open -> CloseSession\n");
    for(int i = 0; i < RepeatCount; i++)
    {
        const int StageCount = 6;
        fileId.applicationId.value++;
        int stage = 0;
        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NN_LOG("  Open() -> Empty\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        if(i % StageCount == stage)
        {
            goto CLOSE;
        }
        stage++;
        NN_LOG("  StartDataSection() -> WritingData\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        if(i % StageCount == stage)
        {
            goto CLOSE;
        }
        stage++;
        NN_LOG("  EndDataSection() -> DataComplete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
        if(i % StageCount == stage)
        {
            goto CLOSE;
        }
        stage++;
        NN_LOG("  StartMetaSection() -> WritingMeta\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
        if(i % StageCount == stage)
        {
            goto CLOSE;
        }
        stage++;

        NN_LOG("  Writing Meta\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize));

        NN_LOG("  EndMetaSection() -> MetaComplete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
        if(i % StageCount == stage)
        {
            goto CLOSE;
        }
        stage++;
        NN_LOG("  Finish() -> Complete\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
        //if(i % StageCount == stage)
        {
            goto CLOSE;
        }


    CLOSE:
        NN_LOG("  CloseSession()!!!!\n");
        nn::capsrv::FinalizeAlbumControl();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
        // ファイルが存在しないこと。
        size_t size = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound, nn::capsrv::GetAlbumFileSize(&size, &fileId));
    }

    // キャッシュを確認
    {
        nn::capsrv::AlbumCacheData cache = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot));
        EXPECT_EQ(0, cache.fileCount);
    }
    {
        nn::capsrv::AlbumCacheData cache = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cache, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_Movie));
        EXPECT_EQ(0, cache.fileCount);
    }

}

TEST(AlbumAccessApi, AlbumMovieWriteStream_OpenUnmount)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    std::mt19937 rand(0x444444);
    auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, 0, rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);

    NN_LOG("Open -> Unmount\n");
    for(int i = 0; i < RepeatCount; i++)
    {
        EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
        NNT_CAPSRV_FOREACH_STORAGE(s)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::ResetAlbumMountStatus(s));
        }
        NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
        }

        auto openStreams = [&fileId]()->std::vector<nn::capsrv::AlbumMovieWriteStreamHandle>
        {
            std::vector<nn::capsrv::AlbumMovieWriteStreamHandle> handleList;
            for(;;)
            {
                fileId.applicationId.value++;
                nn::capsrv::AlbumMovieWriteStreamHandle h = {};
                auto result = nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId);
                if(result.IsSuccess())
                {
                    handleList.push_back(h);
                }
                else if(nn::capsrv::ResultAlbumResourceLimit::Includes(result))
                {
                    break;
                }
                else
                {
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                }
            }
            NN_LOG("    %d streams opened\n", handleList.size());
            NN_ASSERT(!handleList.empty());
            return handleList;
        };


        const int StageCount = 6;
        fileId.applicationId.value++;
        int stage = 0;
        NN_LOG("  Open() -> Empty\n");
        auto handles = openStreams();
        if(i % StageCount == stage)
        {
            goto UNMOUNT;
        }
        stage++;
        NN_LOG("  StartDataSection() -> WritingData\n");
        for(auto h : handles)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        }
        if(i % StageCount == stage)
        {
            goto UNMOUNT;
        }
        stage++;
        NN_LOG("  EndDataSection() -> DataComplete\n");
        for(auto h : handles)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
        }
        if(i % StageCount == stage)
        {
            goto UNMOUNT;
        }
        stage++;
        NN_LOG("  StartMetaSection() -> WritingMeta\n");
        for(auto h : handles)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
        }
        if(i % StageCount == stage)
        {
            goto UNMOUNT;
        }
        stage++;

        NN_LOG("  Writing Meta\n");
        for(auto h : handles)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize));
        }

        NN_LOG("  EndMetaSection() -> MetaComplete\n");
        for(auto h : handles)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
        }
        if(i % StageCount == stage)
        {
            goto UNMOUNT;
        }
        stage++;
        NN_LOG("  Finish() -> Complete\n");
        for(auto h : handles)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
        }
        //if(i % StageCount == stage)
        {
            goto UNMOUNT;
        }


    UNMOUNT:
        NN_LOG("  Unmount!\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::NotifyAlbumStorageIsUnavailable(fileId.storage));

        NN_LOG("  Check broken and discard stream\n");
        for(auto& h : handles)
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumIsNotMounted, nn::capsrv::GetAlbumMovieWriteStreamBrokenReason(h));
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::DiscardAlbumMovieWriteStreamImpl(h));
        }
    }
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, AlbumMovieWriteStream_AccessControl)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    for(int i = 0; i < RepeatCount; i++)
    {
        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        NN_UTIL_SCOPE_EXIT{ nn::capsrv::DiscardAlbumMovieWriteStreamImpl(h); };

        // GetFileCount の対象にならない
        {
            int count = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumFileCount(&count, fileId.storage));
            EXPECT_EQ(0, count);
        }

        // GetFileList の対象にならない
        {
            int count = 0;
            nn::capsrv::AlbumEntry e = {};
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumFileList(&count, &e, 1, fileId.storage));
            EXPECT_EQ(0, count);
        }

        // FileSize は取得できない
        {
            size_t size = 0;
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound,
                nn::capsrv::GetAlbumFileSize(&size, &fileId)
            );
        }

        // LoadImage できない
        {
            std::vector<char> buf(4 * 1024 * 1024);
            std::vector<char> work(nn::capsrv::AlbumFileSizeLimit_ScreenShot);

            int width = 0;
            int height = 0;
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData = {};
            auto option = nn::capsrv::ScreenShotDecodeOption::GetDefaultValue();
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound,
                nn::capsrv::LoadAlbumScreenShotImage(&width, &height, &attribute, &appletData, buf.data(), buf.size(), &fileId, option, work.data(), work.size())
            );
        }

        // LoadThumbnail できない
        {
            std::vector<char> buf(4 * 1024 * 1024);
            std::vector<char> work(nn::capsrv::AlbumFileSizeLimit_ScreenShot);

            int width = 0;
            int height = 0;
            nn::capsrv::ScreenShotAttribute attribute = {};
            nn::capsrv::AppletData appletData ={};
            auto option = nn::capsrv::ScreenShotDecodeOption::GetDefaultValue();
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound,
                nn::capsrv::LoadAlbumScreenShotThumbnailImage(&width, &height, &attribute, &appletData, buf.data(), buf.size(), &fileId, option, work.data(), work.size())
            );
        }

        // Delete できない
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound,
                nn::capsrv::DeleteAlbumFile(&fileId)
            );
        }

        // StorageCopy のコピー元に指定できない
        if(nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound,
                nn::capsrv::StorageCopyAlbumFile(&fileId, nn::capsrv::AlbumStorage_Sd)
            );
        }

        //// StorageCopy のコピー先に指定できない
        //if(nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
        //{
        //    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound,
        //        nn::capsrv::StorageCopyAlbumFile(&fileId, nn::capsrv::AlbumStorage_Sd)
        //    );
        //}

        // GetUsage でカウントされない
        {
            nn::capsrv::AlbumUsage usage = {};
            nn::capsrv::AlbumFileContentsFlag mask;
            mask.Set();
            NNT_EXPECT_RESULT_SUCCESS(
                nn::capsrv::GetAlbumUsage(&usage, fileId.storage, mask)
            );
            EXPECT_EQ(0, usage.GetContentsUsage(fileId.contents)->count);
        }

        // OpenReadStream できない
        {
            nn::capsrv::AlbumMovieReadStreamHandle h2 = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound,
                nn::capsrv::OpenAlbumMovieReadStream(&h2, fileId)
            );
        }

        // 多重で OpenWriteStream できない
        {
            nn::capsrv::AlbumMovieWriteStreamHandle h2 = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDestinationAccessCorrupted,
                nn::capsrv::OpenAlbumMovieWriteStream(&h2, fileId)
            );
        }
    }
}// NOLINT(impl/function_size)


TEST(AlbumAccessApi, AlbumMovieWriteStream_Write_Boundary)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);

    int64_t UnitSize = nn::capsrv::AlbumMovieDataUnitSize;
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    nn::capsrv::AlbumMovieWriteStreamHandle h ={};
    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::DiscardAlbumMovieWriteStream(h); };

    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));

    std::vector<uint8_t> data(static_cast<size_t>(nn::capsrv::AlbumMovieDataUnitSize * 2));

    // offset == 0 && size == 0 で書ける
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, 0, nullptr, 0)
    );

    // offset が UnitSize の整数倍で書ける
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize, nullptr, 0)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize * 2, nullptr, 0)
    );

    // 負の offset で書けない
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, -UnitSize, data.data(), static_cast<size_t>(UnitSize))
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, -UnitSize * 2, data.data(), static_cast<size_t>(UnitSize) * 2)
    );
    // ただし size == 0 ならば成功になる
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, -UnitSize, nullptr, 0)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, -UnitSize * 2, nullptr, 0)
    );

    // size が UnitSize の整数倍で書ける
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, 0, data.data(), static_cast<size_t>(UnitSize))
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, 0, data.data(), static_cast<size_t>(UnitSize) * 2)
    );

    // offset が UnitSize の整数倍でないと書けない
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, 1, nullptr, 0)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize + 1, nullptr, 0)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize - 2, nullptr, 0)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, 1, data.data(), static_cast<size_t>(UnitSize))
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize + 1, data.data(), static_cast<size_t>(UnitSize))
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize - 2, data.data(), static_cast<size_t>(UnitSize))
    );

    // size が UnitSize の整数倍でないと書けない
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, 0, data.data(), 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, 0, data.data(), static_cast<size_t>(UnitSize) + 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, 0, data.data(), static_cast<size_t>(UnitSize) - 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize, data.data(), 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize, data.data(), static_cast<size_t>(UnitSize) + 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::WriteDataToAlbumMovieWriteStream(h, UnitSize, data.data(), static_cast<size_t>(UnitSize) - 1)
    );
}

TEST(AlbumAccessApi, AlbumMovieWriteStream_Read_Boundary)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);

    int64_t UnitSize = nn::capsrv::AlbumMovieDataUnitSize;
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    nn::capsrv::AlbumMovieWriteStreamHandle h ={};
    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::DiscardAlbumMovieWriteStream(h); };

    // NOTE:
    // 書いていない領域を読むと 0 埋めになるので先にデータを書かなくても Read できる。
    // 実装上、入力値のチェックの方が先にあるのでとりあえずこれで OK

    size_t readSize = 0;
    std::vector<uint8_t> data(static_cast<size_t>(nn::capsrv::AlbumMovieDataUnitSize * 2));

    // offset == 0 && size == 0 で読める
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), 0, h, 0)
    );

    // offset が UnitSize の整数倍で読める
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), 0, h, UnitSize)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), 0, h, UnitSize * 2)
    );

    // size が UnitSize の整数倍で読める
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), static_cast<size_t>(UnitSize), h, 0)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), static_cast<size_t>(UnitSize) * 2, h, 0)
    );

    // offset が UnitSize の整数倍でないと読めない
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), static_cast<size_t>(UnitSize), h, 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), static_cast<size_t>(UnitSize), h, UnitSize - 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), static_cast<size_t>(UnitSize), h, UnitSize + 1)
    );

    // offset が負で読めない
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), 0, h, -UnitSize)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), 0, h, -UnitSize * 2)
    );

    // size が UnitSize の整数倍でないと読めない
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), 1, h, UnitSize)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), static_cast<size_t>(UnitSize) - 1, h, UnitSize)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, data.data(), static_cast<size_t>(UnitSize) + 1, h, UnitSize)
    );
}

TEST(AlbumAccessApi, AlbumMovieWriteStream_SetSize_Boundary)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NNT_CAPSRV_FILELIST(fileList, fileCount);

    int64_t UnitSize = nn::capsrv::AlbumMovieDataUnitSize;
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    nn::capsrv::AlbumMovieWriteStreamHandle h ={};
    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::DiscardAlbumMovieWriteStream(h); };

    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));

    // size == 0 にできる（ 1 回目）
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, 0)
    );

    // UnitSize の整数倍にできる（ 1 回目）
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, UnitSize)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, UnitSize * 2)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, UnitSize * 3)
    );

    // size == 0 にできる（ 2 回目）
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, 0)
    );

    // size != UnitSize の整数倍は失敗する
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, UnitSize + 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, UnitSize - 2)
    );

    // size が負では失敗する
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, -UnitSize)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, -UnitSize * 2)
    );

    // UnitSize の整数倍にできる（ 2 回目）
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, UnitSize)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, UnitSize * 2)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, UnitSize * 3)
    );

}


TEST(AlbumAccessApi, AlbumMovieWriteStream_Write_WholeFile)
{
    const size_t SizeToWrite = nn::capsrv::AlbumMovieDataUnitSize * 2;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    std::mt19937 rand(0x555555);

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    for(int i = 0; i < fileCount; i++)
    {
        auto fileId = fileList[i].GetAlbumFileId();
        if(!nn::capsrv::IsAlbumMounted(fileId.storage))
        {
            continue;
        }

        auto fileData = fileList[i].GetData();
        // SizeToWrite の整数倍まで切り上げ
        fileData.resize(((fileData.size() + SizeToWrite - 1) / SizeToWrite) * SizeToWrite);

        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        {
            size_t dataSize = fileData.size();
            int64_t remain = static_cast<int64_t>(dataSize);
            int64_t pos = 0;
            while(remain > 0)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteDataToAlbumMovieWriteStream(h, pos, fileData.data() + static_cast<ptrdiff_t>(pos), SizeToWrite));
                remain -= static_cast<int64_t>(SizeToWrite);
                pos    += static_cast<int64_t>(SizeToWrite);
            }
        }
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
        auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, fileData.size(), rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CommitAlbumMovieWriteStream(h));

        //TODO:
        //// 書き込んだ内容を検証
        //size_t writtenDataSize = 0;
        //std::vector<uint8_t> writtenData(fileData.size());
        //NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotFile(&writtenDataSize, writtenData.data(), writtenData.size(), &fileId));
        //EXPECT_TRUE(fileList[i].CheckData(writtenData.data(), fileList[i].filesize));
    }
}

TEST(AlbumAccessApi, AlbumMovieWriteStream_Write_LargeFile)
{
    const size_t SizeToWrite = nn::capsrv::AlbumMovieDataUnitSize * 2;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    std::mt19937 rand(0x666666);

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    for(int i = 0; i < fileCount; i++)
    {
        auto fileId = fileList[i].GetAlbumFileId();
        if(!nn::capsrv::IsAlbumMounted(fileId.storage))
        {
            continue;
        }

        auto fileData = fileList[i].GetData();
        // SizeToWrite の整数倍まで切り上げ
        fileData.resize(((fileData.size() + SizeToWrite - 1) / SizeToWrite) * SizeToWrite);

        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        int n = 10;
        for(int k = 0; k < n; k++)
        {
            size_t dataSize = fileData.size();
            int64_t remain = static_cast<int64_t>(dataSize);
            int64_t pos = 0;
            while(remain > 0)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteDataToAlbumMovieWriteStream(h, pos + dataSize * k, fileData.data() + static_cast<ptrdiff_t>(pos), SizeToWrite));
                remain -= static_cast<int64_t>(SizeToWrite);
                pos    += static_cast<int64_t>(SizeToWrite);
            }
        }
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
        auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, 0, rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CommitAlbumMovieWriteStream(h));

        //TODO:
        //// 書き込んだ内容を検証
        //size_t writtenDataSize = 0;
        //std::vector<uint8_t> writtenData(fileData.size());
        //NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotFile(&writtenDataSize, writtenData.data(), writtenData.size(), &fileId));
        //EXPECT_TRUE(fileList[i].CheckData(writtenData.data(), fileList[i].filesize));
    }
}

TEST(AlbumAccessApi, AlbumMovieWriteStream_Read_WholeFile)
{
    const size_t SizeToWrite = nn::capsrv::AlbumMovieDataUnitSize;
    const size_t SizeToRead  = nn::capsrv::AlbumMovieDataUnitSize * 2;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    std::mt19937 rand(0x777777);

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    for(int i = 0; i < fileCount; i++)
    {
        auto fileId = fileList[i].GetAlbumFileId();
        if(!nn::capsrv::IsAlbumMounted(fileId.storage))
        {
            continue;
        }

        auto fileData = fileList[i].GetData();
        // SizeToWrite の整数倍まで切り上げ
        fileData.resize(((fileData.size() + SizeToWrite - 1) / SizeToWrite) * SizeToWrite);

        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        {
            size_t dataSize = fileData.size();
            int64_t remain = dataSize;
            int64_t pos = 0;
            while(remain > 0)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteDataToAlbumMovieWriteStream(h, pos, fileData.data() + pos, SizeToWrite));
                remain -= static_cast<int64_t>(SizeToWrite);
                pos    += static_cast<int64_t>(SizeToWrite);

                // 適当なところで。
                if(pos >= static_cast<int64_t>(SizeToRead))
                {
                    // 書き込んだ内容が読み込めることをチェック
                    std::vector<uint8_t> readData(SizeToRead);
                    size_t readSize = 0;
                    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, readData.data(), readData.size(), h, pos - SizeToRead));

                    EXPECT_EQ(readSize, SizeToRead);
                    EXPECT_EQ(0, std::memcmp(fileData.data() + static_cast<ptrdiff_t>(pos) - SizeToRead, readData.data(), SizeToRead));
                }
            }
        }
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));
        auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, 0, rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CommitAlbumMovieWriteStream(h));

        //TODO:
        //// 書き込んだ内容を検証
        //size_t writtenDataSize = 0;
        //std::vector<uint8_t> writtenData(fileData.size());
        //NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotFile(&writtenDataSize, writtenData.data(), writtenData.size(), &fileId));
        //EXPECT_TRUE(fileList[i].CheckData(writtenData.data(), fileList[i].filesize));
    }
}


// 正しく不正な呼出を弾けるかの検査
TEST(AlbumAccessApi, AlbumMovieWriteStream_ReadWriteSetSize_ConditionCheck)
{
    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(s));
    }

    nn::ncm::ApplicationId appId = {0x12345612341};

    static const int BlockCount = 4;
    const int64_t UnitSize = nn::capsrv::AlbumMovieDataUnitSize;
    const size_t BufferSize = BlockCount * nn::capsrv::AlbumMovieDataUnitSize;
    std::vector<uint8_t> readBuf(BufferSize);
    std::vector<uint8_t> writeBuf(BufferSize);

    // write buf をランダムに埋める
    {
        std::mt19937 rand(0x11111);
        std::uniform_int_distribution<int> dist;

        for(auto& e : writeBuf)
        {
            e = static_cast<uint8_t>(dist(rand));
        }
    }

    auto checkBufferIsZero = [](const void* buf, size_t size) -> size_t
    {
        size_t badCount = 0;
        const char* p = reinterpret_cast<const char*>(buf);
        for(size_t i = 0; i < size; i++)
        {
            badCount += (p[i] == 0 ? 0 : 1);
        }
        return badCount;
    };

    auto checkBufferIsSame = [](const void* buf0, const void* buf1, size_t size) -> size_t
    {
        size_t badCount = 0;
        const char* p0 = reinterpret_cast<const char*>(buf0);
        const char* p1 = reinterpret_cast<const char*>(buf1);
        for(size_t i = 0; i < size; i++)
        {
            badCount += (p0[i] == p1[i] ? 0 : 1);
        }
        return badCount;
    };

    // 空のファイル
    {
        nn::capsrv::AlbumFileId fileId = {};
        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GenerateCurrentAlbumFileId(&fileId, appId, nn::capsrv::AlbumFileContents_Movie));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        NN_UTIL_SCOPE_EXIT{ NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::DiscardAlbumMovieWriteStreamImpl(h)); };

        {
            // 作成した直後はサイズ 0
            int64_t size = -1;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumMovieWriteStreamDataSize(&size, h));
            EXPECT_EQ(0, size);
        }

        {
            // 範囲外は 0 が読める
            size_t readSize = -1;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, readBuf.data(), BufferSize, h, 0));
            EXPECT_EQ(BufferSize, readSize);
            EXPECT_EQ(0, checkBufferIsZero(readBuf.data(), BufferSize));
        }
    }

    const int TagResizeBase = 0x0100;

    auto markResize = [&](int blockCount) -> int
    {
        return TagResizeBase + blockCount;
    };

    // でたらめな順番で書き込む
    auto checkWritingRandomOrder = [&](const std::vector<int>& orderList) -> void
    {
        nn::capsrv::AlbumFileId fileId = {};
        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GenerateCurrentAlbumFileId(&fileId, appId, nn::capsrv::AlbumFileContents_Movie));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
        NN_UTIL_SCOPE_EXIT{ NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::DiscardAlbumMovieWriteStreamImpl(h)); };

        NN_LOG("  streamId = %lld\n", h);

        bool isBlockWritten[BlockCount] = {};

        for(size_t i = 0; i < orderList.size(); i++)
        {
            int tag = orderList[i];
            if(tag >= TagResizeBase)
            {
                int newBlockCount = tag - TagResizeBase;
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::SetAlbumMovieWriteStreamDataSize(h, newBlockCount * UnitSize));
                for(int i = newBlockCount; i < BlockCount; i++)
                {
                    isBlockWritten[i] = false;
                }
            }
            else
            {
                int targetBlockIndex = tag;
                // 指定のブロックを書込み
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::WriteDataToAlbumMovieWriteStream(h, targetBlockIndex * UnitSize, writeBuf.data() + targetBlockIndex * UnitSize, UnitSize));
                isBlockWritten[targetBlockIndex] = true;
            }

            // 読める値のチェック
            {
                size_t readSize = -1;
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadDataFromAlbumMovieWriteStream(&readSize, readBuf.data(), BufferSize, h, 0));
                EXPECT_EQ(BufferSize, readSize);
                for(int k = 0; k < BlockCount; k++)
                {
                    if(isBlockWritten[k])
                    {
                        EXPECT_EQ(0, checkBufferIsSame(writeBuf.data() + k * UnitSize, readBuf.data() + k * UnitSize, UnitSize));
                    }
                    else
                    {
                        EXPECT_EQ(0, checkBufferIsZero(readBuf.data() + k * UnitSize, UnitSize));
                    }
                }
            }

            // 書込み完了できるかチェック
            // 最後以外では失敗するような入力にすること
            if(i != orderList.size() - 1)
            {
                NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                    nn::capsrv::EndAlbumMovieWriteStreamDataSection(h)
                );
            }
        }

        // EndDataSection が成功するとそれ以上書けないので成功は最後だけチェックする。
        // 必ず最後に成功するような入力にすること。
        NNT_EXPECT_RESULT_SUCCESS(
            nn::capsrv::EndAlbumMovieWriteStreamDataSection(h)
        );
    };

#define NNT_CHECK_WRITING_ORDER( list ) \
    NN_LOG("checking " #list "\n"); \
    checkWritingRandomOrder list ;

    // NOTE:
    //   コマンド：
    //     0 ～ (BlockCount - 1) : そのインデックスブロックに書き込む
    //     markResize(n)         : ブロック数が n になるように SetSize を呼ぶ
    //
    //   コマンドは頭から実行。
    //   各コマンドの終了時に読込と EndDataSection のチェックを行う。
    //
    //   ・コマンドリストの途中では EndDataSection が失敗すること
    //     例えば markResize(0) するとそこで EndDataSection が成功してしまう。
    //   ・コマンドリストの末尾で EndDataSection が成功すること

    // 適当な順番で書いていく。
    NNT_CHECK_WRITING_ORDER(({markResize(4), 0, 1, 2, 3}));
    NNT_CHECK_WRITING_ORDER(({1, 2, 3, 0}));
    NNT_CHECK_WRITING_ORDER(({2, 3, 0, 1}));
    NNT_CHECK_WRITING_ORDER(({3, 0, 1, 2}));
    NNT_CHECK_WRITING_ORDER(({3, 2, 1, 0}));
    NNT_CHECK_WRITING_ORDER(({markResize(4), 2, 1, 0, 3}));
    NNT_CHECK_WRITING_ORDER(({markResize(4), 1, 0, 3, 2}));
    NNT_CHECK_WRITING_ORDER(({markResize(4), 0, 3, 2, 1}));

    // 途中でリサイズ

    // 途中で縮小
    NNT_CHECK_WRITING_ORDER(({1, 2, markResize(2), 3, 0, 2}));
    // 途中で縮小して即座に拡大
    NNT_CHECK_WRITING_ORDER(({1, 2, markResize(1), markResize(4), 3, 0, 1, 2}));
    // 最後に全範囲破棄
    NNT_CHECK_WRITING_ORDER(({3, 2, 1, markResize(0)}));
    // 大きめに作って最後に縮小
    NNT_CHECK_WRITING_ORDER(({markResize(4), 0, 1, 2, markResize(3)}));

#undef NNT_CHECK_WRITING_ORDER

}// NOLINT(impl/function_size)
