﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "../../Common/testCapsrv_MovieFileCreator.h"
#include "testCapsrv_StartupTestCase.h"

static int RepeatCount = 100;

#define NNT_CAPSRV_FILELIST(varList, varCount)  \
    const nnt::capsrv::FileInfo varList[] = {  \
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .mp4, 10000000, 12345),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .mp4, 10000000, 12346),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 05, 03, 18, 31, 58, 02, 0xBBBBBBBBBBBBBBBB, .mp4, 10485760, 12347),   \
        NNT_CAPSRV_FILEINFO(SD, 2016, 07, 04, 15, 20, 57, 03, 0xCCCCCCCCCCCCCCCC, .mp4, 10485759, 12348),   \
        NNT_CAPSRV_FILEINFO(SD, 2016, 08, 06, 14, 10, 56, 04, 0xDDDDDDDDDDDDDDDD, .mp4, 10485761, 12349),   \
    };  \
    static const int varCount = sizeof(varList) / sizeof(varList[0]); \
    NN_UNUSED(varList); \
    NN_UNUSED(varCount);

namespace {

    std::vector<std::vector<nn::capsrv::AlbumEntry>> SetupTestMovieFiles() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());

        EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

        NNT_CAPSRV_FILELIST(fileList, fileCount);
        // ファイル作成
        for(int i = 0; i < fileCount; i++){ fileList[i].CreateMovie(); }

        std::vector<std::vector<nn::capsrv::AlbumEntry>> entryListList;
        entryListList.resize(nn::capsrv::AlbumStorageCount);

        // ファイルリスト取得
        NNT_CAPSRV_FOREACH_STORAGE_I(s, storage)
        {
            auto& list = entryListList[s];
            list = nnt::capsrv::GetAlbumEntryList(storage);
            nnt::capsrv::SortAlbumEntryListStorageTime(list);
        }
        nn::capsrv::FinalizeAlbumAccess();

        return entryListList;
    }

}


TEST(AlbumAccessApi, AlbumMovieReadStream_OpenClose)
{
    nnt::capsrv::StartupTestCase();
    auto albumEntryLists = SetupTestMovieFiles();
    auto fileEntry = albumEntryLists[0][0];
    auto fileId = fileEntry.fileId;

    NN_LOG("Open -> Close\n");
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
        for(int i = 0; i < RepeatCount; i++)
        {
            nn::capsrv::AlbumMovieReadStreamHandle h = {};
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&h, fileId));
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CloseAlbumMovieReadStreamImpl(h));
        }
        nn::capsrv::FinalizeAlbumAccess();
    }
}

TEST(AlbumAccessApi, AlbumMovieReadStream_OpenFinalize)
{
    nnt::capsrv::StartupTestCase();
    auto albumEntryLists = SetupTestMovieFiles();
    auto fileEntry = albumEntryLists[0][0];
    auto fileId = fileEntry.fileId;

    NN_LOG("Open -> Finalize\n");
    {
        for(int i = 0; i < RepeatCount; i++)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
            nn::capsrv::AlbumMovieReadStreamHandle h = {};
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&h, fileId));
            //NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CloseAlbumMovieReadStreamImpl(h));
            nn::capsrv::FinalizeAlbumAccess();
        }
    }
}

TEST(AlbumAccessApi, AlbumMovieReadStream_OpenUnmount)
{
    nnt::capsrv::StartupTestCase();
    auto albumEntryLists = SetupTestMovieFiles();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    NN_LOG("Open -> Unmount\n");
    for(int i = 0; i < RepeatCount; i++)
    {
        NNT_CAPSRV_FOREACH_STORAGE(s)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::ResetAlbumMountStatus(s));
        }

        NN_LOG("  Open() -> Reading\n");
        int openCount = 0;
        std::vector<std::vector<nn::capsrv::AlbumMovieReadStreamHandle>> handles;
        for(auto& v : albumEntryLists)
        {
            std::vector<nn::capsrv::AlbumMovieReadStreamHandle> hs;
            for(auto& e : v)
            {
                nn::capsrv::AlbumMovieReadStreamHandle h = {};
                auto result = nn::capsrv::OpenAlbumMovieReadStream(&h, e.fileId);
                if(result.IsSuccess())
                {
                    hs.push_back(h);
                    openCount++;
                }
                else if(nn::capsrv::ResultAlbumResourceLimit::Includes(result))
                {
                    break;
                }
                else
                {
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                }
            }
            handles.push_back(hs);
        }
        NN_LOG("    %d streams opened\n", openCount);

        NN_LOG("  Unmount storage %d!\n", i % nn::capsrv::AlbumStorageCount);
        bool isUnmounted = false;
        int unmountedStorageIndex = 0;
        NNT_CAPSRV_FOREACH_MOUNTED_STORAGE_I(k, s)
        {
            if(k != i % nn::capsrv::AlbumStorageCount)
            {
                continue;
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::NotifyAlbumStorageIsUnavailable(s));
            isUnmounted = true;
            unmountedStorageIndex = k;
            break;
        }
        if(isUnmounted)
        {
            auto& v = handles[unmountedStorageIndex];
            NN_LOG("    Checking %d streams broken\n", v.size());
            for(auto& h : v)
            {
                int64_t size = 0;
                NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDenied, nn::capsrv::GetAlbumMovieReadStreamDataSize(&size, h));
                NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumIsNotMounted, nn::capsrv::GetAlbumMovieReadStreamBrokenReason(h));
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CloseAlbumMovieReadStreamImpl(h));
                NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::GetAlbumMovieReadStreamBrokenReason(h));
            }
            v.clear();
        }

        NN_LOG("    Checking&Closing all other streams\n");
        // close all
        for(auto& v : handles)
        {
            for(auto& h : v)
            {
                int64_t size = 0;
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumMovieReadStreamDataSize(&size, h));
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CloseAlbumMovieReadStreamImpl(h));
                NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumNotFound, nn::capsrv::GetAlbumMovieReadStreamBrokenReason(h));
            }
        }
    }
}

TEST(AlbumAccessApi, AlbumMovieReadStream_AccessControl)
{
    nnt::capsrv::StartupTestCase();
    auto albumEntryLists = SetupTestMovieFiles();

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    auto fileId = fileList[0].GetAlbumFileId();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    for(int i = 0; i < RepeatCount; i++)
    {
        nn::capsrv::AlbumMovieReadStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&h, fileId));
        NN_UTIL_SCOPE_EXIT{ nn::capsrv::CloseAlbumMovieReadStreamImpl(h); };

        // 削除できない
        {
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDestinationAccessCorrupted,
                nn::capsrv::DeleteAlbumFile(&fileId)
            );
        }

        // StorageCoy のコピー先に指定できない
        // TODO:

        // OpenWriteStream できない
        {
            nn::capsrv::AlbumMovieWriteStreamHandle h2 = {};
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumDestinationAccessCorrupted,
                nn::capsrv::OpenAlbumMovieWriteStream(&h2, fileId)
            );
        }
    }
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, AlbumMovieReadStream_ReadMovieData_Boundary)
{
    int64_t UnitSize = nn::capsrv::AlbumMovieDataUnitSize;

    nnt::capsrv::StartupTestCase();
    auto entryListList = SetupTestMovieFiles();


    NNT_CAPSRV_FILELIST(fileList, fileCount);

    auto fileId = entryListList[0][0].fileId;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

    nn::capsrv::AlbumMovieReadStreamHandle h ={};
    NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&h, fileId));
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::CloseAlbumMovieReadStream(h); };

    size_t readSize = 0;
    std::vector<uint8_t> data(static_cast<size_t>(nn::capsrv::AlbumMovieDataUnitSize * 2));

    // offset == 0 && size == 0 で読める
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, nullptr, 0, h, 0)
    );

    // offset が UnitSize の整数倍で読める
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, nullptr, 0, h, UnitSize)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, nullptr, 0, h, UnitSize * 2)
    );

    // size が UnitSize の整数倍で読める
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize), h, 0)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize) * 2, h, 0)
    );

    // offset が UnitSize の整数倍でないと読めない
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, nullptr, 0, h, 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, nullptr, 0, h, UnitSize - 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, nullptr, 0, h, UnitSize + 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize), h, 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize), h, UnitSize - 1)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize), h, UnitSize + 1)
    );

    // size が UnitSize の整数倍でないと読めない
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), 1           , h, 0)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize) - 1, h, 0)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize) + 1, h, 0)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), 1           , h, UnitSize)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize) - 1, h, UnitSize)
    );
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumOutOfRange,
        nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data(), static_cast<size_t>(UnitSize) + 1, h, UnitSize)
    );



}

TEST(AlbumAccessApi, AlbumMovieReadStream_ReadMovieData_WholeFile)
{
    nnt::capsrv::StartupTestCase();
    SetupTestMovieFiles();

    NNT_CAPSRV_FILELIST(fileList, fileCount);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());

    const size_t SizeToReadMax = nn::capsrv::AlbumMovieDataUnitSize * 2;

    for(int i = 0; i < fileCount; i++)
    {
        NN_LOG("FileList[%d]\n", i);
        auto fileId = fileList[i].GetAlbumFileId();
        if(!nn::capsrv::IsAlbumMounted(fileId.storage))
        {
            continue;
        }

        nn::capsrv::AlbumMovieReadStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&h, fileId));

        int64_t dataSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumMovieReadStreamDataSize(&dataSize, h));
        EXPECT_EQ(fileList[i].filesize, dataSize); // 署名対応したら署名の分 dataSize は小さくなる（？）
        NN_LOG("  dataSize = %llu\n", dataSize);

        std::vector<char> data;
        data.resize(static_cast<size_t>(dataSize + SizeToReadMax));
        EXPECT_EQ(data.size(), static_cast<size_t>(dataSize + SizeToReadMax));

        int64_t pos = 0;
        int64_t remain = dataSize;
        while(remain > 0)
        {
            int64_t remainingBlockCount = (remain + nn::capsrv::AlbumMovieDataUnitSize - 1) / nn::capsrv::AlbumMovieDataUnitSize;
            size_t sizeToRead = std::min(SizeToReadMax, static_cast<size_t>(remainingBlockCount* nn::capsrv::AlbumMovieDataUnitSize));
            NN_LOG("  reading [%llu-%llu)\n", pos, pos + sizeToRead);
            size_t readSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadDataFromAlbumMovieReadStream(&readSize, data.data() + pos, sizeToRead, h, pos));
            EXPECT_EQ(sizeToRead, readSize);
            if(sizeToRead != readSize)
            {
                NN_LOG("  skiping remaining data\n");
                break;
            }
            pos    += static_cast<int64_t>(readSize);
            remain -= static_cast<int64_t>(readSize);
        }

        EXPECT_TRUE(fileList[i].CheckData(data.data(), dataSize));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CloseAlbumMovieReadStreamImpl(h));

    }

    nn::capsrv::FinalizeAlbumAccess();
}


TEST(AlbumAccessApi, AlbumMovieReadStream_ReadImageData_Basic)
{
    nnt::capsrv::StartupTestCase();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    int repeatCount = 10;

    std::mt19937 rand(0x98762);

    std::uniform_int_distribution<size_t> movieDataSizeDist(2 * 1024 * 1024, 4 * 1024 * 1024);
    std::uniform_int_distribution<size_t> imageDataSizeDist(100 * 1024, 500 * 1024);
    std::uniform_int_distribution<size_t> readingImageUnitSizeDist(4 * 1024, 20 * 1024);

    for(int t = 0; t < repeatCount; t++)
    {
        NN_LOG("t = %d\n", t);

        NN_LOG("  creating file\n");
        // ファイル ID 作成
        nn::capsrv::AlbumFileId fileId = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GenerateCurrentAlbumFileId(&fileId, {0x12345678}, nn::capsrv::AlbumFileContents_Movie));

        // データ作成
        size_t movieDataSize = movieDataSizeDist(rand);
        NN_LOG("    movieDataSize = %llu\n", movieDataSize);
        auto movieData = nnt::capsrv::MovieFileCreator::CreateRandomMovieData(movieDataSize, rand);
        auto metaData = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, movieDataSize, rand);
        auto dataHash = nnt::capsrv::MovieFileCreator::CreateValidMovieDataHash(movieData);
        auto metaHash = nnt::capsrv::MovieFileCreator::CreateValidMovieMetaHash(metaData);
        auto signature = nnt::capsrv::MovieFileCreator::CreateValidMovieSignature(dataHash, metaHash);
        auto version = nnt::capsrv::MovieFileCreator::CreateValidMovieVersion(nn::capsrv::movie::MovieMetaDataVersion_1);

        // ファイル作成
        {
            auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
            nn::fs::FileHandle hFile = {};
            // 念の為削除
            nn::fs::DeleteFile(path.c_str());
            nnt::capsrv::DirectAlbumAccessor::CreateFile(0, path.c_str());
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&hFile, path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

            std::vector<char> fileData;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::capsrv::MovieFileCreator::WriteFileData(
                [&](int64_t offset, const void* buf, size_t size)->nn::Result
                {
                    return nn::fs::WriteFile(hFile, offset, buf, size, nn::fs::WriteOption::MakeValue(0));
                },
                movieData,
                metaData,
                dataHash,
                metaHash,
                signature,
                version
            ));

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::FlushFile(hFile));
            nn::fs::CloseFile(hFile);
        }

        NN_LOG("  checking image data read\n");

        // ReadStream を Open
        nn::capsrv::AlbumMovieReadStreamHandle hStream = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&hStream, fileId));

        // 念の為 MovieSize を取得
        int64_t readMovieSize = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumMovieReadStreamDataSize(&readMovieSize, hStream));
        EXPECT_EQ(static_cast<int64_t>(movieDataSize), readMovieSize);

        // ImageSize を取得
        int64_t readImageSize = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GetAlbumMovieReadStreamImageDataSize(&readImageSize, hStream));
        EXPECT_EQ(metaData.pMeta->infoData.imageDataSize, readImageSize);

        // 読込
        std::vector<uint8_t> readImageData(static_cast<size_t>(readImageSize));
        {
            size_t unitSize = readingImageUnitSizeDist(rand);
            size_t remain = readImageData.size();
            size_t offset = 0;
            while(remain > 0)
            {
                size_t sizeToRead = std::min(unitSize, remain);
                size_t readSize = 0;
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadImageDataFromAlbumMovieReadStream(&readSize, readImageData.data() + offset, sizeToRead, hStream, static_cast<int64_t>(offset)));
                EXPECT_EQ(readSize, sizeToRead);

                remain -= sizeToRead;
                offset += sizeToRead;
            }
        }

        // 読込んだ内容の検査
        EXPECT_EQ(0, std::memcmp(metaData.value.data(), readImageData.data(), readImageSize));

        // 範囲外は 0 が読めることの確認（その1）
        {
            std::vector<uint8_t> buffer(100 * 1024, 0xFF);
            for(int i = 0; i < 10; i++)
            {
                size_t sizeToRead = buffer.size();
                size_t readSize = 0;
                int64_t offset = readImageSize + static_cast<int64_t>(buffer.size()) * i;
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadImageDataFromAlbumMovieReadStream(&readSize, buffer.data(), sizeToRead, hStream, offset));
                EXPECT_EQ(readSize, sizeToRead);

                int64_t badCount = 0;
                for(size_t pos = 0; pos < buffer.size(); pos++)
                {
                    badCount += (buffer[pos] == 0 ? 0 : 1);
                }
                EXPECT_EQ(0, badCount);
            }
        }

        // 範囲外は 0 が読めることの確認（その2）
        {
            std::vector<uint8_t> buffer(256 * 1024, 0xFF);
            int64_t overlap = 10 * 1024;
            size_t sizeToRead = buffer.size();
            size_t readSize = 0;
            int64_t offset = readImageSize - overlap;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadImageDataFromAlbumMovieReadStream(&readSize, buffer.data(), sizeToRead, hStream, offset));
            EXPECT_EQ(readSize, sizeToRead);

            int64_t badCount = 0;
            for(size_t pos = overlap; pos < buffer.size(); pos++)
            {
                badCount += (buffer[pos] == 0 ? 0 : 1);
            }
            EXPECT_EQ(0, badCount);
        }

        nn::capsrv::CloseAlbumMovieReadStream(hStream);
    }

}// NOLINT(impl/function_size)
