﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <vector>
#include <list>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/capsrv/movie/capsrv_CachedMovieStream.h>
#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "testCapsrv_StartupTestCase.h"

#include "detail/testCapsrv_TestMovieStreamAccessor.h"
#include "detail/testCapsrv_TestMovieStreamCacheStrategy.h"
#include "detail/testCapsrv_TestMovieStreamWrapperBase.h"
#include "detail/testCapsrv_TestMovieStreamRandomAccessImpl.h"

//static int RepeatCount = 100;

#define NNT_CAPSRV_IO_LOG(...) NN_LOG(__VA_ARGS__)

#ifndef NNT_CAPSRV_IO_LOG
#define NNT_CAPSRV_IO_LOG(...)
#endif

#define NNT_CAPSRV_FILELIST(varList, varCount)  \
    const nnt::capsrv::FileInfo varList[] = {  \
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .mp4, 10000000, 12345),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .mp4, 10000000, 12346),   \
    };  \
    static const int varCount = sizeof(varList) / sizeof(varList[0]);   \
    NN_UNUSED(varCount);    \
    NN_UNUSED(varList);

namespace {

    class TestStream
        : public nnt::capsrv::detail::TestMovieStreamWrapperBase
    {
    public:
        void Initialize(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT
        {
            m_Work.resize(nn::capsrv::movie::CachedMovieStream::GetRequiredWorkMemorySize(cacheChunkSize, cacheChunkCount));
            m_Stream.Initialize(cacheChunkSize, cacheChunkCount, m_Accessor.GetAccessor(), m_Strategy.GetStrategy(), m_Work.data(), m_Work.size(), 0);
            m_ChunkSize = cacheChunkSize;
            m_DataSize = 0;
        }

        void Finalize() NN_NOEXCEPT
        {
            m_Stream.Finalize();
        }

    protected:

        virtual nn::Result WriteImpl(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            if(size > 0)
            {
                m_DataSize = std::max(m_DataSize, offset + static_cast<int64_t>(size));
            }
            return m_Stream.Write(offset, buffer, size);
        }

        virtual nn::Result ReadImpl(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Stream.Read(pOutReadSize, buffer, size, offset);
        }

        virtual nn::Result GetSizeImpl(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *pOutValue = m_DataSize;
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result SetSizeImpl(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            m_DataSize = size;
            return m_Stream.Resize(size);
        }

        virtual nn::Result FlushImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Stream.Flush();
        }

        virtual const uint8_t* GetInternalStorageImpl(size_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
        {
            EXPECT_GE(m_Accessor.data.size(), static_cast<size_t>(m_DataSize));
            NN_ASSERT(m_Accessor.data.size() >= static_cast<size_t>(m_DataSize));
            *pOutSize = static_cast<size_t>(m_DataSize);
            return reinterpret_cast<const uint8_t*>(m_Accessor.data.data());
        }


    public:
        int64_t m_DataSize;
        int64_t m_ChunkSize;
        nn::capsrv::movie::CachedMovieStream m_Stream;
        nnt::capsrv::detail::TestMovieStreamAccessor m_Accessor;
        nnt::capsrv::detail::TestMovieStreamCacheStrategy m_Strategy;
        std::vector<char> m_Work;
    };
}

namespace {

    void TestAlbumCachedMovieStreamWriterImpl(
        size_t sizeToWriteMax,
        size_t sizeToReadMax,
        int64_t chunkSize,
        int64_t cacheCount
    )
    {
        NN_LOG("sizeToWriteMax  = %llu\n", sizeToWriteMax);
        NN_LOG("sizeToReadMax   = %llu\n", sizeToReadMax);
        NN_LOG("cacheChunkSize  = %llu\n", chunkSize);
        NN_LOG("cacheChunkCount = %llu\n", cacheCount);

        NNT_CAPSRV_FILELIST(fileList, fileCount);
        for(int i = 0; i < fileCount; i++)
        {
            auto fileId = fileList[i].GetAlbumFileId();
            if(!nn::capsrv::IsAlbumMounted(fileId.storage))
            {
                continue;
            }

            auto fileData = fileList[i].GetData();

            TestStream stream;
            stream.Initialize(chunkSize, cacheCount);
            NN_UTIL_SCOPE_EXIT{ stream.Finalize(); };

            {

                size_t dataSize = fileData.size();
                size_t remain = dataSize;
                int64_t pos = 0;
                while(remain > 0)
                {
                    size_t sizeToWrite = std::min(remain, sizeToWriteMax);
                    NNT_EXPECT_RESULT_SUCCESS(stream.Write(pos, fileData.data() + pos, sizeToWrite));
                    remain -= sizeToWrite;
                    pos += sizeToWrite;

                    // 適当なところで。
                    if(pos >= static_cast<int64_t>(sizeToReadMax))
                    {
                        // 書き込んだ内容が読み込めることをチェック
                        std::vector<uint8_t> readData(sizeToReadMax);
                        size_t readSize = 0;
                        NNT_EXPECT_RESULT_SUCCESS(stream.Read(&readSize, readData.data(), readData.size(), pos - static_cast<int64_t>(sizeToReadMax)));

                        EXPECT_EQ(readSize, sizeToReadMax);
                        EXPECT_EQ(0, std::memcmp(fileData.data() + pos - sizeToReadMax, readData.data(), sizeToReadMax));
                    }
                }
            }

            stream.Flush();
            // 書き込んだ内容を検証
            stream.CheckStorage();
        }
    }

}



TEST(AlbumAccessApi, AlbumCachedMovieStream_Writer)
{
    const size_t SizeToWriteMax = 4 * 1024;
    const size_t SizeToReadMax = 64 * 1024;
    const int64_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    const int64_t CacheCount = 4;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    TestAlbumCachedMovieStreamWriterImpl(SizeToWriteMax, SizeToReadMax, ChunkSize, CacheCount);

    TestAlbumCachedMovieStreamWriterImpl(       200, SizeToReadMax, ChunkSize, CacheCount);
    TestAlbumCachedMovieStreamWriterImpl(250 * 1024, SizeToReadMax, ChunkSize, CacheCount);

    TestAlbumCachedMovieStreamWriterImpl(SizeToWriteMax,        200, ChunkSize, CacheCount);
    TestAlbumCachedMovieStreamWriterImpl(SizeToWriteMax, 250 * 1024, ChunkSize, CacheCount);

    TestAlbumCachedMovieStreamWriterImpl(SizeToWriteMax, SizeToReadMax, 2 * ChunkSize, CacheCount);
    TestAlbumCachedMovieStreamWriterImpl(SizeToWriteMax, SizeToReadMax, 3 * ChunkSize, CacheCount);

    TestAlbumCachedMovieStreamWriterImpl(SizeToWriteMax, SizeToReadMax, ChunkSize, 1);
    TestAlbumCachedMovieStreamWriterImpl(SizeToWriteMax, SizeToReadMax, ChunkSize, 2);
}


TEST(AlbumAccessApi, AlbumCachedMovieStream_Resize)
{
    int64_t cacheCount = 4;
    int64_t chunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    std::mt19937 rand(0x1234);

    // 変化なし(境界)
    {
        NN_LOG("Resize:unchanged:bb\n");
        int64_t size0 = chunkSize * 2;
        int64_t size1 = chunkSize * 2;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.CheckRead(0, size1);
        stream.Flush();
        stream.CheckStorage();
    }
    // 変化なし(中間)
    {
        NN_LOG("Resize:unchanged:ii\n");
        int64_t size0 = chunkSize * 2 + chunkSize / 2;
        int64_t size1 = chunkSize * 2 + chunkSize / 2;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.CheckRead(0, size1);
        stream.Flush();
        stream.CheckStorage();
    }
    //---------------------------------------------------------
    // 拡大(チャンク内. 境界->中間)
    {
        NN_LOG("Resize:expand(in-chunk):bi\n");
        int64_t size0 = chunkSize * 2;
        int64_t size1 = chunkSize * 2 + chunkSize / 2;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.CheckRead(0, size1);
        stream.Flush();
        stream.CheckStorage();
    }
    // 拡大(チャンク内. 中間->中間)
    {
        NN_LOG("Resize:expand(in-chunk):ii\n");
        int64_t size0 = chunkSize * 2 + chunkSize / 3;
        int64_t size1 = chunkSize * 2 + chunkSize / 2;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.CheckRead(0, size1);
        stream.Flush();
        stream.CheckStorage();
    }
    // 拡大(チャンク外. 境界->境界)
    {
        NN_LOG("Resize:expand(out-chunk):bb\n");
        int64_t size0 = chunkSize * cacheCount;
        int64_t size1 = chunkSize * cacheCount * 3;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.CheckRead(0, size1);
        stream.Flush();
        stream.CheckStorage();
    }
    // 拡大(チャンク外. 境界->中間)
    {
        NN_LOG("Resize:expand(out-chunk):bi\n");
        int64_t size0 = chunkSize * cacheCount;
        int64_t size1 = chunkSize * cacheCount * 3 + chunkSize / 2;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.CheckRead(0, size1);
        stream.Flush();
        stream.CheckStorage();
    }
    // 拡大(チャンク外. 中間->境界)
    {
        NN_LOG("Resize:expand(out-chunk):ib\n");
        int64_t size0 = chunkSize * cacheCount + chunkSize / 2;
        int64_t size1 = chunkSize * cacheCount * 3;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.CheckRead(0, size1);
        stream.Flush();
        stream.CheckStorage();
    }
    // 拡大(チャンク外. 中間->中間)
    {
        NN_LOG("Resize:expand(out-chunk):ii\n");
        int64_t size0 = chunkSize * cacheCount + chunkSize / 2;
        int64_t size1 = chunkSize * cacheCount * 3 + chunkSize / 3;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.CheckRead(0, size1);
        stream.Flush();
        stream.CheckStorage();
    }
    //---------------------------------------------------------
    // 縮小・拡大(チャンク内. 境界->中間->境界)
    {
        NN_LOG("Resize:shrink(in-chunk):bi\n");
        int64_t size0 = chunkSize * 3;
        int64_t size1 = chunkSize * 2 + chunkSize / 2;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.Resize(size0);
        stream.CheckRead(0, size0);
        stream.Flush();
        stream.CheckStorage();
    }
    // 縮小・拡大(チャンク内. 中間->中間->中間)
    {
        NN_LOG("Resize:shrink(in-chunk):ii\n");
        int64_t size0 = chunkSize * 2 + chunkSize / 2;
        int64_t size1 = chunkSize * 2 + chunkSize / 3;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.Resize(size0);
        stream.CheckRead(0, size0);
        stream.Flush();
        stream.CheckStorage();
    }
    // 縮小・拡大(チャンク外. 境界->境界->境界)
    {
        NN_LOG("Resize:shrink(out-chunk):bb\n");
        int64_t size0 = chunkSize * cacheCount * 3;
        int64_t size1 = chunkSize * cacheCount;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.Resize(size0);
        stream.CheckRead(0, size0);
        stream.Flush();
        stream.CheckStorage();
    }
    // 縮小・拡大(チャンク外. 中間->境界->中間)
    {
        NN_LOG("Resize:shrink(out-chunk):bi\n");
        int64_t size0 = chunkSize * cacheCount * 3 + chunkSize / 2;
        int64_t size1 = chunkSize * cacheCount;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.Resize(size0);
        stream.CheckRead(0, size0);
        stream.Flush();
        stream.CheckStorage();
    }
    // 縮小・拡大(チャンク外. 境界->中間->境界)
    {
        NN_LOG("Resize:shrink(out-chunk):ib\n");
        int64_t size0 = chunkSize * cacheCount * 3;
        int64_t size1 = chunkSize * cacheCount + chunkSize / 2;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.Resize(size0);
        stream.CheckRead(0, size0);
        stream.Flush();
        stream.CheckStorage();
    }
    // 縮小・拡大(チャンク外. 中間->中間->中間)
    {
        NN_LOG("Resize:shrink(out-chunk):ii\n");
        int64_t size0 = chunkSize * cacheCount * 3 + chunkSize / 3;
        int64_t size1 = chunkSize * cacheCount + chunkSize / 2;
        TestStream stream;
        stream.Initialize(chunkSize, cacheCount);
        stream.WriteRandom(0, size0, rand);
        stream.Resize(size1);
        stream.Resize(size0);
        stream.CheckRead(0, size0);
        stream.Flush();
        stream.CheckStorage();
    }
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, AlbumCachedMovieStream_Random)
{
    const int64_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    const int64_t CacheCount = 4;
    std::mt19937 rand(0x24572);

    nnt::capsrv::detail::TestMovieStreamRandomAccess::Config config = {};
    config.loopCount = 1000;
    config.chunkSize = ChunkSize;
    config.offsetMax = 10 * 1024 * 1024;
    config.sizeMax   = 10 * 1024;
    config.offsetCutoffThreshold   = 256 * 1024;
    config.sizeCutoffThreshold     = 256;
    config.probabilityWrite  = 0.60f;
    config.probabilityRead   = 0.15f;
    config.probabilityResize = 0.20f;
    config.probabilityFlush  = 0.05f;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };


    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        nn::capsrv::AlbumFileId fileId = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GenerateCurrentAlbumFileId(&fileId, {0x123456}, nn::capsrv::AlbumFileContents_Movie));

        TestStream stream;
        stream.Initialize(ChunkSize, CacheCount);
        NN_UTIL_SCOPE_EXIT{ stream.Finalize(); };

        nnt::capsrv::detail::TestMovieStreamRandomAccess::Test(stream, config, rand);

        NNT_EXPECT_RESULT_SUCCESS(stream.Flush());
        EXPECT_TRUE(stream.CheckStorage());
    }
}// NOLINT(impl/function_size)
