﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <random>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "testCapsrv_TestMovieStreamWrapperBase.h"

namespace nnt{ namespace capsrv{ namespace detail{

    class TestMovieStreamRandomAccess
    {
    public:
        struct Config
        {
            int loopCount;

            int64_t chunkSize;

            int64_t offsetMax;
            int64_t sizeMax;
            int64_t offsetCutoffThreshold; // offset / chunkSize がこの値以下なら chunkSize の整数倍まで切り下げる
            int64_t sizeCutoffThreshold;   // size / chunkSize がこの値以下なら chunkSize の整数倍まで切り下げる

            float probabilityWrite;
            float probabilityRead;
            float probabilityResize;
            float probabilityFlush;
        };

    private:
        static int64_t Cutoff(int64_t offset, int64_t unit, int64_t thresh) NN_NOEXCEPT
        {
            int64_t v = offset % unit;
            if(v < thresh)
            {
                return offset - v;
            }
            return offset;
        }

    public:
        static void Test(
            TestMovieStreamWrapperBase& testStream,
            const Config& config,
            std::mt19937& rand
        ) NN_NOEXCEPT
        {
            // アクションの定義
            auto actionWrite = [&](TestMovieStreamWrapperBase& testFile, int t){
                std::uniform_int_distribution<int64_t> offsetDist(0, config.offsetMax);
                std::uniform_int_distribution<int64_t> sizeDist(0, config.sizeMax);
                auto offset = Cutoff(offsetDist(rand), config.chunkSize, config.offsetCutoffThreshold);
                auto size = Cutoff(sizeDist(rand), config.chunkSize, config.sizeCutoffThreshold);
                NN_LOG("  t=%d, write(%lld,%lld)\n", t, offset, size);
                NNT_EXPECT_RESULT_SUCCESS(testFile.WriteRandom(offset, size, rand));
            };

            auto actionRead = [&](TestMovieStreamWrapperBase& testFile, int t){
                int64_t fileSize = 0;
                NNT_EXPECT_RESULT_SUCCESS(testFile.GetSize(&fileSize));
                if(fileSize == 0)
                {
                    NN_LOG("  t=%d, read(0,0)\n", t);
                    return;
                }
                std::uniform_int_distribution<int64_t> offsetDist(0, fileSize - 1);
                auto offset = Cutoff(offsetDist(rand), config.chunkSize, config.offsetCutoffThreshold);
                std::uniform_int_distribution<int64_t> sizeDist(0, fileSize - offset);
                auto size = Cutoff(sizeDist(rand), config.chunkSize, config.sizeCutoffThreshold);
                NN_LOG("  t=%d, read(%lld,%lld)\n", t, offset, size);
                EXPECT_TRUE(testFile.CheckRead(offset, size));
            };

            auto actionResize = [&](TestMovieStreamWrapperBase& testFile, int t){
                std::uniform_int_distribution<int64_t> sizeDist(0, config.offsetMax);
                auto size = Cutoff(sizeDist(rand), config.chunkSize, config.sizeCutoffThreshold);
                NN_LOG("  t=%d, resize(%lld)\n", t, size);
                NNT_EXPECT_RESULT_SUCCESS(testFile.Resize(size));
            };

            auto actionFlush = [&](TestMovieStreamWrapperBase& testFile, int t){
                NN_LOG("  t=%d, flush()\n", t);
                NNT_EXPECT_RESULT_SUCCESS(testFile.Flush());
                EXPECT_TRUE(testFile.CheckStorage());
            };

            std::uniform_real_distribution<float> actionDist(0, 1);
            // loopCount < 0 なら無限ループに
            for(int t = 0; t < config.loopCount || config.loopCount < 0; t++)
            {
                auto p = actionDist(rand);

                if(t == 5)
                {
                    actionFlush(testStream, t);
                }

                if(p < config.probabilityWrite)
                {
                    actionWrite(testStream, t);
                    continue;
                }
                p -=config.probabilityWrite;

                if(p < config.probabilityRead)
                {
                    actionRead(testStream, t);
                    continue;
                }
                p -= config.probabilityRead;

                if(p < config.probabilityResize)
                {
                    actionResize(testStream, t);
                    continue;
                }
                p -= config.probabilityResize;

                // のこり。
                {
                    actionFlush(testStream, t);
                }
            }

        }

    };

}}}
