﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testCapsrv_TestFileUtility.h"

#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/image/image_JpegEncoder.h>
#include <nn/image/image_JpegDecoder.h>

#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/visrv_ConstructExifBuilder.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_CalculateJpegMac.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_EncryptApplicationId.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_ExifWorkStorage.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_ExtractJpegMakerNoteRange.h"
#include "testCapsrv_ScreenShotFileCreator.h"

namespace nnt{ namespace capsrv{

    namespace {
        struct NN_CAPSRV_ALIGNAS_JPEG ExifBuilderStorage
        {
            char value[6 * 1024];
        };
    }


    TestRawImage TestFileUtility::CreateRawImageDataRgba32(int width, int height, int blockSize, std::mt19937& engine) NN_NOEXCEPT
    {
        // テスト用なので効率は気にしない方向で。

        std::uniform_int_distribution<int> rand(0, 255);
        int blockWidth  = (width + blockSize - 1) / blockSize;
        int blockHeight = (height + blockSize - 1) / blockSize;
        int blockCount = blockWidth * blockHeight;

        // ブロックごとの色を決定
        std::vector<uint8_t> blocks;
        blocks.resize(4 * blockCount);
        {
            auto it = blocks.begin();
            while(it != blocks.end())
            {
                *(it++) = static_cast<uint8_t>(rand(engine));
                *(it++) = static_cast<uint8_t>(rand(engine));
                *(it++) = static_cast<uint8_t>(rand(engine));
                *(it++) = 255;
            }
        }

        // ブロックの色で塗りつぶし
        std::vector<uint8_t> image;
        image.resize(4 * width * height);
        for(int y = 0; y < height; y++)
        {
            int blockY = y / blockSize;
            for(int x = 0; x < width; x++)
            {
                int blockX = x / blockSize;
                const uint8_t* pSrc = &blocks[4 * (blockX + blockWidth * blockY)];
                uint8_t* pDst = &image[4 * (x + width * y)];
                std::memcpy(pDst, pSrc, 4);
            }
        }

        TestRawImage v;
        v.width = width;
        v.height = height;
        v.data = std::move(image);

        return v;
    }

    uint64_t TestFileUtility::CompareRawImageDataRgba32(const TestRawImage& a, const TestRawImage& b) NN_NOEXCEPT
    {
        NN_ASSERT(a.width == b.width);
        NN_ASSERT(a.height == b.height);
        uint64_t diff = 0;

        int count = a.width * a.height * 4;
        for(int i = 0; i < count; i++)
        {
            int v = a.data[i] - b.data[i];
            diff += static_cast<uint64_t>(std::abs(v));
        }

        return diff;
    }

    std::vector<uint8_t> TestFileUtility::CreateScreenShotFileData(
        const TestRawImage& raw,
        const TestScreenShotFileMetaData& meta,
        TestFileSignaturePattern signaturePattern
        ) NN_NOEXCEPT
    {
        ExifBuilderStorage exifBuilderWork;
        nn::capsrv::server::detail::MakerNoteInfo info = {};
        info.version = nn::capsrv::server::detail::MakerNoteVersion_Version1;
        info.encryptedApplicationId = nn::capsrv::server::detail::EncryptApplicationId(meta.applicationId, false);
        info.fileDescription = nn::capsrv::AlbumFileDescription_ScreenShotCaptured;
        info.dateTime = meta.time;
        info.dataWidth = raw.width;
        info.dataHeight = raw.height;
        info.dataOrientation = meta.orientation;
        info.frameCount = 1;
        info.frameRateNumerator = 0;
        info.frameRateDenominator = 0;
        info.dataDurationMilliseconds = 0;
        info.keyFrameInterval = 0;

        auto pExifBuilder = nn::capsrv::server::detail::ConstructExifBuilder(&exifBuilderWork, sizeof(ExifBuilderStorage), info, nullptr, 0);

        std::vector<uint8_t> v;
        v.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
        std::vector<char> work;

        nn::image::JpegEncoder encoder;
        nn::image::Dimension dim;
        dim.width = raw.width;
        dim.height = raw.height;
        encoder.SetPixelData(raw.data.data(), dim, 1);
        int quality;
        for(quality = 95; quality > 0; quality -= 5)
        {
            encoder.SetQuality(quality);
            encoder.Analyze();
            work.resize(encoder.GetAnalyzedWorkBufferSize());
            size_t size = 0;
            auto result = encoder.Encode(&size, v.data(), v.size(), work.data(), work.size(), pExifBuilder);
            if(result == nn::image::JpegStatus_Ok)
            {
                v.resize(size);
                break;
            }
            else if(result == nn::image::JpegStatus_ShortOutput)
            {
                continue;
            }
            else
            {
                NN_ABORT("Unexpected result\n");
            }
        }
        NN_ABORT_UNLESS(quality > 0);

        nn::capsrv::server::detail::DestructExifBuilder(pExifBuilder);
        {
            int64_t makerNoteOffset = 0;
            int64_t makerNoteSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::capsrv::server::detail::ExtractJpegMakerNoteRange(&makerNoteOffset, &makerNoteSize, v.data(), v.size(), &exifBuilderWork, sizeof(ExifBuilderStorage))
            );

            ScreenShotData data;
            data.makerNoteVersion = info.version;
            data.makerNoteOffset = makerNoteOffset;
            data.makerNoteSize = makerNoteSize;
            data.value = v;
            ScreenShotSignature sign = nnt::capsrv::ScreenShotFileCreator::CreateSignature(data.makerNoteVersion, data);
            nn::capsrv::server::detail::Signature signature = sign.value;

            if(signaturePattern == TestFileSignaturePattern_Valid)
            {
                // do nothing
            }
            else if(signaturePattern == TestFileSignaturePattern_Zero)
            {
                std::memset(&signature, 0, sizeof(signature));
            }
            else if(signaturePattern == TestFileSignaturePattern_HeadBitFlip)
            {
                signature.value[0] ^= 0x80;
            }
            else if(signaturePattern == TestFileSignaturePattern_TailBitFlip)
            {
                signature.value[sizeof(signature) - 1] ^= 0x01;
            }

            std::memcpy(v.data() + sign.offset, &signature, sizeof(signature));
        }

        return v;
    }

    TestRawImage TestFileUtility::DecodeJpeg(const std::vector<uint8_t>& jpeg) NN_NOEXCEPT
    {
        nn::image::JpegDecoder decoder;
        decoder.SetImageData(jpeg.data(), jpeg.size());
        NN_ABORT_UNLESS(
            decoder.Analyze() == nn::image::JpegStatus_Ok
        );
        auto dim = decoder.GetAnalyzedDimension();

        std::vector<char> work;
        work.resize(decoder.GetAnalyzedWorkBufferSize());

        TestRawImage image;
        image.width = dim.width;
        image.height = dim.height;
        image.data.resize(4 * dim.width * dim.height);

        NN_ABORT_UNLESS(
            decoder.Decode(image.data.data(), image.data.size(), 1, work.data(), work.size()) == nn::image::JpegStatus_Ok
        );

        return image;

    }

}}
