﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testCapsrv_MovieFileCreator.h"

#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/capsrv/movie/capsrv_MovieMetaDataBuilder.h>

namespace nnt{ namespace capsrv{

    const nn::capsrv::movie::MovieMetaDataVersion DefaultMetaDataVersion = nn::capsrv::movie::MovieMetaDataVersion_1;
    const size_t BlockSize = nn::capsrv::movie::MovieDataChunkUnitSize;
    const size_t HashSize  = nn::crypto::Sha256Generator::HashSize;

    //-----------------------------------------------------------------------
    //-----------------------------------------------------------------------

    MovieData MovieFileCreator::CreateRandomMovieData(
        size_t movieDataSize,
        std::mt19937& rand
    ) NN_NOEXCEPT
    {
        NN_ASSERT_MINMAX(movieDataSize, 0, nn::capsrv::AlbumFileSizeLimit_Movie);

        MovieData data;
        data.value.resize(movieDataSize, 0);

        // 乱数で埋める
        for(size_t i = 0; i < movieDataSize; i++)
        {
            char value = static_cast<char>(std::uniform_int_distribution<int>()(rand));
            data.value[i] = value;
        }

        return data;
    }

    //-----------------------------------------------------------------------

    ScreenShotMetaInfo MovieFileCreator::CreateScreenShotMetaDefault(
        const nn::capsrv::AlbumFileId& fileId
    ) NN_NOEXCEPT
    {
        ScreenShotMetaInfo out = ScreenShotFileCreator::CreateMetaDefault(
            nn::capsrv::server::detail::MakerNoteVersion_Version1,
            fileId.applicationId,
            fileId.time
        );

        out.description = nn::capsrv::AlbumFileDescription_MovieContinuous;
        out.frameCount = 30 * 30;
        out.frameRateNumerator = 1;
        out.frameRateDenominator = 30;
        out.dataDurationMilliseconds = 30 * 1000;
        out.keyFrameInterval = 30;
        return out;
    }


    MovieMeta MovieFileCreator::CreateMetaWithRandomImageData(
        const nn::capsrv::AlbumFileId& fileId,
        size_t movieDataSize,
        std::mt19937& rand,
        MovieMeta::Flag flags
    ) NN_NOEXCEPT
    {
        MovieMeta meta;
        meta.value.resize(nn::capsrv::movie::MovieMetaDataSize, 0);

        auto pMeta = reinterpret_cast<nn::capsrv::movie::MovieMetaData*>(meta.value.data());

        // スクリーンショットを生成
        int64_t sssize = 0;
        {
            auto ssmeta = CreateScreenShotMetaDefault(fileId);
            auto ssexif = ScreenShotFileCreator::CreateExif(ssmeta);
            auto ssdata = ScreenShotFileCreator::CreateDataWithRandomImage(ssmeta, ssexif, rand);
            auto sssign = ScreenShotFileCreator::CreateSignature(ssmeta, ssdata);
            if(flags & MovieMeta::Flag_ZeroSignature)
            {
                sssign = ScreenShotFileCreator::CreateZeroSignature(ssmeta, ssdata);
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(ScreenShotFileCreator::WriteFileData(
                [&](int64_t offset, const void* buffer, size_t size) -> nn::Result
                {
                    NN_ABORT_UNLESS(offset + size <= sizeof(pMeta->imageData));
                    std::memcpy(pMeta->imageData + offset, buffer, size);
                    sssize = std::max<int64_t>(sssize, offset + size);
                    NN_RESULT_SUCCESS;
                },
                ssdata,
                sssign
            ));

            meta.makerNoteVersion = ssdata.makerNoteVersion;
            meta.makerNoteOffset = ssdata.makerNoteOffset;
            meta.makerNoteSize = ssdata.makerNoteSize;
        }

        pMeta->infoData.movieDataSize = movieDataSize;
        pMeta->infoData.imageDataSize = sssize;
        pMeta->infoData.version = DefaultMetaDataVersion;
        meta.pMeta = pMeta;
        return meta;
    };

    void MovieFileCreator::SetValidMetaSignature(
        MovieMeta& meta
    ) NN_NOEXCEPT
    {
        ScreenShotData ssdata = {};
        ssdata.value.insert(ssdata.value.end(), meta.value.data(), meta.value.data() + meta.pMeta->infoData.imageDataSize);
        ssdata.makerNoteVersion = meta.makerNoteVersion;
        ssdata.makerNoteOffset = meta.makerNoteOffset;
        ssdata.makerNoteSize = meta.makerNoteSize;

        auto sssign = ScreenShotFileCreator::CreateSignature(meta.makerNoteVersion, ssdata);

        std::memcpy(meta.value.data() + sssign.offset, &sssign.value, sizeof(sssign.value));
    }

    //-----------------------------------------------------------------------

    MovieDataHash MovieFileCreator::CreateValidMovieDataHash(
        const MovieData& movieData
    ) NN_NOEXCEPT
    {
        std::vector<char> tmp;
        tmp.resize(BlockSize);

        MovieDataHash v;
        v.value.reserve((movieData.value.size() + BlockSize - 1) / BlockSize * HashSize);

        ptrdiff_t offset = 0;
        size_t remain = movieData.value.size();
        while(remain > 0)
        {
            size_t sizeToCopy = std::min(remain, BlockSize);
            std::memset(tmp.data(), 0, tmp.size());
            std::memcpy(tmp.data(), movieData.value.data() + offset, sizeToCopy);

            char hash[nn::crypto::Sha256Generator::HashSize] = {};
            nn::crypto::GenerateSha256Hash(hash, sizeof(hash), tmp.data(), tmp.size());

            v.value.insert(v.value.end(), hash, hash + HashSize);

            offset += sizeToCopy;
            remain -= sizeToCopy;
        }

        return v;
    }

    //-----------------------------------------------------------------------

    MovieMetaHash MovieFileCreator::CreateValidMovieMetaHash(
        const MovieMeta& movieMeta
    ) NN_NOEXCEPT
    {
        std::vector<char> tmp;
        tmp.resize(BlockSize);

        MovieMetaHash v;
        v.value.reserve((movieMeta.value.size() + BlockSize - 1) / BlockSize * HashSize);

        ptrdiff_t offset = 0;
        size_t remain = movieMeta.value.size();
        while(remain > 0)
        {
            size_t sizeToCopy = std::min(remain, BlockSize);
            std::memset(tmp.data(), 0, tmp.size());
            std::memcpy(tmp.data(), movieMeta.value.data() + offset, sizeToCopy);

            char hash[nn::crypto::Sha256Generator::HashSize] = {};
            nn::crypto::GenerateSha256Hash(hash, sizeof(hash), tmp.data(), tmp.size());

            v.value.insert(v.value.end(), hash, hash + HashSize);

            offset += sizeToCopy;
            remain -= sizeToCopy;
        }

        return v;
    }

    //-----------------------------------------------------------------------

    MovieSignature MovieFileCreator::CreateValidMovieSignature(
        const MovieDataHash& movieDataHash,
        const MovieMetaHash& movieMetaHash
    ) NN_NOEXCEPT
    {
        NN_UNUSED(movieDataHash);
        NN_UNUSED(movieMetaHash);

        std::vector<uint8_t> in;
        in.insert(in.end(), movieDataHash.value.begin(), movieDataHash.value.end());
        in.insert(in.end(), movieMetaHash.value.begin(), movieMetaHash.value.end());

        MovieSignature v = {};
        nn::crypto::GenerateSha256Hash(v.value, sizeof(v.value), in.data(), in.size());
        return v;
    }

    //-----------------------------------------------------------------------

    MovieVersion MovieFileCreator::CreateValidMovieVersion(
        uint64_t version
    ) NN_NOEXCEPT
    {
        MovieVersion v;
        v.value = version;
        return v;
    }

}}
