﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testCapsrv_FileInfo.h"

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nnt.h>

#include "testCapsrv_DirectAlbumAccessor.h"
#include "testCapsrv_TestFileUtility.h"
#include "testCapsrv_MovieFileCreator.h"

extern nnt::capsrv::DirectAlbumAccessor g_DirectAlbumAccessor;

namespace nnt{ namespace capsrv{

    bool FileInfo::Create() const
    {
        if(!DirectAlbumAccessor::IsMounted(this->storage))
        {
            NN_LOG("Creating file skipped. storage %d is not mounted\n", this->storage);
            return false;
        }

        auto path = GetDirectFilePath();

        NN_LOG("Creating random file '%s' ... ", path.c_str());
        std::mt19937 engine(static_cast<unsigned int>(this->seed));
        std::uniform_int_distribution<int> dist(0, 255);

        std::vector<uint8_t> data;
        data.resize(static_cast<size_t>(this->filesize));

        for(auto it = data.begin(); it != data.end(); ++it)
        {
            *it = static_cast<uint8_t>(dist(engine));
        }

        DirectAlbumAccessor::SaveFile(data, path.c_str());

        NN_LOG("done\n");
        return true;
    }

    bool FileInfo::CreateJpeg(TestFileSignaturePattern signaturePattern) const
    {
        if(!DirectAlbumAccessor::IsMounted(this->storage))
        {
            NN_LOG("Creating file skipped. storage %d is not mounted\n", this->storage);
            return false;
        }

        auto path = GetDirectFilePath();

        NN_LOG("Creating jpeg file '%s' ... ", path.c_str());
        std::mt19937 engine(static_cast<unsigned int>(this->seed));

        TestScreenShotFileMetaData meta;
        meta.time = this->time;
        meta.applicationId = this->appId;
        meta.orientation = nn::capsrv::ScreenShotOrientation_Default;

        auto raw = TestFileUtility::CreateRawImageDataRgba32(ImageWidth, ImageHeight, ImageBlockSize, engine);
        auto jpeg = TestFileUtility::CreateScreenShotFileData(raw, meta, signaturePattern);

        DirectAlbumAccessor::SaveFile(jpeg, path.c_str());

        NN_LOG("done\n");
        return true;
    }

    bool FileInfo::CheckData(const void* targetData, int64_t dataSize) const
    {
        auto path = GetDirectFilePath();
        NN_LOG("Checking random file data '%s' ... ", path.c_str());

        if(dataSize != this->filesize)
        {
            NN_LOG("NG\n");
            return false;
        }

        std::mt19937 engine(static_cast<unsigned int>(this->seed));
        std::uniform_int_distribution<int> dist(0,255);
        const uint8_t* data = reinterpret_cast<const uint8_t*>(targetData);
        for(int64_t i = 0; i < this->filesize; i++)
        {
            if(data[i] != static_cast<uint8_t>(dist(engine)))
            {
                NN_LOG("NG\n");
                return false;
            }
        }
        NN_LOG("ok\n");
        return true;
    }

    double FileInfo::CheckDataJpeg(const std::vector<uint8_t>& jpeg) const
    {
        auto path = GetDirectFilePath();
        NN_LOG("Checking jpeg file data '%s' ... ", path.c_str());

        std::mt19937 engine(static_cast<unsigned int>(this->seed));

        auto raw1 = TestFileUtility::DecodeJpeg(jpeg);
        auto raw2 = TestFileUtility::CreateRawImageDataRgba32(ImageWidth, ImageHeight, ImageBlockSize, engine);
        auto diff = TestFileUtility::CompareRawImageDataRgba32(raw1, raw2);

        auto v = static_cast<double>(diff) / ImageWidth / ImageHeight / 4;
        NN_LOG("%lf\n", v);
        return v;
    }

    bool FileInfo::IsMatchTime(const nn::capsrv::AlbumEntry& entry) const
    {
        return std::memcmp(&this->time, &entry.fileId.time, sizeof(nn::capsrv::AlbumFileDateTime)) == 0;
    }

    namespace {

        nn::capsrv::AlbumFileContents GetContentsFromExtension(const char* extension) NN_NOEXCEPT
        {
            nn::capsrv::AlbumFileContents contents;
            if(std::strcmp(extension, ".jpg") == 0)
            {
                contents = nn::capsrv::AlbumFileContents_ScreenShot;
            }
            else if(std::strcmp(extension, ".mp4") == 0)
            {
                contents = nn::capsrv::AlbumFileContents_Movie;
            }
            else
            {
                NN_ABORT("Unknown file extension\n");
            }
            return contents;
        }

    }

    std::string FileInfo::GetDirectFilePath() const
    {
        char filepath[DirectAlbumAccessor::PathSize];

        nn::capsrv::AlbumFileContents contents = GetContentsFromExtension(this->extension);
        DirectAlbumAccessor::GetFilePath(filepath, sizeof(filepath), this->storage, this->time, this->appId, static_cast<nn::capsrv::AlbumFileContentsType>(contents));
        return filepath;
    }

    nn::capsrv::AlbumFileId FileInfo::GetAlbumFileId() const
    {
        nn::capsrv::AlbumFileId id = {};
        id.applicationId = this->appId;
        id.contents = static_cast<nn::capsrv::AlbumFileContentsType>(GetContentsFromExtension(this->extension));
        id.storage = this->storage;
        id.time = this->time;
        return id;
    }

    std::vector<uint8_t> FileInfo::GetData() const
    {
        std::mt19937 engine(static_cast<unsigned int>(this->seed));
        std::uniform_int_distribution<int> dist(0, 255);

        std::vector<uint8_t> data;
        data.resize(static_cast<size_t>(this->filesize));

        for(auto it = data.begin(); it != data.end(); ++it)
        {
            *it = static_cast<uint8_t>(dist(engine));
        }

        return data;
    }

    bool FileInfo::CreateMovie() const NN_NOEXCEPT
    {
        std::mt19937 engine(static_cast<unsigned int>(this->seed));

        auto movieData = MovieFileCreator::CreateRandomMovieData(this->filesize, engine);
        auto movieMeta = MovieFileCreator::CreateMetaWithRandomImageData(GetAlbumFileId(), this->filesize, engine);
        auto dataHash = MovieFileCreator::CreateValidMovieDataHash(movieData);
        auto metaHash = MovieFileCreator::CreateValidMovieMetaHash(movieMeta);
        auto signature = MovieFileCreator::CreateValidMovieSignature(dataHash, metaHash);
        auto version = MovieFileCreator::CreateValidMovieVersion(1);

        if(!DirectAlbumAccessor::IsMounted(this->storage))
        {
            return false;
        }

        auto filePath = DirectAlbumAccessor::GetFilePath(GetAlbumFileId());
        DirectAlbumAccessor::CreateFile(0, filePath.c_str());

        nn::fs::FileHandle hFile = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&hFile, filePath.c_str(), nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{
            nn::fs::FlushFile(hFile);
            nn::fs::CloseFile(hFile);
        };

        NN_ABORT_UNLESS_RESULT_SUCCESS(MovieFileCreator::WriteFileData(
            [&](int64_t offset, const void* buffer, size_t size) -> nn::Result
            {
                return nn::fs::WriteFile(hFile, offset, buffer, size, nn::fs::WriteOption::MakeValue(0));
            },
            movieData,
            movieMeta,
            dataHash,
            metaHash,
            signature,
            version
        ));

        return true;
    }

}}
