﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testCapsrv_DirectAlbumAccessor.h"

#include <cstring>
#include <vector>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/fs/fs_ImageDirectory.h>
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_Config.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/album/capsrvServer_AlbumPathUtility.h"

#include "testCapsrv_Macro.h"


#define NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX "t"

namespace nnt{ namespace capsrv{

    namespace{
        const char* MountNameList[nn::capsrv::AlbumStorageCount] = {
            NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX NN_CAPSRV_MOUNT_NAME_NAND,
            NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX NN_CAPSRV_MOUNT_NAME_SD,
        };

        const nn::fs::ImageDirectoryId ImageDirIdList[nn::capsrv::AlbumStorageCount] = {
            nn::fs::ImageDirectoryId::Nand,
            nn::fs::ImageDirectoryId::SdCard
        };

        const char* RootPathList[nn::capsrv::AlbumStorageCount] = {
            NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX NN_CAPSRV_MOUNT_NAME_NAND ":/",
            NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX NN_CAPSRV_MOUNT_NAME_SD ":/",
        };


        struct DirectAlbumAccessorStatus
        {
            bool isMounted[nn::capsrv::AlbumStorageCount];
        };
        DirectAlbumAccessorStatus g_Status = {};
    }

    void DirectAlbumAccessor::MountAllStorages() NN_NOEXCEPT
    {
        NNT_CAPSRV_FOREACH_STORAGE_I(i, storage)
        {
            NN_UNUSED(storage);
            auto result = nn::fs::MountImageDirectory(MountNameList[i], ImageDirIdList[i]);
            g_Status.isMounted[i] = result.IsSuccess();
        }
    }

    void DirectAlbumAccessor::UnmountAllStorages() NN_NOEXCEPT
    {
        NNT_CAPSRV_FOREACH_STORAGE_I(i, storage)
        {
            NN_UNUSED(storage);
            if(g_Status.isMounted[i])
            {
                nn::fs::Unmount(MountNameList[i]);
                g_Status.isMounted[i] = false;
            }
        }
    }

    bool DirectAlbumAccessor::IsMounted(nn::capsrv::AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(storage, 0, nn::capsrv::AlbumStorageCount);
        return g_Status.isMounted[storage];
    }

    namespace{
        bool ConcatPath(char* pOutBuffer, int bufferSize, const char* basePath, const char* relPath) NN_NOEXCEPT
        {
            int length = 0;
            length += nn::util::Strlcpy(pOutBuffer + length, basePath, bufferSize - length);
            length += nn::util::Strlcpy(pOutBuffer + length, "/", bufferSize - length);
            length += nn::util::Strlcpy(pOutBuffer + length, relPath, bufferSize - length);
            return length < bufferSize;
        }
    }

    bool DirectAlbumAccessor::CleanupAlbum(nn::capsrv::AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(storage, 0, nn::capsrv::AlbumStorageCount);
        const char* path = GetRootPath(storage);

        nn::fs::DirectoryHandle hDir;

        nn::Result fsResult;
        fsResult = nn::fs::OpenDirectory(&hDir, path, nn::fs::OpenDirectoryMode_All);
        if(!fsResult.IsSuccess())
        {
            return false;
        }

        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(hDir); };

        int64_t count = 0;
        fsResult = nn::fs::GetDirectoryEntryCount(&count, hDir);
        if(!fsResult.IsSuccess())
        {
            return false;
        }

        if(count == 0)
        {
            return true;
        }

        std::vector<nn::fs::DirectoryEntry> entries;
        entries.resize(static_cast<size_t>(count));

        int64_t entryCount = 0;
        fsResult = nn::fs::ReadDirectory(&entryCount, entries.data(), hDir, count);
        if(!fsResult.IsSuccess())
        {
            return false;
        }

        entries.resize(static_cast<size_t>(entryCount));

        char filepath[512];
        bool ok = true;
        for(const auto& e : entries)
        {
            if(!ConcatPath(filepath, sizeof(filepath), path, e.name))
            {
                ok = false;
                continue;
            }
            if(e.directoryEntryType == nn::fs::DirectoryEntryType_File)
            {
                fsResult = nn::fs::DeleteFile(filepath);
                if(!fsResult.IsSuccess())
                {
                    ok = false;
                    continue;
                }
            }
            else if(e.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                fsResult = nn::fs::DeleteDirectoryRecursively(filepath);
                if(!fsResult.IsSuccess())
                {
                    ok = false;
                    continue;
                }
            }
            else
            {
                ok = false;
            }
        }

        return ok;
    }

    bool DirectAlbumAccessor::CleanupAllAlbums() NN_NOEXCEPT
    {
        NN_LOG("Cleaning up album...\n");
        bool ok = true;
        NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE(storage)
        {
            ok &= CleanupAlbum(storage);
        }
        NN_LOG("Cleaning up album...complete\n");
        return ok;
    }

    const char* DirectAlbumAccessor::GetRootPath(nn::capsrv::AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(storage, 0, nn::capsrv::AlbumStorageCount);
        return RootPathList[storage];
    }

    void DirectAlbumAccessor::GetFilePath(
        char* pOutBuffer,
        size_t bufferSize,
        nn::capsrv::AlbumStorageType storage,
        const nn::capsrv::AlbumFileDateTime& time,
        nn::ncm::ApplicationId appId,
        nn::capsrv::AlbumFileContentsType contents
        ) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(bufferSize, static_cast<size_t>(PathSize));

        const size_t prefixLength = std::strlen(NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX);
        std::memcpy(pOutBuffer, NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX, prefixLength);
        nn::capsrv::AlbumFileId fileId = {};
        fileId.storage = storage;
        fileId.time = time;
        fileId.applicationId = appId;
        fileId.contents = contents;

        auto env = nn::capsrv::server::EnvironmentInfo::GetValue(true, true, true);

        int n;
        nn::capsrv::server::album::AlbumPathUtility::GetFilepath(&n, pOutBuffer + prefixLength, bufferSize - prefixLength, fileId, env);
    }

    std::string DirectAlbumAccessor::GetFilePath(const nn::capsrv::AlbumFileId& fileId) NN_NOEXCEPT
    {
        char path[PathSize];
        GetFilePath(path, sizeof(path), fileId.storage, fileId.time, fileId.applicationId, fileId.contents);
        return path;
    }

    void DirectAlbumAccessor::GetSubDirectoryPath(
        int* pOutLength,
        char* pBuffer,
        size_t bufferSize,
        const nn::capsrv::AlbumFileId& fileId,
        int depth
        ) NN_NOEXCEPT
    {
        const size_t prefixLength = std::strlen(NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX);
        std::memcpy(pBuffer, NNT_CAPSRV_TEST_MOUNT_NAME_PREFIX, prefixLength);

        int n = 0;
        nn::capsrv::server::album::AlbumPathUtility::GetSubDirectoryPath(&n, pBuffer + prefixLength, bufferSize - prefixLength, fileId, depth, nn::capsrv::IsExtraAlbumFileContents(fileId.contents));

        *pOutLength = static_cast<int>(prefixLength) + n;
    }

    int DirectAlbumAccessor::GetSubDirectoryDepth() NN_NOEXCEPT
    {
        return nn::capsrv::server::album::AlbumPathUtility::SubDirectoryDepth;
    }


    std::vector<uint8_t> DirectAlbumAccessor::LoadFile(const char* path) NN_NOEXCEPT
    {
        nn::fs::FileHandle h;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::OpenFile(&h, path, nn::fs::OpenMode_Read)
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(h);
        };

        int64_t size;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::GetFileSize(&size, h)
        );

        std::vector<uint8_t> v;
        v.resize(static_cast<size_t>(size));

        size_t readSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::ReadFile(&readSize, h, 0, v.data(), v.size())
        );

        v.resize(readSize);

        return v;
    }

    void DirectAlbumAccessor::SaveFile(const std::vector<uint8_t>& data, const char* path) NN_NOEXCEPT
    {
        CreateParentDirectories(path);

        nn::fs::FileHandle h;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::CreateFile(path, static_cast<int64_t>(data.size()))
        );
        if(data.size() > 0)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::OpenFile(&h, path, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(h);
            };
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::WriteFile(h, 0, data.data(), data.size(), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush))
            );
        }
    }

    void DirectAlbumAccessor::CreateFile(int64_t size, const char* path) NN_NOEXCEPT
    {
        CreateParentDirectories(path);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::CreateFile(path, size)
        );
    }

    void DirectAlbumAccessor::CreateDirectories(const char* path) NN_NOEXCEPT
    {
        char work[PathSize] = {};
        int length = nn::util::Strlcpy(work, path, sizeof(work));

        for(;;)
        {
            NN_ABORT_UNLESS_GREATER(length, 0);
            if(work[length - 1] == '/')
            {
                work[length - 1] = '\0';
                length--;
            }
            else
            {
                break;
            }
        }
        work[length] = '/';
        length++;

        int pos = 0;

        // skip "ST:/"
        for(pos = 0; pos < length; pos++)
        {
            if(work[pos] == ':')
            {
                pos++;
                break;
            }
        }
        NN_ABORT_UNLESS_EQUAL(work[pos], '/');
        pos++;

        for(; pos < length; pos++)
        {
            if(work[pos] == '/')
            {
                work[pos] = '\0';
                auto result = nn::fs::CreateDirectory(work);
                NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));
                work[pos] = '/';
            }
        }
    }

    void DirectAlbumAccessor::CreateParentDirectories(const char* path) NN_NOEXCEPT
    {
        char work[PathSize] = {};
        int length = nn::util::Strlcpy(work, path, sizeof(work));
        int pos = 0;
        for(pos = length - 1; pos >= 0; pos--)
        {
            if(work[pos] == '/')
            {
                work[pos] = '\0';
                break;
            }
        }
        NN_ABORT_UNLESS_GREATER_EQUAL(pos, 0);
        CreateDirectories(work);
    }

}}
