﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
// #define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/init.h>
#include <nn/util/util_Color.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nn/btm/btm.h>
#include <nn/btm/system/btm_SystemApi.h>

#include "../Common/BtmTools_GraphicsSystem.h"
#include "../Common/BtmTools_Pad.h"

namespace
{
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nnt::btm::GraphicsSystem* g_pGraphicsSystem;

    bool g_IsAirplaneMode;

    const nn::util::Color4u8Type Color1 = { { 255, 128, 0, 255 } };
}

//------------------------------------------------------------------------------
//  キーイベント処理
//------------------------------------------------------------------------------
void KeyEventFunc(nn::hid::NpadButtonSet& state)
{
    if (state.Test<nn::hid::NpadJoyButton::A>())
    {
        bool radioMode = g_IsAirplaneMode;
        NN_LOG("@@@ Set Radio %s\n", radioMode ? "ON" : "OFF");
        nn::btm::system::EnableRadio(radioMode);
        // 遷移の完了を待ちます
        for (int i = 0; ; ++i)
        {
            bool isRadioOn = nn::btm::system::IsRadioEnabled();
            if (isRadioOn == radioMode)
            {
                g_IsAirplaneMode = !isRadioOn;
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
    }
}
//------------------------------------------------------------------------------
//  プリント 関数
//------------------------------------------------------------------------------
void Print(nn::gfx::util::DebugFontTextWriter& writer, int frame)
{
    writer.SetScale(1.3f, 1.3f);
    static const float Top = 100;
    static const float Left = 100;
    float w = writer.GetFontWidth();
    float h = writer.GetFontHeight();

    writer.SetTextColor(nn::util::Color4u8::White());
    writer.SetCursor(Left, Top);

    writer.Print("Bluetooth Airplane mode :   ");
    if (g_IsAirplaneMode)
    {
        writer.SetTextColor(Color1);
        writer.Print("ON\n");
    }
    else
    {
        writer.SetTextColor(nn::util::Color4u8::White());
        writer.Print("OFF\n");
    }

    writer.SetCursor(Left + w * 3, writer.GetCursorY() + h);
    if (g_IsAirplaneMode)
    {
        writer.SetTextColor(nn::util::Color4u8::Gray());
        writer.Print("ON (Disable Bluetooth)\n");
        writer.SetTextColor(nn::util::Color4u8::White());
        writer.Print("OFF (Enable Bluetooth)\n");
        writer.SetCursor(Left + w, writer.GetCursorY() - h);
    }
    else
    {
        writer.SetTextColor(nn::util::Color4u8::White());
        writer.Print("ON (Disable Bluetooth)\n");
        writer.SetTextColor(nn::util::Color4u8::Gray());
        writer.Print("OFF (Enable Bluetooth)\n");
        writer.SetCursor(Left + w, writer.GetCursorY() - h * 2);
    }
    writer.SetTextColor(nn::util::Color4u8::White());
    writer.Print(">");

    writer.SetCursor(Left, Top + h * 5);
    writer.Print("[A] Turn on/off\n");
}

bool ExitApplicationRequest()
{
    nn::oe::Message message;
    if (nn::oe::TryPopNotificationMessage(&message))
    {
        switch(message)
        {
        // フォーカス状態変更通知
        case nn::oe::MessageFocusStateChanged:
            break;
        case nn::oe::MessageExitRequest:
            {
                NN_LOG("Received MessageExitRequest\n");
                return true;
            }
        // 動作モード（携帯／据置）が変更
        case nn::oe::MessageOperationModeChanged:
            break;
        // 性能モード（ノーマル／ブースト）が変更
        case nn::oe::MessagePerformanceModeChanged:
            break;
        // 未定義メッセージ
        default:
            {
                NN_LOG("Received unknown message= 0x%08x", message);
            }
            break;
        }
    }
    return false;
}
//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    NN_LOG( "Start demo.\n" );
    // FS を使用する場合はフレームワークよりも前に初期化します

    nn::oe::Initialize();
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);
    nn::oe::EnterExitRequestHandlingSection();

    // フレームワーク初期化
    g_pGraphicsSystem = new nnt::btm::GraphicsSystem();
    g_pGraphicsSystem->Initialize(GraphicsSystemMemorySize);

    // デバッグフォント初期化
    g_pGraphicsSystem->InitializeDebugFont();
    NN_LOG("Debugfont initialized\n");

    nn::gfx::util::DebugFontTextWriter* pWriter = g_pGraphicsSystem->GetDebugFontTextWriter();

    nnt::btm::Pad pad;
    pad.Initialize();

    // 機内モードの状態取得をします
    g_IsAirplaneMode = !nn::btm::system::IsRadioEnabled();
    NN_LOG("Bluetooth Airplane mode: %s\n", g_IsAirplaneMode ? "ON" : "OFF");

    for (int frame = 0; !ExitApplicationRequest(); ++frame)
    {
        nn::hid::NpadButtonSet buttonState = pad.Update();
        KeyEventFunc(buttonState);
        Print(*pWriter, frame);
        g_pGraphicsSystem->Update();
    }

    // デバッグフォント終了
    g_pGraphicsSystem->FinalizeDebugFont();

    // フレームワーク終了
    g_pGraphicsSystem->Finalize();
    delete g_pGraphicsSystem;

    NN_LOG("End demo.\n");
    nn::oe::LeaveExitRequestHandlingSection();
}
