﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <cstdlib>

class SlotSimulator
{
public:
    SlotSimulator();
    void AddDevice(uint8_t slotSize);//指定された連続数を前方検索で確保。レコード更新チェックもする
    void RemoveDevice();//ランダムに、どれかを消す
    void PrintRecord() const;//レコードをログ出力
    void PrintSlotList() const;//現在のSlotList出力

protected:
    enum OccupiedType
    {
        OccupiedType_Blank,
        OccupiedType_Solo,
        OccupiedType_LeftOfDuo,
        OccupiedType_RightOfDuo,
    };

    struct Slot
    {
        OccupiedType occupiedType;
    };

    static const int COUNT_OF_SLOT = 36;
    struct SlotList
    {
        Slot slot[COUNT_OF_SLOT];
    };

    struct Record
    {
        SlotList slotList;
    };

    static const int COUNT_OF_DEVICE = 8;
    static const int COUNT_OF_RECORD = COUNT_OF_DEVICE + 1;
    struct RecordList
    {
        Record record[COUNT_OF_RECORD];
    };

    SlotList m_SlotList;
    RecordList m_RecordList;

    //対象は引数のslotList。メンバ変数ではない。
    uint8_t CountDevice(const SlotList& slotList) const;//Device数をカウント
    uint8_t CountDuoDevice(const SlotList& slotList) const;//DuoDevice数をカウント。StaticJoyでのDuoデバイス上限制御用
    uint8_t CountSlotWithDiscrete(const SlotList& slotList) const;//歯抜け含んだ占有スロット数をカウント
    bool ValidateStaticJoy(const SlotList& slotList) const;//StaticJoyの事後条件確認
    void PrintSlotList(const SlotList& slotList) const;

    void UpdateRecord();//slotListが最大占有だった場合、その値を保持
};



//[Todo] 違反命令をどの層でチェックするか。まずは指示側は考えないほうが楽か。

SlotSimulator::SlotSimulator()
{
    for(int i=0;i<COUNT_OF_SLOT;i++)
    {
        m_SlotList.slot[i].occupiedType = OccupiedType_Blank;
    }
    for(int i=0;i<COUNT_OF_RECORD;i++)
    {
        m_RecordList.record[i].slotList = m_SlotList;
    }
}

void SlotSimulator::AddDevice(uint8_t slotSize)
{
    if(slotSize == 0)
    {
        return;
    }
    if(slotSize > 2)
    {
        return;
    }

    //事後条件確認で正当性を確かめるため、コピーを操作
    SlotList tempSlotList = m_SlotList;

    uint8_t multiBlankCount = 0;
    for(int i=0;i<COUNT_OF_SLOT;i++)
    {
        if(tempSlotList.slot[i].occupiedType == OccupiedType_Blank)
        {
            multiBlankCount++;
            if(multiBlankCount == slotSize)
            {
                if(slotSize == 1)
                {
                    tempSlotList.slot[i].occupiedType = OccupiedType_Solo;
                    break;
                }
                else if(slotSize == 2)
                {
                    tempSlotList.slot[i - 1].occupiedType = OccupiedType_LeftOfDuo;
                    tempSlotList.slot[i].occupiedType = OccupiedType_RightOfDuo;
                    break;
                }
                else
                {
                    NN_ABORT("faulty design\n");
                }
            }
        }
        else
        {
            multiBlankCount = 0;
        }
    }

    bool isValid = ValidateStaticJoy(tempSlotList);
    if(isValid)
    {
        m_SlotList = tempSlotList;
        UpdateRecord();
    }
}

void SlotSimulator::RemoveDevice()
{
    SlotList tempSlotList = m_SlotList;

    //全部Blankのこともあるので、何回か試したら止める
    for(int i=0;i<50;i++)
    {
        uint8_t index;
        index = rand() % COUNT_OF_SLOT;

        if(tempSlotList.slot[index].occupiedType == OccupiedType_Solo)
        {
            tempSlotList.slot[index].occupiedType = OccupiedType_Blank;
            break;
        }
        else if(tempSlotList.slot[index].occupiedType == OccupiedType_LeftOfDuo)
        {
            tempSlotList.slot[index].occupiedType = OccupiedType_Blank;
            tempSlotList.slot[index + 1].occupiedType = OccupiedType_Blank;
            break;
        }
        else if(tempSlotList.slot[index].occupiedType == OccupiedType_RightOfDuo)
        {
            tempSlotList.slot[index].occupiedType = OccupiedType_Blank;
            tempSlotList.slot[index - 1].occupiedType = OccupiedType_Blank;
            break;
        }
        else
        {
            //元々Blank
        }
    }

    bool isValid = ValidateStaticJoy(tempSlotList);
    if(isValid)
    {
        m_SlotList = tempSlotList;
        UpdateRecord();//ここには不要だけど、一応
    }
    else
    {
        NN_ABORT("faulty design\n");
    }
}

void SlotSimulator::PrintSlotList() const
{
    PrintSlotList(m_SlotList);
}

void SlotSimulator::PrintSlotList(const SlotList& slotList) const
{
    NN_LOG("[%d] OccupiedSlotsWithDiscrete = %2d, ", CountDevice(slotList), CountSlotWithDiscrete(slotList));
    for(int i=0;i<COUNT_OF_SLOT;i++)
    {
        OccupiedType occupiedType = slotList.slot[i].occupiedType;
        if(occupiedType == OccupiedType_Blank)
        {
            NN_LOG("-");
        }
        else if(occupiedType == OccupiedType_Solo)
        {
            NN_LOG("S");
        }
        else if(occupiedType == OccupiedType_LeftOfDuo)
        {
            NN_LOG("L");
        }
        else if(occupiedType == OccupiedType_RightOfDuo)
        {
            NN_LOG("R");
        }
        else
        {
            NN_LOG("?");
        }
    }
    NN_LOG("\n");
}

void SlotSimulator::PrintRecord() const
{
    NN_LOG("-- 1 slot means FromMaster + FromDevice --\n");
    for(int i=0;i<COUNT_OF_RECORD;i++)
    {
        PrintSlotList(m_RecordList.record[i].slotList);
    }
}

uint8_t SlotSimulator::CountDevice(const SlotList& slotList) const
{
    uint8_t count = 0;
    for(int i=0;i<COUNT_OF_SLOT;i++)
    {
        if(slotList.slot[i].occupiedType == OccupiedType_Solo)
        {
            count++;
        }
        else if(slotList.slot[i].occupiedType == OccupiedType_LeftOfDuo)
        {
            //Duoは左のみカウントする
            count++;
        }
        else
        {

        }
    }

    return count;
}

uint8_t SlotSimulator::CountDuoDevice(const SlotList& slotList) const
{
    uint8_t count = 0;
    for(int i=0;i<COUNT_OF_SLOT;i++)
    {
        if(slotList.slot[i].occupiedType == OccupiedType_LeftOfDuo)
        {
            //Duoは左のみカウントする
            count++;
        }
        else
        {

        }
    }

    return count;
}

uint8_t SlotSimulator::CountSlotWithDiscrete(const SlotList& slotList) const
{
    uint8_t index = 0;
    for(int i=0;i<COUNT_OF_SLOT;i++)
    {
        if(slotList.slot[i].occupiedType != OccupiedType_Blank)
        {
            index = i;
        }
    }

    //配列添え字なので、+1にしてカウント数化する
    if(index != 0)
    {
        index++;
    }

    return index;
}

bool SlotSimulator::ValidateStaticJoy(const SlotList& slotList) const
{
    if(CountDevice(slotList) <= 4)
    {
        //4台以下はスロットの制約無し
        return true;
    }
    else if(CountDevice(slotList) <= 8)
    {
        if(CountDuoDevice(slotList) <= 1)
        {
            return true;
        }
        else
        {
            //5台以上は、2スロットは1台まで
            return false;
        }
    }
    else
    {
        //最大台数は8台まで
        return false;
    }
}

void SlotSimulator::UpdateRecord()
{
    uint8_t deviceCount;
    deviceCount = CountDevice(m_SlotList);

    uint8_t recordedSlotCount;
    uint8_t slotCount;
    recordedSlotCount = CountSlotWithDiscrete(m_RecordList.record[deviceCount].slotList);
    slotCount = CountSlotWithDiscrete(m_SlotList);

    if(recordedSlotCount < slotCount)
    {
        m_RecordList.record[deviceCount].slotList = m_SlotList;
    }
}

void Simulate()
{
    SlotSimulator slotSimulator;
    for(int i=0;i<1000000;i++)
    {
        if(rand() % 2 == 0)
        {
            slotSimulator.AddDevice(rand() % 2 + 1);
        }
        else
        {
            slotSimulator.RemoveDevice();
        }
        //slotSimulator.PrintSlotList();
    }
    slotSimulator.PrintRecord();
}

extern "C" void nnMain()
{
    NN_LOG("Hi\n");
    for(int i=0;i<20;i++)
    {
        Simulate();
    }
}
