﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Types.h>
#include <nn/btm/btm_Result.h>

#include <nn/btm/debug/btm_DebugApi.h>

#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>

#include <nn/btm/user/btm_UserApi.h>
#include <nn/btm/user/btm_UserResult.h>
#include <nn/bluetooth/bluetooth_BleScanParameterIdTestPeripheral.h>

#include "testBtmModule_TestBase.h"

TestBtmModule_TestBase::TestBtmModule_TestBase()
{
    nn::os::InitializeEvent(&m_ConnectedDeviceConditionEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_RegisterDeviceInfoEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_LlrStateEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::InitializeEvent(&m_RadioEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_GamepadPairingEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::InitializeEvent(&m_BleScanEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_BleConnectionEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_BleServiceDiscoveryEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_BlePairingEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_BleMtuConfigEvent, false, nn::os::EventClearMode_AutoClear);
}

TestBtmModule_TestBase::~TestBtmModule_TestBase()
{
    nn::os::FinalizeEvent(&m_ConnectedDeviceConditionEvent);
    nn::os::FinalizeEvent(&m_RegisterDeviceInfoEvent);
    nn::os::FinalizeEvent(&m_LlrStateEvent);

    nn::os::FinalizeEvent(&m_RadioEvent);
    nn::os::FinalizeEvent(&m_GamepadPairingEvent);

    nn::os::FinalizeEvent(&m_BleScanEvent);
    nn::os::FinalizeEvent(&m_BleConnectionEvent);
    nn::os::FinalizeEvent(&m_BleServiceDiscoveryEvent);
    nn::os::FinalizeEvent(&m_BlePairingEvent);
    nn::os::FinalizeEvent(&m_BleMtuConfigEvent);
}

void TestBtmModule_TestBase::HandleConnectedDeviceConditionEvent()
{
    nn::os::SignalEvent(&m_ConnectedDeviceConditionEvent);
}

void TestBtmModule_TestBase::HandleRegisterDeviceInfoEvent()
{
    nn::os::SignalEvent(&m_RegisterDeviceInfoEvent);
}

void TestBtmModule_TestBase::HandleLlrStateEvent()
{
    nn::os::SignalEvent(&m_LlrStateEvent);
}

void TestBtmModule_TestBase::HandleRadioEvent()
{
    nn::os::SignalEvent(&m_RadioEvent);
}

void TestBtmModule_TestBase::HandleGamepadPairingEvent()
{
    nn::os::SignalEvent(&m_GamepadPairingEvent);
}

void TestBtmModule_TestBase::HandleBleScanEvent()
{
    nn::os::SignalEvent(&m_BleScanEvent);
}

void TestBtmModule_TestBase::HandleBleConnectionEvent()
{
    nn::os::SignalEvent(&m_BleConnectionEvent);
}

void TestBtmModule_TestBase::HandleBleServiceDiscoveryEvent()
{
    nn::os::SignalEvent(&m_BleServiceDiscoveryEvent);
}

void TestBtmModule_TestBase::HandleBlePairingEvent()
{
    nn::os::SignalEvent(&m_BlePairingEvent);
}

void TestBtmModule_TestBase::HandleBleMtuConfigEvent()
{
    nn::os::SignalEvent(&m_BleMtuConfigEvent);
}

void TestBtmModule_TestBase::DisconnectAllControllers()
{
    nn::btm::DeviceConditionList deviceConditionList;
    nn::btm::GetConnectedDeviceCondition(&deviceConditionList);

    while(deviceConditionList.deviceCount > 0)
    {
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        nn::btm::HidDisconnect(&deviceConditionList.device[0].bdAddress);
        nn::os::WaitEvent(&m_ConnectedDeviceConditionEvent);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
    }

    nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
    NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 0);
}

void TestBtmModule_TestBase::UnpairAllControllers()
{
    nn::btm::DeviceInfoList deviceInfoList;
    nn::btm::GetDeviceInfo(&deviceInfoList);

    while(deviceInfoList.deviceCount > 0)
    {
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::RemoveDeviceInfo(&deviceInfoList.device[0].bdAddress);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&deviceInfoList);
    }

    nn::btm::GetDeviceInfo(&deviceInfoList);
    NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, 0);
}

void TestBtmModule_TestBase::WaitControllerConnection(uint8_t deviceCount)
{
    // 呼ばれた時点のRadio の状態を確認
    bool isRadioEnabled = nn::btm::system::IsRadioEnabled();

    // Radio OFF なら一旦ON にする
    if (!isRadioEnabled)
    {
        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(true);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());
    }

    DisconnectAllControllers();
    UnpairAllControllers();

    // 念のため、実行前にクリアしておく
    nn::os::ClearEvent(&m_GamepadPairingEvent);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::system::StartGamepadPairing());
    nn::os::WaitEvent(&m_GamepadPairingEvent);
    NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());

    NN_TEST_BTM_MODULE_LOG0("\n");
    NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
    NN_TEST_BTM_MODULE_LOG("Press %d Joy-Con(s) sync button for long to start gamepad pairing.\n", deviceCount);

    bool isGamepadPaired            = false;
    bool isGamepadConnected         = false;

    nn::btm::DeviceConditionList deviceConditionList;
    nn::btm::DeviceInfoList deviceInfoList;

    while (!isGamepadPaired || !isGamepadConnected)
    {
        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        isGamepadConnected = (deviceConditionList.deviceCount >= deviceCount);

        nn::btm::GetDeviceInfo(&deviceInfoList);
        isGamepadPaired = (deviceInfoList.deviceCount >= deviceCount);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

    // 念のため、実行前にクリアしておく
    nn::os::ClearEvent(&m_GamepadPairingEvent);

    nn::btm::system::CancelGamepadPairing();

    nn::os::WaitEvent(&m_GamepadPairingEvent);

    NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());

    // 元に戻す
    if (!isRadioEnabled)
    {
        DisableRadio();
    }
}

void TestBtmModule_TestBase::WaitDeviceCondition(uint8_t deviceCount, nn::btm::BluetoothMode bluetoothMode, nn::btm::WlanMode wlanMode,
                                                 bool isSlotSavingForPairing, bool isSlotSaving, uint8_t largeSlotDeviceCount,
                                                 uint64_t durationMsec)
{
    nn::btm::DeviceConditionList deviceConditionList;
    nn::os::TimerEventType timerEvent;
    nn::TimeSpan oneShotTime;

    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
    oneShotTime = nn::TimeSpan::FromMilliSeconds(durationMsec);
    nn::os::StartOneShotTimerEvent(&timerEvent, oneShotTime);

    while (NN_STATIC_CONDITION(true))
    {
        int largeSlotDeviceNum = 0;

        if (nn::os::TryWaitTimerEvent(&timerEvent))
        {
            NN_TEST_BTM_MODULE_LOG0("Timeout to wait device condition list update as expected\n");

            NN_TEST_BTM_MODULE_LOG0("Expected device condition is like below:\n");
            NN_TEST_BTM_MODULE_LOG("Device Count = %d, Large Slot Device Count = %d\n", deviceCount, largeSlotDeviceCount);
            NN_TEST_BTM_MODULE_LOG("Bluetooth Mode = %d, Wlan Mode = %d, isSlotSavingForPairing = %s, isSlotSaving = %s\n",
                bluetoothMode, wlanMode,
                isSlotSavingForPairing ? "TRUE" : "FALSE",
                isSlotSaving ? "TRUE" : "FALSE");

            NN_TEST_BTM_MODULE_LOG0("Final device condition list is like below:\n");
            NN_TEST_BTM_MODULE_LOG("Device Count = %d\n", deviceConditionList.deviceCount);
            NN_TEST_BTM_MODULE_LOG("Bluetooth Mode = %d, Wlan Mode = %d, isSlotSavingForPairing = %s, isSlotSaving = %s\n",
                deviceConditionList.bluetoothMode, deviceConditionList.wlanMode,
                deviceConditionList.isSlotSavingForPairing ? "TRUE" : "FALSE",
                deviceConditionList.isSlotSaving ? "TRUE)" : "FALSE"
            );

            for (auto deviceInfo : deviceConditionList.device)
            {
                NN_TEST_BTM_MODULE_LOG("Address = %02X:%02X:%02X:%02X:%02X:%02X\n",
                    deviceInfo.bdAddress.address[0], deviceInfo.bdAddress.address[1], deviceInfo.bdAddress.address[2],
                    deviceInfo.bdAddress.address[3], deviceInfo.bdAddress.address[4], deviceInfo.bdAddress.address[5]);
                NN_TEST_BTM_MODULE_LOG("    Slot Mode = %d, Sniff Mode = %d, isLargeSlotRequired = %s\n",
                    deviceInfo.hidDeviceCondition.slotMode, deviceInfo.hidDeviceCondition.sniffMode,
                    deviceInfo.hidDeviceCondition.isLargeSlotRequired ? "TRUE" : "FALSE");
            }

            NN_ABORT();
        }

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);

        for (int i = 0; i < deviceConditionList.deviceCount; ++i)
        {
            if (deviceConditionList.device[i].hidDeviceCondition.slotMode == nn::btm::SlotMode_4)
            {
                largeSlotDeviceNum++;
            }
        }

        // 満たすべき条件は以下
        if (deviceConditionList.deviceCount             == deviceCount              &&
            deviceConditionList.bluetoothMode           == bluetoothMode            &&
            deviceConditionList.wlanMode                == wlanMode                 &&
            deviceConditionList.isSlotSavingForPairing  == isSlotSavingForPairing   &&
            deviceConditionList.isSlotSaving            == isSlotSaving             &&
            largeSlotDeviceNum                          == largeSlotDeviceCount)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3));
    }

    nn::os::FinalizeTimerEvent(&timerEvent);
}

void TestBtmModule_TestBase::DisconnectAllBleDevices()
{
    nn::btm::user::BleClientConnState connectionState[nn::bluetooth::BleConnectionCountMaxClient];
    uint8_t deviceCount = nn::btm::user::BleGetConnectionState(connectionState, NN_ARRAY_SIZE(connectionState));

    while (deviceCount > 0)
    {
        for (int i = 0; i < deviceCount; ++i)
        {
            if (connectionState[i].connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
            {
                nn::os::ClearEvent(&m_BleConnectionEvent);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleDisconnect(connectionState[i].connectionHandle));
                nn::os::WaitEvent(&m_BleConnectionEvent);
            }
        }

        deviceCount = nn::btm::user::BleGetConnectionState(connectionState, NN_ARRAY_SIZE(connectionState));
    }

    for (auto state : connectionState)
    {
        NN_ABORT_UNLESS_EQUAL(state.connectionHandle, nn::bluetooth::BleInvalidConnectionHandle);
    }
}

void TestBtmModule_TestBase::UnpairAllBleDevices()
{
    nn::btm::user::BleAdvFilterForGeneral filter;
    nn::btm::BdAddress pairedAddress[nn::bluetooth::BlePairingCountMax];

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&filter, nn::bluetooth::BleScanParameterId_TestPeripheral));

    uint8_t pairingNum = nn::btm::user::BleGetPairedDevices(pairedAddress, NN_ARRAY_SIZE(pairedAddress), filter);

    for (int i = 0; i < pairingNum; ++i)
    {
        nn::os::ClearEvent(&m_BlePairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleUnPairDevice(pairedAddress[i], filter));
        nn::os::WaitEvent(&m_BlePairingEvent);
    }

    NN_ABORT_UNLESS_EQUAL(nn::btm::user::BleGetPairedDevices(pairedAddress, NN_ARRAY_SIZE(pairedAddress), filter), 0);
}

void TestBtmModule_TestBase::WaitBleConnection(uint8_t deviceCount)
{
    DisconnectAllBleDevices();
    UnpairAllBleDevices();

    // 呼ばれた時点のRadio の状態を確認
    bool isRadioEnabled = nn::btm::system::IsRadioEnabled();

    // Radio OFF なら一旦ON にする
    if (!isRadioEnabled)
    {
        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(true);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());
    }

    DisconnectAllBleDevices();
    UnpairAllBleDevices();

    nn::btm::user::BleAdvFilterForGeneral   filter;
    nn::btm::user::ScanResult               scanResult[nn::bluetooth::BleScanResultCountMax / 2];
    nn::btm::user::BleClientConnState       connState[nn::bluetooth::BleConnectionCountMaxClient];

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&filter, nn::bluetooth::BleScanParameterId_TestPeripheral));

    nn::os::ClearEvent(&m_BleScanEvent);

    while (NN_STATIC_CONDITION(true))
    {
        auto result = nn::btm::user::StartBleScanForGeneral(filter);

        if (nn::btm::user::ResultBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            break;
        }
    }

    NN_TEST_BTM_MODULE_LOG0("\n");
    NN_TEST_BTM_MODULE_LOG0("###### NEED USER OPERATION #####\n");
    NN_TEST_BTM_MODULE_LOG("Start BLE advertising on %d devices with a specific packet\n", deviceCount);

    while (NN_STATIC_CONDITION(true))
    {
        uint8_t connectionCount = nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState));
        bool    isAlreadyConnected = false;

        if (connectionCount == deviceCount)
        {
            while (NN_STATIC_CONDITION(true))
            {
                auto result = nn::btm::user::StopBleScanForGeneral();

                if (nn::btm::user::ResultBusy().Includes(result))
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    break;
                }
            }

            break;
        }

        nn::os::WaitEvent(&m_BleScanEvent);
        NN_ABORT_UNLESS_GREATER(nn::btm::user::GetBleScanResultsForGeneral(scanResult, NN_ARRAY_SIZE(scanResult)), 0);

        for (int i = 0; i < connectionCount; ++i)
        {
            if (scanResult[0].address == connState[i].address)
            {
                isAlreadyConnected = true;
                break;
            }
        }

        // すでに接続されているデバイスなら、接続を試行せず、Scan を続行
        if (!isAlreadyConnected)
        {
            nn::os::ClearEvent(&m_BleConnectionEvent);
            while (NN_STATIC_CONDITION(true))
            {
                auto result = nn::btm::user::BleConnect(scanResult[0].address);

                if (nn::btm::user::ResultBusyWorking().Includes(result) ||
                    nn::btm::ResultBusyWorking().Includes(result))
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
                }
                else
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    break;
                }

            }
            nn::os::WaitEvent(&m_BleConnectionEvent);
        }
    }

    NN_TEST_BTM_MODULE_LOG0("###### USER OPERATION DONE #####\n\n");

    // 元に戻す
    if (!isRadioEnabled)
    {
        DisableRadio();
    }
} // NOLINT(impl/function_size)

void TestBtmModule_TestBase::WaitGattClientCondition(uint8_t deviceCount,
                                                     bool largeCeLengthRequired, uint8_t largeCeLengthDeviceCount,
                                                     uint64_t durationMsec)
{
    nn::btm::GattClientConditionList gattClientConditionList;
    nn::os::TimerEventType timerEvent;
    nn::TimeSpan oneShotTime;

    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
    oneShotTime = nn::TimeSpan::FromMilliSeconds(durationMsec);
    nn::os::StartOneShotTimerEvent(&timerEvent, oneShotTime);

    while (NN_STATIC_CONDITION(true))
    {
        int largeCeLengthDeviceNum = 0;

        if (nn::os::TryWaitTimerEvent(&timerEvent))
        {
            NN_TEST_BTM_MODULE_LOG0("Timeout to wait GATT client condition list update as expected\n");

            NN_TEST_BTM_MODULE_LOG0("Expected GATT client condition is like below:\n");
            NN_TEST_BTM_MODULE_LOG("Device Count = %d, Large CE length required = %s, Large CE length Device Count = %d\n",
                deviceCount,
                largeCeLengthRequired ? "TRUE" : "FALSE",
                largeCeLengthDeviceCount);

            NN_TEST_BTM_MODULE_LOG0("Final GATT client condition list is like below:\n");
            NN_TEST_BTM_MODULE_LOG("Device Count = %d Large CE length required = %s, Large CE length Device Count = %d\n",
                gattClientConditionList.deviceCount,
                gattClientConditionList.isLargeCeLengthRequired ? "TRUE" : "FALSE",
                largeCeLengthDeviceNum);

            for (auto client : gattClientConditionList.gattClients)
            {
                NN_TEST_BTM_MODULE_LOG("    Address = %02X:%02X:%02X:%02X:%02X:%02X    CE Length = %d\n",
                    client.connectedServer.address.address[0], client.connectedServer.address.address[1], client.connectedServer.address.address[2],
                    client.connectedServer.address.address[3], client.connectedServer.address.address[4], client.connectedServer.address.address[5],
                    client.maxCeLength);
            }

            NN_ABORT();
        }

        nn::btm::BleGetGattClientConditionList(&gattClientConditionList);

        for (auto client : gattClientConditionList.gattClients)
        {
            if (client.maxCeLength == nn::btm::CeLength_8)
            {
                largeCeLengthDeviceNum++;
            }
        }

        // 満たすべき条件は以下
        if (gattClientConditionList.deviceCount             == deviceCount &&
            gattClientConditionList.isLargeCeLengthRequired == largeCeLengthRequired &&
            largeCeLengthDeviceCount                        == largeCeLengthDeviceNum)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3));
    }

    nn::os::FinalizeTimerEvent(&timerEvent);
}

void TestBtmModule_TestBase::DisableRadio()
{
    if (nn::btm::system::IsRadioEnabled())
    {
        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::EnableRadio(false);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(!nn::btm::system::IsRadioEnabled());
    }
}



void TestBtmModule_Init::ResetRadio()
{
    // 念のため、実行前にクリアしておく
    nn::os::ClearEvent(&m_RadioEvent);

    nn::btm::EnableRadio(true);
    nn::btm::BtmState btmState;
    nn::btm::GetState(&btmState);
    NN_ABORT_UNLESS_EQUAL(btmState, nn::btm::BtmState_Initialized);

    NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());
}

void TestBtmModule_Init::ResetGamepadPairing()
{
    // 念のため、実行前にクリアしておく
    nn::os::ClearEvent(&m_GamepadPairingEvent);

    nn::btm::system::CancelGamepadPairing();

    nn::os::WaitEvent(&m_GamepadPairingEvent);

    NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());
}

void TestBtmModule_Init::ResetDeviceInfo()
{
    DisconnectAllControllers();
    UnpairAllControllers();
}

void TestBtmModule_Init::ResetBle()
{
    DisconnectAllBleDevices();
    UnpairAllBleDevices();
}

void TestBtmModule_Init::ResetWlanMode()
{
    // 念のため、実行前にクリアしておく
    nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

    while (NN_STATIC_CONDITION(true))
    {
        auto result = nn::btm::SetWlanMode(nn::btm::WlanMode_None);

        if (nn::btm::ResultBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            break;
        }
    }

    nn::os::WaitEvent(&m_ConnectedDeviceConditionEvent);

    nn::btm::DeviceConditionList deviceConditionList;
    nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
    NN_ABORT_UNLESS_EQUAL(deviceConditionList.wlanMode, nn::btm::WlanMode_None);
}

void TestBtmModule_Init::ResetBluetoothMode()
{
    // 念のため、実行前にクリアしておく
    nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

    while (NN_STATIC_CONDITION(true))
    {
        auto result = nn::btm::SetBluetoothMode(nn::btm::BluetoothMode_Auto);

        if (nn::btm::ResultBusy().Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            break;
        }
    }

    nn::os::WaitEvent(&m_ConnectedDeviceConditionEvent);

    nn::btm::DeviceConditionList deviceConditionList;
    nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
    NN_ABORT_UNLESS_EQUAL(deviceConditionList.bluetoothMode, nn::btm::BluetoothMode_Auto);
}

void TestBtmModule_Init::ResetSlotSaving()
{
    // 念のため、実行前にクリアしておく
    nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

    nn::btm::EnableSlotSaving(false);

    nn::os::WaitEvent(&m_ConnectedDeviceConditionEvent);

    nn::btm::DeviceConditionList deviceConditionList;
    nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
    NN_ABORT_UNLESS(!deviceConditionList.isSlotSaving);
}

void TestBtmModule_Init::ResetDeviceInfoProtection()
{
    nn::btm::debug::GeneralTest(11);

    nn::btm::GeneralInfoList generalInfoList;
    nn::btm::debug::GeneralGet(0, &generalInfoList);

    NN_ABORT_UNLESS_EQUAL(generalInfoList.count, 0);
}
