﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Types.h>

#include <nn/btm/debug/btm_DebugApi.h>

#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>

#include <nn/btm/user/btm_UserApi.h>
#include <nn/btm/user/btm_UserTypes.h>
#include <nn/btm/user/btm_UserResult.h>

#include "testBtmModule_TestBase.h"
#include "TestBtmModule_ApiBtm.h"
#include "TestBtmModule_ApiBtmSystem.h"
#include "TestBtmModule_ApiBtmUser.h"
#include "testBtmModule_Usecase.h"

namespace
{
    // 実行対象のテストオブジェクトへのポインタ
    TestBtmModule_TestBase* gp_CurrentTest;

    // nn::btm
    nn::os::SystemEventType g_SystemEventForConnectedDeviceCondition;       // Joy-Con の接続状態（含ユースケース変更）が変化した際にシグナルされる
    nn::os::SystemEventType g_SystemEventForRegisterDeviceInfo;             // Joy-Con のペアリング情報が変化（追加、削除、ソート）した際にシグナルされる
    nn::os::SystemEventType g_SystemEventForLlrState;                       // Joy-Con からのHome 押し起床要求（LLR）の有効無効が切り替わった際にシグナルされる
    // 以下は nn::user でも取れるので、そちらで代用
    //nn::os::SystemEventType g_SystemEventForBleConnection;                  // BLE デバイスの接続状態が変化した際にシグナルされる
    //nn::os::SystemEventType g_SystemEventForBleServiceDiscovery;            // BLE デバイスのGATT Profile の収集が完了した際にシグナルされる
    //nn::os::SystemEventType g_SystemEventForBleMtuConfig;                   // BLE のMTU 設定が完了した際にシグナルされる

    // nn::btm::system
    nn::os::SystemEventType g_SystemEventForRadio;                          // 機内モードの状態が変化した際にシグナルされる
    nn::os::SystemEventType g_SystemEventForGamepadPairing;                 // Joy-Con のペアリングの有効無効が切り替わった際にシグナルされる

    // nn::btm::user
    nn::os::SystemEventType g_SystemEventForBleScan;                        // BLE アドバタイズパケット発見時にシグナルされる
    nn::os::SystemEventType g_SystemEventForBleConnection;                  // BLE デバイスの接続状態が変化した際にシグナルされる
    nn::os::SystemEventType g_SystemEventForBlePairing;                     // BLE ペアリングが完了した際にシグナルされる
    nn::os::SystemEventType g_SystemEventForBleServiceDiscovery;            // BLE デバイスのGATT Profile の収集が完了した際にシグナルされる
    nn::os::SystemEventType g_SystemEventForBleMtuConfig;                   // BLE のMTU 設定が完了した際にシグナルされる

    nn::os::MultiWaitType       g_MultiWait;

    // nn::btm
    nn::os::MultiWaitHolderType g_MultiWaitHolderForConnectedDeviceCondition;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForRegisterDeviceInfo;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForLlrState;

    // nn::btm::system
    nn::os::MultiWaitHolderType g_MultiWaitHolderForRadio;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForGamepadPairing;

    // nn::btm::user
    nn::os::MultiWaitHolderType g_MultiWaitHolderForBleScan;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForBleConnection;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForBlePairing;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForBleServiceDiscovery;
    nn::os::MultiWaitHolderType g_MultiWaitHolderForBleMtuConfig;

    typedef void(*Callback)();

    const size_t ThreadStackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK char g_MultiWaitThreadStack[ThreadStackSize];
    nn::os::ThreadType g_MultiWaitThread;
    void MultiWaitThreadFunction(void* arg);
    volatile bool exit = false;

    //!< ConnectedDeviceConditionEvent を処理
    void HandleConnectedDeviceConditionEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForConnectedDeviceCondition);
        gp_CurrentTest->HandleConnectedDeviceConditionEvent();
    }

    //!< RegisterDeviceInfoEvent を処理
    void HandleRegisterDeviceInfoEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForRegisterDeviceInfo);
        gp_CurrentTest->HandleRegisterDeviceInfoEvent();
    }

    //!< LlrStateEvent を処理
    void HandleLlrStateEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForLlrState);
        gp_CurrentTest->HandleLlrStateEvent();
    }

    //!< RadioEvent を処理
    void HandleRadioEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForRadio);
        gp_CurrentTest->HandleRadioEvent();
    }

    //!< GamepadPairingEvent を処理
    void HandleGamepadPairingEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForGamepadPairing);
        gp_CurrentTest->HandleGamepadPairingEvent();
    }

    //!< BleScanEvent を処理
    void HandleBleScanEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleScan);
        gp_CurrentTest->HandleBleScanEvent();
    }

    //!< BleConnectionEvent を処理
    void HandleBleConnectionEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleConnection);
        gp_CurrentTest->HandleBleConnectionEvent();
    }

    //!< BlePairingEvent を処理
    void HandleBlePairingEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBlePairing);
        gp_CurrentTest->HandleBlePairingEvent();
    }

    //!< BleServiceDiscoveryEvent を処理
    void HandleBleServiceDiscoveryEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleServiceDiscovery);
        gp_CurrentTest->HandleBleServiceDiscoveryEvent();
    }

    //!< BleMtuConfigEvent を処理
    void HandleBleMtuConfigEvent()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleMtuConfig);
        gp_CurrentTest->HandleBleMtuConfigEvent();
    }

    //!< テストの初期化処理
    void Initialize()
    {
        // IPC セッションの初期化
        nn::btm::InitializeBtmInterface();

        nn::btm::debug::InitializeBtmDebugInterface();

        nn::btm::user::InitializeBtmUserInterface();

        // システムイベントの取得
        nn::btm::RegisterSystemEventForConnectedDeviceCondition(&g_SystemEventForConnectedDeviceCondition);
        nn::btm::RegisterSystemEventForRegisteredDeviceInfo(&g_SystemEventForRegisterDeviceInfo);
        nn::btm::AcquireLlrStateEvent(&g_SystemEventForLlrState);

        nn::btm::system::AcquireRadioEvent(&g_SystemEventForRadio);
        nn::btm::system::AcquireGamepadPairingEvent(&g_SystemEventForGamepadPairing);

        nn::btm::user::AcquireBleScanEvent(&g_SystemEventForBleScan);
        nn::btm::user::AcquireBleConnectionEvent(&g_SystemEventForBleConnection);
        nn::btm::user::AcquireBlePairingEvent(&g_SystemEventForBlePairing);
        nn::btm::user::AcquireBleServiceDiscoveryEvent(&g_SystemEventForBleServiceDiscovery);
        nn::btm::user::AcquireBleMtuConfigEvent(&g_SystemEventForBleMtuConfig);

        // 多重待ちオブジェクトの初期化とリンク
        nn::os::InitializeMultiWait(&g_MultiWait);

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForConnectedDeviceCondition, &g_SystemEventForConnectedDeviceCondition);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForConnectedDeviceCondition, reinterpret_cast<uintptr_t>(HandleConnectedDeviceConditionEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForConnectedDeviceCondition);

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForRegisterDeviceInfo, &g_SystemEventForRegisterDeviceInfo);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForRegisterDeviceInfo, reinterpret_cast<uintptr_t>(HandleRegisterDeviceInfoEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForRegisterDeviceInfo);

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForLlrState, &g_SystemEventForLlrState);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForLlrState, reinterpret_cast<uintptr_t>(HandleLlrStateEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForLlrState);



        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForRadio, &g_SystemEventForRadio);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForRadio, reinterpret_cast<uintptr_t>(HandleRadioEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForRadio);

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForGamepadPairing, &g_SystemEventForGamepadPairing);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForGamepadPairing, reinterpret_cast<uintptr_t>(HandleGamepadPairingEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForGamepadPairing);



        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForBleScan, &g_SystemEventForBleScan);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForBleScan, reinterpret_cast<uintptr_t>(HandleBleScanEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForBleScan);

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForBleConnection, &g_SystemEventForBleConnection);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForBleConnection, reinterpret_cast<uintptr_t>(HandleBleConnectionEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForBleConnection);

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForBlePairing, &g_SystemEventForBlePairing);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForBlePairing, reinterpret_cast<uintptr_t>(HandleBlePairingEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForBlePairing);

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForBleServiceDiscovery, &g_SystemEventForBleServiceDiscovery);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForBleServiceDiscovery, reinterpret_cast<uintptr_t>(HandleBleServiceDiscoveryEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForBleServiceDiscovery);

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolderForBleMtuConfig, &g_SystemEventForBleMtuConfig);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolderForBleMtuConfig, reinterpret_cast<uintptr_t>(HandleBleMtuConfigEvent));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolderForBleMtuConfig);

        // 多重待ちスレッドを生成、起動
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_MultiWaitThread, MultiWaitThreadFunction, nullptr, g_MultiWaitThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_MultiWaitThread);
    }

    //!< テストの終了処理
    void Finalize()
    {
        exit = true;

        // スレッドの終了を待つ
        nn::os::WaitThread(&g_MultiWaitThread);

        // IPC セッションの末期化
        nn::btm::FinalizeBtmInterface();

        nn::btm::debug::FinalizeBtmDebugInterface();

        nn::btm::user::FinalizeBtmUserInterface();

        // 多重待ちオブジェクトのアンリンクと破棄
        nn::os::UnlinkAllMultiWaitHolder(&g_MultiWait);
        nn::os::FinalizeMultiWait(&g_MultiWait);

        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForConnectedDeviceCondition);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForRegisterDeviceInfo);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForLlrState);

        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForRadio);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForGamepadPairing);

        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForBleScan);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForBleConnection);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForBlePairing);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForBleServiceDiscovery);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolderForBleMtuConfig);

        // システムイベントの破棄
        nn::os::DestroySystemEvent(&g_SystemEventForConnectedDeviceCondition);
        nn::os::DestroySystemEvent(&g_SystemEventForRegisterDeviceInfo);
        nn::os::DestroySystemEvent(&g_SystemEventForLlrState);

        nn::os::DestroySystemEvent(&g_SystemEventForRadio);
        nn::os::DestroySystemEvent(&g_SystemEventForGamepadPairing);

        nn::os::DestroySystemEvent(&g_SystemEventForBleScan);
        nn::os::DestroySystemEvent(&g_SystemEventForBleConnection);
        nn::os::DestroySystemEvent(&g_SystemEventForBlePairing);
        nn::os::DestroySystemEvent(&g_SystemEventForBleServiceDiscovery);
        nn::os::DestroySystemEvent(&g_SystemEventForBleMtuConfig);
    }

    void ClearSystemEvents()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForConnectedDeviceCondition);
        nn::os::ClearSystemEvent(&g_SystemEventForRegisterDeviceInfo);
        nn::os::ClearSystemEvent(&g_SystemEventForLlrState);

        nn::os::ClearSystemEvent(&g_SystemEventForRadio);
        nn::os::ClearSystemEvent(&g_SystemEventForGamepadPairing);

        nn::os::ClearSystemEvent(&g_SystemEventForBleScan);
        nn::os::ClearSystemEvent(&g_SystemEventForBleConnection);
        nn::os::ClearSystemEvent(&g_SystemEventForBlePairing);
        nn::os::ClearSystemEvent(&g_SystemEventForBleServiceDiscovery);
        nn::os::ClearSystemEvent(&g_SystemEventForBleMtuConfig);
    }

    //!< BTM からのイベントを待ち受けるスレッド関数
    void MultiWaitThreadFunction(void* arg)
    {
        NN_UNUSED(arg);

        while (!exit)
        {
            nn::os::MultiWaitHolderType* pHolder = nn::os::TryWaitAny(&g_MultiWait);

            if (pHolder != nullptr)
            {
                auto callback = reinterpret_cast<Callback>(nn::os::GetMultiWaitHolderUserData(pHolder));

                callback();
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

    //!< BTM のユースケースを初期化
    void ResetTestCondition()
    {
        NN_TEST_BTM_MODULE_LOG0("Reset test condition\n");

        TestBtmModule_Init testInit;

        gp_CurrentTest = &testInit;

        ClearSystemEvents();
        testInit.ResetRadio();

        ClearSystemEvents();
        testInit.ResetGamepadPairing();

        ClearSystemEvents();
        testInit.ResetDeviceInfo();

        ClearSystemEvents();
        testInit.ResetBle();

        ClearSystemEvents();
        testInit.ResetWlanMode();

        ClearSystemEvents();
        testInit.ResetBluetoothMode();

        ClearSystemEvents();
        testInit.ResetSlotSaving();

        ClearSystemEvents();
        testInit.ResetDeviceInfoProtection();

        NN_TEST_BTM_MODULE_LOG0("Complete\n\n");
    }

    void TestApiBtm()
    {
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!\n");
        NN_TEST_BTM_MODULE_LOG0("Test nn::btm APIs Start\n");
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!\n\n");

        ApiBtm::TestBtmModule_ApiBtm testApiBtm;

        gp_CurrentTest = &testApiBtm;

        testApiBtm.TestInterface();

        testApiBtm.TestBtmState(nn::btm::BtmState_Initialized);

        const char* deviceName                  = "Nintendo Switch";
        nn::btm::ClassOfDevice classOfDevice    = { { 0x00, 0x04, 0x3C } };
        uint8_t featureSet                      = 0x68;
        testApiBtm.TestHostDeviceProperty(deviceName, sizeof(deviceName) + 1, classOfDevice, featureSet);

        ClearSystemEvents();
        testApiBtm.TestDeviceInfo();

        ClearSystemEvents();
        testApiBtm.TestMode();

        ClearSystemEvents();
        testApiBtm.TestEnableSlotSaving();

        ClearSystemEvents();
        testApiBtm.TestEnableRadio();

        ClearSystemEvents();
        testApiBtm.TestLlrNotify();

        ClearSystemEvents();
        testApiBtm.TestBtmStateRadioOff(nn::btm::BtmState_RadioOff);

        ClearSystemEvents();
        testApiBtm.TestHostDevicePropertyRadioOff(deviceName, sizeof(deviceName) + 1, classOfDevice, featureSet);

        ClearSystemEvents();
        testApiBtm.TestDeviceInfoRadioOff();

        ClearSystemEvents();
        testApiBtm.TestModeRadioOff();

        ClearSystemEvents();
        testApiBtm.TestEnableSlotSavingRadioOff();

        ClearSystemEvents();
        testApiBtm.TestLlrNotifyRadioOff();

        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
        NN_TEST_BTM_MODULE_LOG0("Test nn::btm APIs Complete\n");
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n");
    }

    void TestApiBtmSystem()
    {
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
        NN_TEST_BTM_MODULE_LOG0("Test nn::btm::system APIs Start\n");
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n");

        ApiBtmSystem::TestBtmModule_ApiBtmSystem testApiBtmSyste;

        gp_CurrentTest = &testApiBtmSyste;

        ClearSystemEvents();
        testApiBtmSyste.TestGamepadPairing();

        ClearSystemEvents();
        testApiBtmSyste.TestRadio();

        ClearSystemEvents();
        testApiBtmSyste.TestGamepadPairingRadioOff();

        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
        NN_TEST_BTM_MODULE_LOG0("Test nn::btm::system APIs Complete\n");
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n");
    }

    void TestApiBtmUser()
    {
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
        NN_TEST_BTM_MODULE_LOG0("Test nn::btm::user APIs Start\n");
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n");

        ApiBtmUser::TestBtmModule_ApiBtmUser testApiBtmUser;

        gp_CurrentTest = &testApiBtmUser;

        ClearSystemEvents();
        testApiBtmUser.TestBleScan();

        ClearSystemEvents();
        testApiBtmUser.TestBleConnect();

        ClearSystemEvents();
        testApiBtmUser.TestBleServiceDiscovery();

        ClearSystemEvents();
        testApiBtmUser.TestBlePairing();

        ClearSystemEvents();
        testApiBtmUser.TestBleMtuConfig();

        ClearSystemEvents();
        testApiBtmUser.TestBleDataPath();

        ClearSystemEvents();
        testApiBtmUser.TestBleScanRadioOff();

        ClearSystemEvents();
        testApiBtmUser.TestBleConnectRadioOff();

        ClearSystemEvents();
        testApiBtmUser.TestBleServiceDiscoveryRadioOff();

        ClearSystemEvents();
        testApiBtmUser.TestBlePairingRadioOff();

        ClearSystemEvents();
        testApiBtmUser.TestBleMtuConfigRadioOff();

        ClearSystemEvents();
        testApiBtmUser.TestBleDataPathRadioOff();

        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
        NN_TEST_BTM_MODULE_LOG0("Test nn::btm::user APIs Complete\n");
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n\n");
    }

    void TestBtmUsecase()
    {
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!\n");
        NN_TEST_BTM_MODULE_LOG0("Test Usecase Start\n");
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!\n\n");

        BtmUsecase::TestBtmModule_Usecase testUsecase;

        gp_CurrentTest = &testUsecase;

        ClearSystemEvents();
        testUsecase.TestDynamic2Slot2();

        ClearSystemEvents();
        testUsecase.TestDynamic2Slot6();

        ClearSystemEvents();
        testUsecase.TestDynamic2Slot8();

        ClearSystemEvents();
        testUsecase.TestStaticJoy4();

        ClearSystemEvents();
        testUsecase.TestStaticJoy8();

        ClearSystemEvents();
        testUsecase.TestActive();

        ClearSystemEvents();
        testUsecase.TestStaticJoy2Local8();

        ClearSystemEvents();
        testUsecase.TestStaticJoy4Local4();

        ClearSystemEvents();
        testUsecase.TestBle1();

        ClearSystemEvents();
        testUsecase.TestBle2();

        ClearSystemEvents();
        testUsecase.TestBle3();

        ClearSystemEvents();
        testUsecase.TestPrintAllUsecase();

        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!\n");
        NN_TEST_BTM_MODULE_LOG0("Test Usecase Complete\n");
        NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!\n\n");
    }
}   // namespace

extern "C" void nnMain()
{
    NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!\n");
    NN_TEST_BTM_MODULE_LOG0("!!!!!! STARTED !!!!!!\n");
    NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!\n\n");

    // 初期化
    Initialize();

    // nn::btm のAPI のテスト
    ResetTestCondition();
    TestApiBtm();

    // nn::btm::system のAPI のテスト
    ResetTestCondition();
    TestApiBtmSystem();

    // nn::btm::user のAPI のテスト
    ResetTestCondition();
    TestApiBtmUser();

    // Usecase のテスト
    ResetTestCondition();
    TestBtmUsecase();

    // 末期化
    Finalize();

    NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!\n");
    NN_TEST_BTM_MODULE_LOG0("!!!!! COMPLETED !!!!!\n");
    NN_TEST_BTM_MODULE_LOG0("!!!!!!!!!!!!!!!!!!!!!\n\n");
}
