﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include <nn/btm/user/btm_UserApi.h>
#include <nn/btm/user/btm_UserTypes.h>
#include <nn/btm/user/btm_UserResult.h>

#include <nn/bluetooth/bluetooth_BleScanParameterIdTestPeripheral.h>

#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Types.h>

#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>

#include "TestBtmModule_ApiBtmUser.h"

namespace ApiBtmUser
{
    void TestBtmModule_ApiBtmUser::TestBleScan()
    {
        DisconnectAllBleDevices();
        UnpairAllBleDevices();

        nn::btm::user::BleAdvFilterForGeneral       filterForBleDevice;
        nn::btm::user::BleAdvFilterForSmartDevice   filterForSmartDevice;
        nn::btm::user::ScanResult                   scanResult[10];

        NN_TEST_BTM_MODULE_LOG("%s: Get valid scan parameter\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&filterForBleDevice, nn::bluetooth::BleScanParameterId_TestPeripheral));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for general device\n", NN_CURRENT_FUNCTION_NAME);

        memset(filterForBleDevice.manufacturerId, 0xFF, NN_ARRAY_SIZE(filterForBleDevice.manufacturerId));
        memset(filterForBleDevice.clientId, 0xFF, NN_ARRAY_SIZE(filterForBleDevice.clientId));
        memset(filterForBleDevice.serverId, 0xFF, NN_ARRAY_SIZE(filterForBleDevice.serverId));

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(filterForBleDevice));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for general device\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for general device for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(filterForBleDevice));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanGeneral(filterForBleDevice));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for general device for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanGeneral());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get BLE Scan retults for general device\n", NN_CURRENT_FUNCTION_NAME);

        uint8_t resultNum = nn::btm::user::GetBleScanResultsForGeneral(scanResult, NN_ARRAY_SIZE(scanResult));

        NN_TEST_BTM_MODULE_LOG("%s: Found %d devices\n", NN_CURRENT_FUNCTION_NAME, resultNum);
        for (auto result : scanResult)
        {
            NN_TEST_BTM_MODULE_LOG("%s:     Addresss = %02X:%02X:%02X:%02X:%02X:%02X, RSSI = %d\n",
                NN_CURRENT_FUNCTION_NAME,
                result.address.address[0], result.address.address[0], result.address.address[0],
                result.address.address[0], result.address.address[0], result.address.address[0],
                result.rssi);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);



        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for smart device\n", NN_CURRENT_FUNCTION_NAME);

        filterForSmartDevice.length = nn::bluetooth::GattAttributeUuidLength_128;
        memset(filterForSmartDevice.uu.uuid128, 0xFF, NN_ARRAY_SIZE(filterForSmartDevice.uu.uuid128));

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanSmartDevice(filterForSmartDevice));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for smart device\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanSmartDevicel());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for smart device for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanSmartDevice(filterForSmartDevice));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanSmartDevice(filterForSmartDevice));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for smart device for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanSmartDevicel());
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanSmartDevicel());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get BLE Scan retults for smart device\n", NN_CURRENT_FUNCTION_NAME);

        resultNum = nn::btm::user::GetBleScanResultsForSmartDevice(scanResult, NN_ARRAY_SIZE(scanResult));

        NN_TEST_BTM_MODULE_LOG("%s: Found %d devices\n", NN_CURRENT_FUNCTION_NAME, resultNum);
        for (auto result : scanResult)
        {
            NN_TEST_BTM_MODULE_LOG("%s:     Addresss = %02X:%02X:%02X:%02X:%02X:%02X, RSSI = %d\n",
                NN_CURRENT_FUNCTION_NAME,
                result.address.address[0], result.address.address[0], result.address.address[0],
                result.address.address[0], result.address.address[0], result.address.address[0],
                result.rssi);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);



        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for paired general device\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanPaired(filterForBleDevice));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for paired general device\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanPaired());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for paired general evice for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanPaired(filterForBleDevice));
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanPaired(filterForBleDevice));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for paired general device for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanPaired());
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanPaired());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtmUser::TestBleConnect()
    {
        nn::btm::user::BleClientConnState   connState[nn::bluetooth::BleConnectionCountMaxClient];
        uint32_t                            connectionHandle;       // 保存用

        NN_TEST_BTM_MODULE_LOG("%s: Connect to a BLE device\n", NN_CURRENT_FUNCTION_NAME);

        WaitBleConnection(1);
        WaitGattClientCondition(1, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get BLE connection state\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_GREATER(nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState)), 0);

        for (auto state : connState)
        {
            if (state.connectionHandle != nn::bluetooth::BleInvalidConnectionHandle)
            {
                NN_TEST_BTM_MODULE_LOG("%s:    Address = %02X:%02X:%02X:%02X:%02X:%02X, Handle = %d\n",
                    NN_CURRENT_FUNCTION_NAME,
                    state.address.address[0], state.address.address[1], state.address.address[2],
                    state.address.address[3], state.address.address[4], state.address.address[5],
                    state.connectionHandle);
            }
        }

        connectionHandle = connState[0].connectionHandle;

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Connect to a connected BLE device\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS(nn::btm::user::ResultBleConnectionAlreadyExist().Includes(nn::btm::user::BleConnect(connState[0].address)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Disconnect from a BLE device\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_BleConnectionEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleDisconnect(connState[0].connectionHandle));
        nn::os::WaitEvent(&m_BleConnectionEvent);

        WaitGattClientCondition(0, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get BLE connection state\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_EQUAL(nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState)), 0);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Disconnect from an unconnected BLE device\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS(nn::btm::user::ResultBleConnectionNotFound().Includes(nn::btm::user::BleDisconnect(connectionHandle)));
        NN_ABORT_UNLESS(nn::btm::user::ResultBleConnectionNotFound().Includes(nn::btm::user::BleDisconnect(nn::bluetooth::BleInvalidConnectionHandle)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtmUser::TestBleServiceDiscovery()
    {
        nn::btm::user::BleClientConnState   connState[nn::bluetooth::BleConnectionCountMaxClient];

        nn::btm::user::GattService          services[nn::bluetooth::GattAttributeCountMaxClient];
        nn::btm::user::GattService          includedeServices[nn::bluetooth::GattAttributeCountMaxClient];
        nn::btm::user::GattCharacteristic   characteristics[nn::bluetooth::GattAttributeCountMaxClient];
        nn::btm::user::GattDescriptor       descriptors[nn::bluetooth::GattAttributeCountMaxClient];

        nn::btm::user::GattService          genericAttributeService;
        nn::bluetooth::GattAttributeUuid    genericAttributeServiceUuid =
        {
            .length = nn::bluetooth::GattAttributeUuidLength_16,
            .uu.uuid16 = 0x1801
        };

        nn::os::ClearEvent(&m_BleServiceDiscoveryEvent);
        WaitBleConnection(1);
        nn::os::WaitEvent(&m_BleServiceDiscoveryEvent);

        WaitGattClientCondition(1, false, 0, 10000);

        NN_ABORT_UNLESS_GREATER(nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState)), 0);

        NN_TEST_BTM_MODULE_LOG("%s: Get GATT services\n", NN_CURRENT_FUNCTION_NAME);

        uint8_t serviceNum = nn::btm::user::GetGattServices(services, NN_ARRAY_SIZE(services), connState[0].connectionHandle);

        for (int i = 0; i < serviceNum; ++i)
        {
            NN_TEST_BTM_MODULE_LOG("%s: Handle = %d, UUID = %02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:\n",
                NN_CURRENT_FUNCTION_NAME,
                services[i].handle,
                services[i].uuid.uu.uuid128[0], services[i].uuid.uu.uuid128[1], services[i].uuid.uu.uuid128[2], services[i].uuid.uu.uuid128[3],
                services[i].uuid.uu.uuid128[4], services[i].uuid.uu.uuid128[5], services[i].uuid.uu.uuid128[6], services[i].uuid.uu.uuid128[7],
                services[i].uuid.uu.uuid128[8], services[i].uuid.uu.uuid128[9], services[i].uuid.uu.uuid128[10], services[i].uuid.uu.uuid128[11],
                services[i].uuid.uu.uuid128[12], services[i].uuid.uu.uuid128[13], services[i].uuid.uu.uuid128[14], services[i].uuid.uu.uuid128[15]);
        }
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get Generic Attribute service\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS(nn::btm::user::GetGattService(&genericAttributeService, connState[0].connectionHandle, genericAttributeServiceUuid));
        NN_ABORT_UNLESS_EQUAL(genericAttributeService.uuid, genericAttributeServiceUuid);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get GATT included services\n", NN_CURRENT_FUNCTION_NAME);

        uint8_t includedServiceNum = 0;
        for (int i = 0; i < serviceNum; ++i)
        {
            includedServiceNum += nn::btm::user::GetGattIncludedServices(&includedeServices[includedServiceNum],
                                                                         NN_ARRAY_SIZE(includedeServices) - includedServiceNum,
                                                                         connState[0].connectionHandle, services[i].handle);
        }

        for (int i = 0; i < includedServiceNum; ++i)
        {
            NN_TEST_BTM_MODULE_LOG("%s: Handle = %d, UUID = %02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:\n",
                NN_CURRENT_FUNCTION_NAME,
                includedeServices[i].handle,
                includedeServices[i].uuid.uu.uuid128[0], includedeServices[i].uuid.uu.uuid128[1], includedeServices[i].uuid.uu.uuid128[2], includedeServices[i].uuid.uu.uuid128[3],
                includedeServices[i].uuid.uu.uuid128[4], includedeServices[i].uuid.uu.uuid128[5], includedeServices[i].uuid.uu.uuid128[6], includedeServices[i].uuid.uu.uuid128[7],
                includedeServices[i].uuid.uu.uuid128[8], includedeServices[i].uuid.uu.uuid128[9], includedeServices[i].uuid.uu.uuid128[10], includedeServices[i].uuid.uu.uuid128[11],
                includedeServices[i].uuid.uu.uuid128[12], includedeServices[i].uuid.uu.uuid128[13], includedeServices[i].uuid.uu.uuid128[14], includedeServices[i].uuid.uu.uuid128[15]);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get GATT characteristics\n", NN_CURRENT_FUNCTION_NAME);

        uint8_t characteristicNum = 0;
        for (int i = 0; i < serviceNum; ++i)
        {
            characteristicNum += nn::btm::user::GetGattCharacteristics(&characteristics[characteristicNum],
                                                                       NN_ARRAY_SIZE(characteristics) - characteristicNum,
                                                                       connState[0].connectionHandle, services[i].handle);
        }

        for (int i = 0; i < characteristicNum; ++i)
        {
            NN_TEST_BTM_MODULE_LOG("%s: Handle = %d, UUID = %02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:\n",
                NN_CURRENT_FUNCTION_NAME,
                characteristics[i].handle,
                characteristics[i].uuid.uu.uuid128[0], characteristics[i].uuid.uu.uuid128[1], characteristics[i].uuid.uu.uuid128[2], characteristics[i].uuid.uu.uuid128[3],
                characteristics[i].uuid.uu.uuid128[4], characteristics[i].uuid.uu.uuid128[5], characteristics[i].uuid.uu.uuid128[6], characteristics[i].uuid.uu.uuid128[7],
                characteristics[i].uuid.uu.uuid128[8], characteristics[i].uuid.uu.uuid128[9], characteristics[i].uuid.uu.uuid128[10], characteristics[i].uuid.uu.uuid128[11],
                characteristics[i].uuid.uu.uuid128[12], characteristics[i].uuid.uu.uuid128[13], characteristics[i].uuid.uu.uuid128[14], characteristics[i].uuid.uu.uuid128[15]);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get GATT descriptors\n", NN_CURRENT_FUNCTION_NAME);

        uint8_t descriptorNum = 0;
        for (int i = 0; i < characteristicNum; ++i)
        {
            descriptorNum += nn::btm::user::GetGattDescriptors(&descriptors[descriptorNum],
                                                               NN_ARRAY_SIZE(descriptors) - descriptorNum,
                                                               connState[0].connectionHandle, characteristics[i].handle);
        }

        for (int i = 0; i < descriptorNum; ++i)
        {
            NN_TEST_BTM_MODULE_LOG("%s: Handle = %d, UUID = %02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:\n",
                NN_CURRENT_FUNCTION_NAME,
                descriptors[i].handle,
                descriptors[i].uuid.uu.uuid128[0], descriptors[i].uuid.uu.uuid128[1], descriptors[i].uuid.uu.uuid128[2], descriptors[i].uuid.uu.uuid128[3],
                descriptors[i].uuid.uu.uuid128[4], descriptors[i].uuid.uu.uuid128[5], descriptors[i].uuid.uu.uuid128[6], descriptors[i].uuid.uu.uuid128[7],
                descriptors[i].uuid.uu.uuid128[8], descriptors[i].uuid.uu.uuid128[9], descriptors[i].uuid.uu.uuid128[10], descriptors[i].uuid.uu.uuid128[11],
                descriptors[i].uuid.uu.uuid128[12], descriptors[i].uuid.uu.uuid128[13], descriptors[i].uuid.uu.uuid128[14], descriptors[i].uuid.uu.uuid128[15]);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get GATT services of an unconnected device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS_EQUAL(nn::btm::user::GetGattServices(services, NN_ARRAY_SIZE(services), nn::bluetooth::BleInvalidConnectionHandle), 0);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get GATT included services of an unconnected device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS_EQUAL(nn::btm::user::GetGattIncludedServices(includedeServices, NN_ARRAY_SIZE(includedeServices),
                              nn::bluetooth::BleInvalidConnectionHandle, nn::bluetooth::GattAttributeInvalidHandle), 0);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get GATT characteristics of an unconnected device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS_EQUAL(nn::btm::user::GetGattCharacteristics(characteristics, NN_ARRAY_SIZE(characteristics),
                              nn::bluetooth::BleInvalidConnectionHandle, nn::bluetooth::GattAttributeInvalidHandle), 0);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get GATT descriptors of an unconnected device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS_EQUAL(nn::btm::user::GetGattDescriptors(descriptors, NN_ARRAY_SIZE(descriptors),
                              nn::bluetooth::BleInvalidConnectionHandle, nn::bluetooth::GattAttributeInvalidHandle), 0);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_ApiBtmUser::TestBlePairing()
    {
        nn::btm::user::BleAdvFilterForGeneral   filterForBleDevice;
        nn::btm::user::BleClientConnState       connState[nn::bluetooth::BleConnectionCountMaxClient];
        nn::btm::BdAddress                      pairedAddress[nn::bluetooth::BlePairingCountMax];

        nn::btm::user::GattService          nnGattPairingService;
        nn::bluetooth::GattAttributeUuid    nnGattPairingServiceUuid =
        {
            .length     = nn::bluetooth::GattAttributeUuidLength_128,
            .uu.uuid128 = { 0x6C, 0x1C, 0xE0, 0xB4, 0x5D, 0x73, 0x6A, 0x8A, 0x1A, 0x4C, 0xA5, 0x4A, 0x26, 0x3E, 0xDC, 0xAD }
        };

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&filterForBleDevice, nn::bluetooth::BleScanParameterId_TestPeripheral));

        nn::os::ClearEvent(&m_BleServiceDiscoveryEvent);
        WaitBleConnection(1);
        nn::os::WaitEvent(&m_BleServiceDiscoveryEvent);

        WaitGattClientCondition(1, false, 0, 10000);

        NN_ABORT_UNLESS_GREATER(nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState)), 0);
        NN_ABORT_UNLESS(nn::btm::user::GetGattService(&nnGattPairingService, connState[0].connectionHandle, nnGattPairingServiceUuid));

        NN_TEST_BTM_MODULE_LOG("%s: Pair with a connected device\n", NN_CURRENT_FUNCTION_NAME);

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::ClearEvent(&m_BlePairingEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BlePairDevice(connState[0].connectionHandle, filterForBleDevice));
            nn::os::WaitEvent(&m_BlePairingEvent);

            if (nn::btm::user::BleGetPairedDevices(pairedAddress, NN_ARRAY_SIZE(pairedAddress), filterForBleDevice) > 0)
            {
                NN_ABORT_UNLESS_EQUAL(pairedAddress[0], connState[0].address);
                break;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Pairing aborted. Try again.\n", NN_CURRENT_FUNCTION_NAME);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllBleDevices();
        WaitGattClientCondition(0, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Enable BLE auto connection\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_BleConnectionEvent);
        nn::os::ClearEvent(&m_BleServiceDiscoveryEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStartBleScanPaired(filterForBleDevice));
        nn::os::WaitEvent(&m_BleConnectionEvent);
        nn::os::WaitEvent(&m_BleServiceDiscoveryEvent);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryStopBleScanPaired());

        NN_ABORT_UNLESS_EQUAL(nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState)), 1);
        NN_ABORT_UNLESS_EQUAL(pairedAddress[0], connState[0].address);

        WaitGattClientCondition(1, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Unpair with a paired BLE with connection\n", NN_CURRENT_FUNCTION_NAME);

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::ClearEvent(&m_BlePairingEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleUnPairDevice(connState[0].connectionHandle, filterForBleDevice));
            nn::os::WaitEvent(&m_BlePairingEvent);

            if (nn::btm::user::BleGetPairedDevices(pairedAddress, NN_ARRAY_SIZE(pairedAddress), filterForBleDevice) == 0)
            {
                break;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Unpairing aborted. Try again.\n", NN_CURRENT_FUNCTION_NAME);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Unpair with a paired BLE without connection\n", NN_CURRENT_FUNCTION_NAME);

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::ClearEvent(&m_BlePairingEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BlePairDevice(connState[0].connectionHandle, filterForBleDevice));
            nn::os::WaitEvent(&m_BlePairingEvent);

            if (nn::btm::user::BleGetPairedDevices(pairedAddress, NN_ARRAY_SIZE(pairedAddress), filterForBleDevice) > 0)
            {
                NN_ABORT_UNLESS_EQUAL(pairedAddress[0], connState[0].address);
                break;
            }

            NN_TEST_BTM_MODULE_LOG("%s: Pairing aborted. Try again.\n", NN_CURRENT_FUNCTION_NAME);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }

        DisconnectAllBleDevices();
        WaitGattClientCondition(0, false, 0, 10000);

        nn::os::ClearEvent(&m_BlePairingEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleUnPairDevice(pairedAddress[0], filterForBleDevice));
        nn::os::WaitEvent(&m_BlePairingEvent);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Pair with an unconnected device\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_BlePairingEvent);
        NN_ABORT_UNLESS(nn::btm::user::ResultBleConnectionNotFound().Includes(nn::btm::user::BlePairDevice(nn::bluetooth::BleInvalidConnectionHandle, filterForBleDevice)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Pair with a connected device for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_BleServiceDiscoveryEvent);
        WaitBleConnection(1);
        nn::os::WaitEvent(&m_BleServiceDiscoveryEvent);

        WaitGattClientCondition(1, false, 0, 10000);

        NN_ABORT_UNLESS_GREATER(nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState)), 0);
        NN_ABORT_UNLESS(nn::btm::user::GetGattService(&nnGattPairingService, connState[0].connectionHandle, nnGattPairingServiceUuid));

        for (int i = 0; i < 3; ++i)
        {
            nn::os::ClearEvent(&m_BlePairingEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BlePairDevice(connState[0].connectionHandle, filterForBleDevice));
            nn::os::WaitEvent(&m_BlePairingEvent);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Unpair with a connected device for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        for(int i = 0; i < 3; ++i)
        {
            nn::os::ClearEvent(&m_BlePairingEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::BleUnPairDevice(connState[0].connectionHandle, filterForBleDevice));
            nn::os::WaitEvent(&m_BlePairingEvent);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllBleDevices();
        UnpairAllBleDevices();
    } // NOLINT(impl/function_size)

    void TestBtmModule_ApiBtmUser::TestBleMtuConfig()
    {
        nn::btm::user::BleClientConnState       connState[nn::bluetooth::BleConnectionCountMaxClient];

        nn::os::ClearEvent(&m_BleServiceDiscoveryEvent);
        WaitBleConnection(1);
        nn::os::WaitEvent(&m_BleServiceDiscoveryEvent);

        WaitGattClientCondition(1, false, 0, 10000);

        NN_ABORT_UNLESS_GREATER(nn::btm::user::BleGetConnectionState(connState, NN_ARRAY_SIZE(connState)), 0);

        NN_TEST_BTM_MODULE_LOG("%s: Configure BLE MTU for a connected device\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_BleMtuConfigEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryConfigureBleMtu(connState[0].connectionHandle, 256));
        nn::os::WaitEvent(&m_BleMtuConfigEvent);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Get BLE MTU\n", NN_CURRENT_FUNCTION_NAME);
        NN_TEST_BTM_MODULE_LOG("%s: MTU = %d\n", NN_CURRENT_FUNCTION_NAME, nn::btm::user::GetBleMtu(connState[0].connectionHandle));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllBleDevices();

        NN_TEST_BTM_MODULE_LOG("%s: Configure BLE MTU for an unconnected device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS(nn::btm::user::ResultBleConnectionNotFound().Includes(TryConfigureBleMtu(connState[0].connectionHandle, 256)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllBleDevices();
    }

    void TestBtmModule_ApiBtmUser::TestBleDataPath()
    {
        nn::bluetooth::GattAttributeUuid testUuids[nn::bluetooth::GATT_DATA_PATH_FILTER_NUM_MAX + 1];

        for (int i = 0; i < NN_ARRAY_SIZE(testUuids); ++i)
        {
            testUuids[i].length = nn::bluetooth::GattAttributeUuidLength_128;

            for (int j = 0; j < nn::bluetooth::GattAttributeUuidLength_128; ++j)
            {
                testUuids[i].uu.uuid128[j] = j;
            }

            testUuids[i].uu.uuid128[nn::bluetooth::GattAttributeUuidLength_128 - 1] = i;
        }

        NN_TEST_BTM_MODULE_LOG("%s: Register a Gatt operation data path\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryRegisterGattDataPath(testUuids[0], true));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Register a Gatt operation data path twice\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryRegisterGattDataPath(testUuids[0], true));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Unregister a Gatt operation data path\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS_RESULT_SUCCESS(TryRegisterGattDataPath(testUuids[0], false));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Register Gatt operation data path up to maximum\n", NN_CURRENT_FUNCTION_NAME);

        for (auto uuid : testUuids)
        {
            auto result = TryRegisterGattDataPath(uuid, true);

            if (nn::btm::user::ResultBleGattDataPathRegisterFailed().Includes(result))
            {
                break;
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Unregister all Gatt operation data path\n", NN_CURRENT_FUNCTION_NAME);

        for (auto uuid : testUuids)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(TryRegisterGattDataPath(uuid, false));
        }
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtmUser::TestBleScanRadioOff()
    {
        DisableRadio();

        DisconnectAllBleDevices();
        UnpairAllBleDevices();

        nn::btm::user::BleAdvFilterForGeneral       filterForBleDevice;
        nn::btm::user::BleAdvFilterForSmartDevice   filterForSmartDevice;

        // 正常動作
        NN_TEST_BTM_MODULE_LOG("%s: Get valid scan parameter\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::btm::user::GetBleScanFilterParameter(&filterForBleDevice, nn::bluetooth::BleScanParameterId_TestPeripheral));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for general device\n", NN_CURRENT_FUNCTION_NAME);

        memset(filterForBleDevice.manufacturerId, 0xFF, NN_ARRAY_SIZE(filterForBleDevice.manufacturerId));
        memset(filterForBleDevice.clientId, 0xFF, NN_ARRAY_SIZE(filterForBleDevice.clientId));
        memset(filterForBleDevice.serverId, 0xFF, NN_ARRAY_SIZE(filterForBleDevice.serverId));

        NN_ABORT_UNLESS(nn::btm::user::ResultBluetoothOff().Includes(TryStartBleScanGeneral(filterForBleDevice)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for general device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS(nn::btm::user::ResultBluetoothOff().Includes(TryStopBleScanGeneral()));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for smart device\n", NN_CURRENT_FUNCTION_NAME);

        filterForSmartDevice.length = nn::bluetooth::GattAttributeUuidLength_128;
        memset(filterForSmartDevice.uu.uuid128, 0xFF, NN_ARRAY_SIZE(filterForSmartDevice.uu.uuid128));

        NN_ABORT_UNLESS(nn::btm::user::ResultBluetoothOff().Includes(TryStartBleScanSmartDevice(filterForSmartDevice)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for smart device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS(nn::btm::user::ResultBluetoothOff().Includes(TryStopBleScanSmartDevicel()));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Start BLE Scan for paired general device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS(nn::btm::user::ResultBluetoothOff().Includes(TryStartBleScanPaired(filterForBleDevice)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Stop BLE Scan for paired general device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS(nn::btm::user::ResultBluetoothOff().Includes(TryStopBleScanPaired()));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtmUser::TestBleConnectRadioOff()
    {
        DisableRadio();

        nn::btm::BdAddress dummyAddress = { {0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF} };

        NN_TEST_BTM_MODULE_LOG("%s: Connect to a general device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS(nn::btm::user::ResultBluetoothOff().Includes(nn::btm::user::BleConnect(dummyAddress)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtmUser::TestBleServiceDiscoveryRadioOff()
    {
        DisableRadio();

        NN_TEST_BTM_MODULE_LOG("%s: Since BLE connection is not available, skip this test\n", NN_CURRENT_FUNCTION_NAME);
        NN_TEST_BTM_MODULE_LOG("%s: BLE Service Discovery test for unconnected devices is done in TestBleServiceDiscovery()\n", NN_CURRENT_FUNCTION_NAME);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtmUser::TestBlePairingRadioOff()
    {
        DisableRadio();

        NN_TEST_BTM_MODULE_LOG("%s: Since BLE connection is not available, skip this test\n", NN_CURRENT_FUNCTION_NAME);
        NN_TEST_BTM_MODULE_LOG("%s: BLE Pairing test for unconnected devices is done in TestBlePairing()\n", NN_CURRENT_FUNCTION_NAME);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtmUser::TestBleMtuConfigRadioOff()
    {
        DisableRadio();

        NN_TEST_BTM_MODULE_LOG("%s: Since BLE connection is not available, skip this test\n", NN_CURRENT_FUNCTION_NAME);
        NN_TEST_BTM_MODULE_LOG("%s: BLE MTU Config test for unconnected devices is done in TestBleMtuConfig()\n", NN_CURRENT_FUNCTION_NAME);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtmUser::TestBleDataPathRadioOff()
    {
        DisableRadio();

        // Radio 状態に無関係
        TestBleDataPath();
    }



    nn::Result TestBtmModule_ApiBtmUser::TryStartBleScanGeneral(const nn::btm::user::BleAdvFilterForGeneral& filter)
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::StartBleScanForGeneral(filter);

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtmUser::TryStopBleScanGeneral()
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::StopBleScanForGeneral();

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtmUser::TryStartBleScanSmartDevice(const nn::btm::user::BleAdvFilterForSmartDevice& filter)
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::StartBleScanForSmartDevice(filter);

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtmUser::TryStopBleScanSmartDevicel()
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::StopBleScanForSmartDevice();

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtmUser::TryStartBleScanPaired(const nn::btm::user::BleAdvFilterForGeneral& filter)
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::StartBleScanForPaired(filter);

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtmUser::TryStopBleScanPaired()
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::StopBleScanForPaired();

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtmUser::TryConfigureBleMtu(uint32_t connectionHandle, uint16_t mtu)
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::user::ConfigureBleMtu(connectionHandle, mtu);

            if (nn::btm::user::ResultBusyProcessingExclusiveApi().Includes(result))
            {
                NN_LOG("Anothre MTU configuration is ongoing...\n");
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtmUser::TryRegisterGattDataPath(const nn::bluetooth::GattAttributeUuid& uuid, bool isToRegister)
    {
        while (NN_STATIC_CONDITION(true))
        {
            nn::btm::user::BleDataPath path;
            path.path = nn::btm::user::BLE_DATA_PATH_GENERAL;
            path.uuid = uuid;

            nn::Result result = nn::ResultSuccess();

            if (isToRegister)
            {
                result = nn::btm::user::RegisterBleGattDataPath(path);
            }
            else
            {
                result = nn::btm::user::UnregisterBleGattDataPath(path);
            }

            if (nn::btm::user::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }
}   // namespace ApiBtmUser

