﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/bluetooth_Client.h"

#include <cstring>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>

using namespace nn::bluetooth;

typedef enum
{
    Default = 0,
    Started,
    Stopped
} DiscoveryStateType;

DiscoveryStateType stateStarted = Started;
DiscoveryStateType stateStopped = Stopped;

nn::bluetooth::BluetoothAddress emptyBluetoothAddress = {{0, 0, 0, 0, 0, 0}};

class pairingTest1Client : public bluetoothClient
{
    /*******************************************************************************
    * EventFromDeviceFoundCallback()
    *******************************************************************************/
    void EventFromDeviceFoundCallback(const nn::bluetooth::InfoFromDeviceFoundCallback* pInfo)
    {
        if(stateStarted == Started && stateStopped != Stopped)
        {
            // Verify data
            NN_LOG("\nVerifying data from EventFromDeviceFoundCallback...\n");

            if(memcmp(emptyBluetoothAddress.address, pInfo->BdAddress.address, 6) == 0)
            {
                NN_LOG("!!!Address validation failure: BdAddress equal to all zeros!!!\n");
                FAIL();
            }
            else
            {
                NN_LOG("Address verification passed, address discovered:%s\n", toHexString(6, pInfo->BdAddress.address));
            }

            if(strlen((char *)pInfo->BdName.name) == 0)
            {
                NN_LOG("!!!Name validation failure: BdName equal to zero!!!\n\n");
                FAIL();
            }
            else
            {
                NN_LOG("Name verification passed, name discovered:%s\n\n", pInfo->BdName.name);
            }
        }
        else
        {
            NN_LOG("\n!!!!!!!!TEST FAILURE!!!!!!!!\n");
            FAIL();
            if(stateStarted != Started)
                NN_LOG("EventFromDeviceFoundCallback received outside of discovery started state!\n\n");
            if(stateStopped == Stopped)
                NN_LOG("EventFromDeviceFoundCallback received inside of discovery stopped state!\n\n");
        }
    }
    /*******************************************************************************
    * EventFromDiscoveryStateChangedCallback()
    *******************************************************************************/
    void EventFromDiscoveryStateChangedCallback(const nn::bluetooth::InfoFromDiscoveryStateChangedCallback* pInfo)
    {
        const char* discStrP;
        switch (pInfo->state)
        {
            case BT_DISCOVERY_STOPPED:
                discStrP = "BT_DISCOVERY_STOPPED";
                stateStopped = Stopped;
            break;
            case BT_DISCOVERY_STARTED:
                discStrP = "BT_DISCOVERY_STARTED";
                stateStarted = Started;
            break;
            default:
                NN_LOG("ERROR: Invalid DiscState: 0x%X\n", pInfo->state);
                discStrP = "UNKNOWN!";
        }

        NN_LOG("DiscoveryStateChanged: %s\n", discStrP);
    }

};

static pairingTest1Client client;
/*******************************************************************************
* nnMain()
*******************************************************************************/
extern "C" void nnMain()
{
    int loopTimeMs = 0;
    int testCycle = 1;

    NN_LOG("\nbluetoothPairingTest1_Client NOW RUNNING \n");

    client.startBluetooth();
    nn::bluetooth::ExtSetVisibility(false,true);

    for(;;)
    {
        if( loopTimeMs == 0 && testCycle <= 10 )
        {
            if( stateStopped == Stopped )
            {
                // Reset test states
                stateStarted = Default;
                stateStopped = Default;

                NN_LOG( "\n***********Beginning Test Cycle [%d]************\n\n", testCycle );

                // Discovery
                NN_LOG( "  [StartDiscovery]\n" );
                nn::bluetooth::StartDiscovery();

                testCycle++;
            }

        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
        loopTimeMs += 100;
        if( loopTimeMs >= 20000 )
        {
           loopTimeMs=0;
        }

        if( testCycle > 10 )
        {
            NN_LOG( "\n***********Test Complete************\n\n" );
            break;
        }
    }

    SUCCEED();
}
