﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testBle_Common.h"
#include "../Common/testBle_ServiceDiscoveryCommon.h"

#include <nn/bluetooth/bluetooth_LeApi.h>

//!< 接続がない状態で GATT Service を取得しようとしたら 0 もしくは見つからない
TEST(TestBleServiceDiscovery, GetServiceWithoutConnection)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::sdp::SetUp();

    nn::bluetooth::GattService services[nn::bluetooth::GattAttributeCountMaxClient];

    // Service 数は 0
    int serviceCount = nn::bluetooth::GetGattServices(services, NN_ARRAY_SIZE(services), nn::bluetooth::BleInvalidConnectionHandle);
    EXPECT_EQ(serviceCount, 0);

    // 何も変化なし
    for (const auto& service : services)
    {
        EXPECT_EQ(service.GetType(), nn::bluetooth::GattAttributeType_Service);
        EXPECT_EQ(service.GetConnectionHandle(), nn::bluetooth::BleInvalidConnectionHandle);
        EXPECT_EQ(service.GetHandle(), nn::bluetooth::GattAttributeInvalidHandle);
        EXPECT_EQ(service.GetEndGroupHandle(), nn::bluetooth::GattAttributeInvalidHandle);
        EXPECT_EQ(service.GetInstanceId(), 0x0000);
        EXPECT_EQ(service.GetUuid().length, nn::bluetooth::GattAttributeUuidLength_16);
        EXPECT_EQ(service.GetUuid().uu.uuid16, 0x0000);
    }

    // UUID を指定した場合は見つからない
    EXPECT_FALSE(nn::bluetooth::GetGattService(&services[0], nnt::bluetooth::common::InvalidUuid, nn::bluetooth::BleInvalidConnectionHandle));

    // 何も変化なし
    EXPECT_EQ(services[0].GetType(), nn::bluetooth::GattAttributeType_Service);
    EXPECT_EQ(services[0].GetConnectionHandle(), nn::bluetooth::BleInvalidConnectionHandle);
    EXPECT_EQ(services[0].GetHandle(), nn::bluetooth::GattAttributeInvalidHandle);
    EXPECT_EQ(services[0].GetEndGroupHandle(), nn::bluetooth::GattAttributeInvalidHandle);
    EXPECT_EQ(services[0].GetInstanceId(), 0x0000);
    EXPECT_EQ(services[0].GetUuid().length, nn::bluetooth::GattAttributeUuidLength_16);
    EXPECT_EQ(services[0].GetUuid().uu.uuid16, 0x0000);

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::sdp::TearDown();
}

//!< 接続がない状態で GATT Included Service を取得しようとしたら 0
TEST(TestBleServiceDiscovery, GetIncludedServiceWithoutConnection)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::sdp::SetUp();

    nn::bluetooth::GattService service;
    nn::bluetooth::GattService includedServices[nn::bluetooth::GattAttributeCountMaxClient];

    // Included Service 数は 0
    int includedServiceCount = service.GetIncludedServices(includedServices, NN_ARRAY_SIZE(includedServices));
    EXPECT_EQ(includedServiceCount, 0);

    // 何も変化なし
    for (const auto& service : includedServices)
    {
        EXPECT_EQ(service.GetType(), nn::bluetooth::GattAttributeType_Service);
        EXPECT_EQ(service.GetConnectionHandle(), nn::bluetooth::BleInvalidConnectionHandle);
        EXPECT_EQ(service.GetHandle(), nn::bluetooth::GattAttributeInvalidHandle);
        EXPECT_EQ(service.GetEndGroupHandle(), nn::bluetooth::GattAttributeInvalidHandle);
        EXPECT_EQ(service.GetInstanceId(), 0x0000);
        EXPECT_EQ(service.GetUuid().length, nn::bluetooth::GattAttributeUuidLength_16);
        EXPECT_EQ(service.GetUuid().uu.uuid16, 0x0000);
    }

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::sdp::TearDown();
}

//!< 接続がない状態で GATT Characteristic を取得しようとしたら 0 もしくは見つからない
TEST(TestBleServiceDiscovery, GetCharacteristicWithoutConnection)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::sdp::SetUp();

    nn::bluetooth::GattService          service;
    nn::bluetooth::GattCharacteristic   characteristics[nn::bluetooth::GattAttributeCountMaxClient];

    // Characteristic 数は 0
    int characteristicCount = service.GetCharacteristics(characteristics, NN_ARRAY_SIZE(characteristics));
    EXPECT_EQ(characteristicCount, 0);

    // 何も変化なし
    for (const auto& characteristic : characteristics)
    {
        EXPECT_EQ(characteristic.GetType(), nn::bluetooth::GattAttributeType_Characteristic);
        EXPECT_EQ(characteristic.GetConnectionHandle(), nn::bluetooth::BleInvalidConnectionHandle);
        EXPECT_EQ(characteristic.GetHandle(), nn::bluetooth::GattAttributeInvalidHandle);
        EXPECT_EQ(characteristic.GetProperties(), 0x00);
        EXPECT_EQ(characteristic.GetInstanceId(), 0x0000);
        EXPECT_EQ(characteristic.GetUuid().length, nn::bluetooth::GattAttributeUuidLength_16);
        EXPECT_EQ(characteristic.GetUuid().uu.uuid16, 0x0000);
    }

    // UUID を指定した場合は見つからない
    EXPECT_FALSE(service.GetCharacteristic(&characteristics[0], nnt::bluetooth::common::InvalidUuid));

    // 何も変化なし
    EXPECT_EQ(characteristics[0].GetType(), nn::bluetooth::GattAttributeType_Characteristic);
    EXPECT_EQ(characteristics[0].GetConnectionHandle(), nn::bluetooth::BleInvalidConnectionHandle);
    EXPECT_EQ(characteristics[0].GetHandle(), nn::bluetooth::GattAttributeInvalidHandle);
    EXPECT_EQ(characteristics[0].GetProperties(), 0x00);
    EXPECT_EQ(characteristics[0].GetInstanceId(), 0x0000);
    EXPECT_EQ(characteristics[0].GetUuid().length, nn::bluetooth::GattAttributeUuidLength_16);
    EXPECT_EQ(characteristics[0].GetUuid().uu.uuid16, 0x0000);

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::sdp::TearDown();
}

//!< 接続がない状態で GATT Characteristic が属する Gatt Service を取得しようとすると、Abort
TEST(TestBleServiceDiscovery, GetBelongingServiceWithoutConnection1)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::sdp::SetUp();

    nn::bluetooth::GattService          service;
    nn::bluetooth::GattCharacteristic   characteristic;

    EXPECT_DEATH_IF_SUPPORTED(characteristic.GetService(&service), "");

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::sdp::TearDown();
}

//!< 接続がない状態で GATT Descriptor を取得しようとしたら 0 もしくは見つからない
TEST(TestBleServiceDiscovery, GetDescriptorWithoutConnection)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::sdp::SetUp();

    nn::bluetooth::GattCharacteristic   characteristic;
    nn::bluetooth::GattDescriptor       descriptors[nn::bluetooth::GattAttributeCountMaxClient];

    // Descriptor 数は 0
    int descriptorCount = characteristic.GetDescriptors(descriptors, NN_ARRAY_SIZE(descriptors));
    EXPECT_EQ(descriptorCount, 0);

    // 何も変化なし
    for (const auto& descriptor : descriptors)
    {
        EXPECT_EQ(descriptor.GetType(), nn::bluetooth::GattAttributeType_Descriptor);
        EXPECT_EQ(descriptor.GetConnectionHandle(), nn::bluetooth::BleInvalidConnectionHandle);
        EXPECT_EQ(descriptor.GetHandle(), nn::bluetooth::GattAttributeInvalidHandle);
        EXPECT_EQ(descriptor.GetUuid().length, nn::bluetooth::GattAttributeUuidLength_16);
        EXPECT_EQ(descriptor.GetUuid().uu.uuid16, 0x0000);
    }

    // UUID を指定した場合は見つからない
    EXPECT_FALSE(characteristic.GetDescriptor(&descriptors[0], nnt::bluetooth::common::InvalidUuid));

    // 何も変化なし
    EXPECT_EQ(descriptors[0].GetType(), nn::bluetooth::GattAttributeType_Descriptor);
    EXPECT_EQ(descriptors[0].GetConnectionHandle(), nn::bluetooth::BleInvalidConnectionHandle);
    EXPECT_EQ(descriptors[0].GetHandle(), nn::bluetooth::GattAttributeInvalidHandle);
    EXPECT_EQ(descriptors[0].GetUuid().length, nn::bluetooth::GattAttributeUuidLength_16);
    EXPECT_EQ(descriptors[0].GetUuid().uu.uuid16, 0x0000);

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::sdp::TearDown();
}

//!< 接続がない状態で GATT Descriptor が属する Gatt Service を取得しようとすると、Abort
TEST(TestBleServiceDiscovery, GetBelongingServiceWithoutConnection2)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::sdp::SetUp();

    nn::bluetooth::GattService      service;
    nn::bluetooth::GattDescriptor   descriptor;

    EXPECT_DEATH_IF_SUPPORTED(descriptor.GetService(&service), "");

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::sdp::TearDown();
}

//!< 接続がない状態で GATT Descriptor が属する Gatt Characteristic を取得しようとすると、Abort
TEST(TestBleServiceDiscovery, GetBelongingCharacteristicWithoutConnection)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::sdp::SetUp();

    nn::bluetooth::GattCharacteristic   characteristic;
    nn::bluetooth::GattDescriptor       descriptor;

    EXPECT_DEATH_IF_SUPPORTED(descriptor.GetCharacteristic(&characteristic), "");

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::sdp::TearDown();
}

