﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include <nn/bluetooth/bluetooth_LeApi.h>

#include "testBle_Common.h"
#include "testBle_ConnectCommon.h"

namespace
{
    bool g_IsInitialized = false;                       //!< Initialize() が呼ばれたかどうか

    nn::os::SystemEventType g_BleConnectionEvent;       //!< BLE Connection イベント

    nn::os::EventType* g_pBleConnectionEventInternal;   //!< 各テストから渡される内部向け BLE 接続イベント

    const size_t ThreadStackSize = 4096;
    NN_OS_ALIGNAS_THREAD_STACK char g_EventHandleThreadStack[ThreadStackSize];
    nn::os::ThreadType g_EventHandleThread;
    void EventHandleThreadFunction(void* arg);
    volatile bool g_IsToFinishEventHandleThread;

    void Initialize()
    {
        if (!g_IsInitialized)
        {
            nn::bluetooth::AcquireBleConnectionStateChangedEvent(&g_BleConnectionEvent);

            g_IsInitialized = true;
        }
    }

    void DisconnectAll()
    {
        nn::bluetooth::BleConnectionInfo connectionInfo[nn::bluetooth::BleConnectionCountMaxClient];

        int connectionCount = nn::bluetooth::GetBleConnectionInfoList(connectionInfo, NN_ARRAY_SIZE(connectionInfo));

        for (int i = 0; i < connectionCount; ++i)
        {
            nn::os::ClearSystemEvent(&g_BleConnectionEvent);

            auto result = nn::bluetooth::DisconnectFromGattServer(connectionInfo[i].connectionHandle);
            EXPECT_TRUE(result.IsSuccess());

            nn::os::WaitSystemEvent(&g_BleConnectionEvent);
        }

        connectionCount = nn::bluetooth::GetBleConnectionInfoList(connectionInfo, NN_ARRAY_SIZE(connectionInfo));
        EXPECT_EQ(connectionCount, 0);
    }

    void EventHandleThreadFunction(void* arg)
    {
        NN_UNUSED(arg);

        while (!g_IsToFinishEventHandleThread)
        {
            if (nn::os::TryWaitSystemEvent(&g_BleConnectionEvent))
            {
                if (g_pBleConnectionEventInternal != nullptr)
                {
                    nn::os::SignalEvent(g_pBleConnectionEventInternal);
                }
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
    }
}

namespace nnt { namespace bluetooth { namespace connect {
    void Finalize()
    {
        if (g_IsInitialized)
        {
            nn::os::DestroySystemEvent(&g_BleConnectionEvent);

            g_IsInitialized = false;
        }
    }

    void SetUp(nn::os::EventType* pConnectionEvent)
    {
        Initialize();

        DisconnectAll();

        g_pBleConnectionEventInternal = pConnectionEvent;

        if (g_pBleConnectionEventInternal != nullptr)
        {
            g_IsToFinishEventHandleThread = false;

            auto result = nn::os::CreateThread(&g_EventHandleThread, EventHandleThreadFunction, nullptr, g_EventHandleThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
            EXPECT_TRUE(result.IsSuccess());

            nn::os::StartThread(&g_EventHandleThread);
        }
    }

    void TearDown()
    {
        if (g_pBleConnectionEventInternal != nullptr)
        {
            g_IsToFinishEventHandleThread = true;
            nn::os::WaitThread(&g_EventHandleThread);
            nn::os::DestroyThread(&g_EventHandleThread);
        }

        DisconnectAll();
    }

}}}; // namespace nnt::bluetooth::connect

