﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>

#include <nn/bcat/service/bcat_NotificationPayload.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_ResultPrivate.h>

TEST(NotificationPayload, Basic1)
{
    const char* s[] =
    {
        "{}",
        "{  }",
        ""
    };

    for (int i = 0; s[i][0] != '\0'; i++)
    {
        nn::bcat::service::NotificationPayload payload(s[i], static_cast<size_t>(nn::util::Strnlen(s[i], 2048)));
        EXPECT_TRUE(payload.IsValid());
    }
}

TEST(NotificationPayload, Basic2)
{
    const char* s[] =
    {
        "{\"key\":\"value\"}",
        "{\"key\":\"value\"  }",
        "{  \"key\":\"value\"}",
        "{  \"key\":\"value\"  }",
        "{  \"key\":  \"value\"  }",
        "{  \"key\"  :\"value\"  }",
        "{  \"key\"  :  \"value\"  }",
        ""
    };

    for (int i = 0; s[i][0] != '\0'; i++)
    {
        nn::bcat::service::NotificationPayload payload(s[i], static_cast<size_t>(nn::util::Strnlen(s[i], 2048)));
        EXPECT_TRUE(payload.IsValid());

        char buffer[1024];
        size_t size;

        EXPECT_TRUE(payload.GetValue(&size, "key", buffer, sizeof (buffer)));
        EXPECT_TRUE(size == static_cast<size_t>(nn::util::Strnlen("value", 2048) + 1));
        EXPECT_TRUE(nn::util::Strncmp(buffer, "value", 2048) == 0);

        int64_t value;

        EXPECT_FALSE(payload.GetValue(&value, "key"));
    }
}

TEST(NotificationPayload, Basic3)
{
    const char* s[] =
    {
        "{ \"key1\":\"value1\",\"key2\":\"value2\" }",
        "{ \"key1\":\"value1\" ,\"key2\":\"value2\" }",
        "{ \"key1\":\"value1\", \"key2\":\"value2\" }",
        "{ \"key1\":\"value1\" , \"key2\":\"value2\" }",
        ""
    };

    for (int i = 0; s[i][0] != '\0'; i++)
    {
        nn::bcat::service::NotificationPayload payload(s[i], static_cast<size_t>(nn::util::Strnlen(s[i], 2048)));
        EXPECT_TRUE(payload.IsValid());

        char buffer[1024];
        size_t size;

        EXPECT_TRUE(payload.GetValue(&size, "key1", buffer, sizeof (buffer)));
        EXPECT_TRUE(size == static_cast<size_t>(nn::util::Strnlen("value1", 2048) + 1));
        EXPECT_TRUE(nn::util::Strncmp(buffer, "value1", 2048) == 0);

        EXPECT_TRUE(payload.GetValue(&size, "key2", buffer, sizeof (buffer)));
        EXPECT_TRUE(size == static_cast<size_t>(nn::util::Strnlen("value2", 2048) + 1));
        EXPECT_TRUE(nn::util::Strncmp(buffer, "value2", 2048) == 0);

        int64_t value;

        EXPECT_FALSE(payload.GetValue(&value, "key1"));
        EXPECT_FALSE(payload.GetValue(&value, "key2"));
    }
}

TEST(NotificationPayload, Basic4)
{
    // null 終端出ないペイロードでも問題なくパースできることを確認する。
    const char* s[] =
    {
        "{}YYYYY",
        "{ \"key1\":\"value1\" ,\"key2\":\"value2\" }YYYYY",
        ""
    };

    for (int i = 0; s[i][0] != '\0'; i++)
    {
        nn::bcat::service::NotificationPayload payload(s[i], static_cast<size_t>(nn::util::Strnlen(s[i], 2048)) - 5);
        EXPECT_TRUE(payload.IsValid());

        char buffer[1024];
        size_t size;

        EXPECT_FALSE(payload.GetValue(&size, "key3", buffer, sizeof (buffer)));

        int64_t value;

        EXPECT_FALSE(payload.GetValue(&value, "key3"));
    }
}

TEST(NotificationPayload, Basic5)
{
    const char* s = "{ \"key1\":\"key2\",\"key2\":\"key3\",\"key3\":\"key4\" }";

    nn::bcat::service::NotificationPayload payload(s, static_cast<size_t>(nn::util::Strnlen(s, 2048)));
    EXPECT_TRUE(payload.IsValid());

    char buffer[1024];
    size_t size;

    EXPECT_TRUE(payload.GetValue(&size, "key1", buffer, sizeof (buffer)));
    EXPECT_TRUE(size == static_cast<size_t>(nn::util::Strnlen("key2", 2048) + 1));
    EXPECT_TRUE(nn::util::Strncmp(buffer, "key2", 2048) == 0);

    EXPECT_TRUE(payload.GetValue(&size, "key2", buffer, sizeof (buffer)));
    EXPECT_TRUE(size == static_cast<size_t>(nn::util::Strnlen("key3", 2048) + 1));
    EXPECT_TRUE(nn::util::Strncmp(buffer, "key3", 2048) == 0);

    EXPECT_TRUE(payload.GetValue(&size, "key3", buffer, sizeof (buffer)));
    EXPECT_TRUE(size == static_cast<size_t>(nn::util::Strnlen("key4", 2048) + 1));
    EXPECT_TRUE(nn::util::Strncmp(buffer, "key4", 2048) == 0);

    int64_t value;

    EXPECT_FALSE(payload.GetValue(&value, "key1"));
    EXPECT_FALSE(payload.GetValue(&value, "key2"));
    EXPECT_FALSE(payload.GetValue(&value, "key3"));
}

TEST(NotificationPayload, Basic6)
{
    const char* s = "{ \"key1\":0,\"key2\":10000,\"key3\":-10000 }";

    nn::bcat::service::NotificationPayload payload(s, static_cast<size_t>(nn::util::Strnlen(s, 2048)));
    EXPECT_TRUE(payload.IsValid());

    char buffer[1024];
    size_t size;

    EXPECT_FALSE(payload.GetValue(&size, "key1", buffer, sizeof (buffer)));
    EXPECT_FALSE(payload.GetValue(&size, "key2", buffer, sizeof (buffer)));
    EXPECT_FALSE(payload.GetValue(&size, "key3", buffer, sizeof (buffer)));

    int64_t value;

    EXPECT_TRUE(payload.GetValue(&value, "key1"));
    EXPECT_TRUE(value == 0);

    EXPECT_TRUE(payload.GetValue(&value, "key2"));
    EXPECT_TRUE(value == 10000);

    EXPECT_TRUE(payload.GetValue(&value, "key3"));
    EXPECT_TRUE(value == -10000);
}

TEST(NotificationPayload, Error1)
{
    const char* s[] =
    {
        "",
        "{",
        "}",
        "{\"key\"",
        "{\"key\":",
        "{\"key\":}",
        "{\"key\":null}",
        "{\"key\":true}",
        "{\"key\":false}",
        "{\"key\":[]}",
        ""
    };

    for (int i = 0; s[i][0] != '\0'; i++)
    {
        nn::bcat::service::NotificationPayload payload(s[i], static_cast<size_t>(nn::util::Strnlen(s[i], 2048)));
        EXPECT_FALSE(payload.IsValid());
    }
}

TEST(NotificationPayload, Error2)
{
    const char* s = "{ \"key\":\"value\" }";

    nn::bcat::service::NotificationPayload payload(s, static_cast<size_t>(nn::util::Strnlen(s, 2048)));
    EXPECT_TRUE(payload.IsValid());

    char buffer[5];
    size_t size;

    EXPECT_FALSE(payload.GetValue(&size, "key", buffer, sizeof (buffer)));
}

TEST(NotificationPayload, Error3)
{
    const char* s = "{ \"key1\":\"key2\",\"key2\":\"key3\",\"key3\":\"key4\" }";

    nn::bcat::service::NotificationPayload payload(s, static_cast<size_t>(nn::util::Strnlen(s, 2048)));
    EXPECT_TRUE(payload.IsValid());

    char buffer[1024];
    size_t size;

    EXPECT_FALSE(payload.GetValue(&size, "key4", buffer, sizeof (buffer)));
}

TEST(NotificationPayload, Error4)
{
    const char* s = "{ \"key2\":\"value2\" }, { \"key2\":\"value2\" }";

    nn::bcat::service::NotificationPayload payload(s, static_cast<size_t>(nn::util::Strnlen(s, 2048)));
    EXPECT_FALSE(payload.IsValid());
}

TEST(NotificationPayload, News)
{
    const char* s = "{\"type\":\"news\",\"topic_id\":\"nx_news\",\"url\":\"https://example.com/\"}";

    nn::bcat::service::NotificationPayload payload(s, static_cast<size_t>(nn::util::Strnlen(s, 2048)));
    EXPECT_TRUE(payload.IsValid());

    char buffer[1024];
    size_t size;

    EXPECT_TRUE(payload.GetValue(&size, "type", buffer, sizeof (buffer)));
    EXPECT_TRUE(nn::util::Strncmp(buffer, "news", 2048) == 0);

    EXPECT_TRUE(payload.GetValue(&size, "topic_id", buffer, sizeof (buffer)));
    EXPECT_TRUE(nn::util::Strncmp(buffer, "nx_news", 2048) == 0);

    EXPECT_TRUE(payload.GetValue(&size, "url", buffer, sizeof (buffer)));
    EXPECT_TRUE(nn::util::Strncmp(buffer, "https://example.com/", 2048) == 0);
}
