﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

#include <nn/bcat/detail/service/core/bcat_DiffView.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>

using namespace nn::bcat::detail::service::core;

namespace
{
    const int FileCountHalf = nn::bcat::DeliveryCacheFileCountMaxPerDirectory / 2;
    const int DirectoryCountHalf = nn::bcat::DeliveryCacheDirectoryCountMax / 2;
}

namespace
{
    FileMetaView g_FileMetaViewHalf;
    FileMetaView g_FileMetaView;

    DirectoryMetaView g_DirectoryMetaViewHalf;
    DirectoryMetaView g_DirectoryMetaView;
}

TEST(DiffView, Initialize)
{
    for (int i = 0; i < FileCountHalf; i++)
    {
        nn::bcat::detail::FileMeta meta = {};
        nn::util::SNPrintf(meta.name.value, sizeof (meta.name.value), "file%03d", i + 1);

        ASSERT_RESULT_SUCCESS(g_FileMetaViewHalf.AddOrUpdate(meta));
    }
    for (int i = 0; i < nn::bcat::DeliveryCacheFileCountMaxPerDirectory; i++)
    {
        nn::bcat::detail::FileMeta meta = {};
        nn::util::SNPrintf(meta.name.value, sizeof (meta.name.value), "file%03d", i + 1);

        ASSERT_RESULT_SUCCESS(g_FileMetaView.AddOrUpdate(meta));
    }

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        nn::bcat::detail::DirectoryMeta meta = {};
        nn::util::SNPrintf(meta.name.value, sizeof (meta.name.value), "dir%03d", i + 1);

        ASSERT_RESULT_SUCCESS(g_DirectoryMetaViewHalf.AddOrUpdate(meta));
    }
    for (int i = 0; i < nn::bcat::DeliveryCacheDirectoryCountMax; i++)
    {
        nn::bcat::detail::DirectoryMeta meta = {};
        nn::util::SNPrintf(meta.name.value, sizeof (meta.name.value), "dir%03d", i + 1);

        ASSERT_RESULT_SUCCESS(g_DirectoryMetaView.AddOrUpdate(meta));
    }
}

TEST(DiffView, FileFilter)
{
    FileDiffView diffView;

    nn::bcat::detail::ListFile remoteFiles[FileCountHalf + 1] = {};

    for (int i = 0; i < FileCountHalf; i++)
    {
        nn::bcat::detail::ListFile& entry = remoteFiles[i];

        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "file%03d", i + 1);
    }

    // 最後の 1 件だけ追加。
    nn::bcat::detail::ListFile& entry = remoteFiles[FileCountHalf];
    nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "file%03d", FileCountHalf + 1);

    ASSERT_RESULT_SUCCESS(diffView.Make(g_FileMetaViewHalf, remoteFiles, NN_ARRAY_SIZE(remoteFiles)));

    int indexes[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    // 要素数を変化させながら取得する。
    for (int i = 1; i < NN_ARRAY_SIZE(indexes); i++)
    {
        diffView.GetIndexListFromRemote(&count, indexes, i, nn::bcat::detail::DiffType_Add);
        EXPECT_EQ(count, 1);
    }
}

TEST(DiffView, FileNotModified)
{
    FileDiffView diffView;

    nn::bcat::detail::ListFile remoteFiles[FileCountHalf] = {};

    for (int i = 0; i < FileCountHalf; i++)
    {
        nn::bcat::detail::ListFile& entry = remoteFiles[i];

        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "file%03d", i + 1);
    }

    ASSERT_RESULT_SUCCESS(diffView.Make(g_FileMetaViewHalf, remoteFiles, NN_ARRAY_SIZE(remoteFiles)));

    int indexes[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, FileCountHalf);

    for (int i = 0; i < FileCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, FileCountHalf);

    for (int i = 0; i < FileCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);
}

TEST(DiffView, FileAdd)
{
    FileDiffView diffView;

    nn::bcat::detail::ListFile remoteFiles[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};

    // 51~ 追加
    for (int i = 0; i < nn::bcat::DeliveryCacheFileCountMaxPerDirectory; i++)
    {
        nn::bcat::detail::ListFile& entry = remoteFiles[i];

        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "file%03d", i + 1);
    }

    ASSERT_RESULT_SUCCESS(diffView.Make(g_FileMetaViewHalf, remoteFiles, NN_ARRAY_SIZE(remoteFiles)));

    int indexes[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, FileCountHalf);

    for (int i = 0; i < FileCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, FileCountHalf);

    for (int i = 0; i < FileCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, FileCountHalf);

    for (int i = 0; i < FileCountHalf; i++)
    {
        EXPECT_EQ(i + FileCountHalf, indexes[i]);
    }

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);
}

TEST(DiffView, FileRemove)
{
    FileDiffView diffView;

    nn::bcat::detail::ListFile remoteFiles[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};

    // 全件削除

    ASSERT_RESULT_SUCCESS(diffView.Make(g_FileMetaViewHalf, remoteFiles, 0));

    int indexes[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, FileCountHalf);

    for (int i = 0; i < FileCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);
}

TEST(DiffView, FileUpdate)
{
    FileDiffView diffView;

    nn::bcat::detail::ListFile remoteFiles[FileCountHalf] = {};

    for (int i = 0; i < FileCountHalf; i++)
    {
        nn::bcat::detail::ListFile& entry = remoteFiles[i];

        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "file%03d", i + 1);

        // 全件更新
        entry.digest.value[0] = 1;
    }

    ASSERT_RESULT_SUCCESS(diffView.Make(g_FileMetaViewHalf, remoteFiles, NN_ARRAY_SIZE(remoteFiles)));

    int indexes[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, FileCountHalf);

    for (int i = 0; i < FileCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, FileCountHalf);

    for (int i = 0; i < FileCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }
}

TEST(DiffView, FileReplacement)
{
    FileDiffView diffView;

    nn::bcat::detail::ListFile remoteFiles[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};

    for (int i = 0; i < nn::bcat::DeliveryCacheFileCountMaxPerDirectory; i++)
    {
        nn::bcat::detail::ListFile& entry = remoteFiles[i];

        // file001 → file1001
        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "file1%03d", i + 1);
    }

    ASSERT_RESULT_SUCCESS(diffView.Make(g_FileMetaView, remoteFiles, NN_ARRAY_SIZE(remoteFiles)));

    int indexes[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, nn::bcat::DeliveryCacheFileCountMaxPerDirectory);

    for (int i = 0; i < nn::bcat::DeliveryCacheFileCountMaxPerDirectory; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, nn::bcat::DeliveryCacheFileCountMaxPerDirectory);

    for (int i = 0; i < nn::bcat::DeliveryCacheFileCountMaxPerDirectory; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);
}

TEST(DiffView, DirectoryFilter)
{
    DirectoryDiffView diffView;

    nn::bcat::detail::ListDirectory remoteDirectories[DirectoryCountHalf + 1] = {};

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        nn::bcat::detail::ListDirectory& entry = remoteDirectories[i];

        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "dir%03d", i + 1);
        entry.mode = nn::bcat::detail::DirectoryMode_Sync;
    }

    // 最後の 1 件だけ追加。
    nn::bcat::detail::ListDirectory& entry = remoteDirectories[DirectoryCountHalf];
    nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "dir%03d", DirectoryCountHalf + 1);
    entry.mode = nn::bcat::detail::DirectoryMode_Sync;

    ASSERT_RESULT_SUCCESS(diffView.Make(g_DirectoryMetaViewHalf, remoteDirectories, NN_ARRAY_SIZE(remoteDirectories)));

    int indexes[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    // 要素数を変化させながら取得する。
    for (int i = 1; i < NN_ARRAY_SIZE(indexes); i++)
    {
        diffView.GetIndexListFromRemote(&count, indexes, i, nn::bcat::detail::DiffType_Add);
        EXPECT_EQ(count, 1);
    }
}

TEST(DiffView, DirectoryNotModified)
{
    DirectoryDiffView diffView;

    nn::bcat::detail::ListDirectory remoteDirectories[DirectoryCountHalf] = {};

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        nn::bcat::detail::ListDirectory& entry = remoteDirectories[i];

        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "dir%03d", i + 1);
        entry.mode = nn::bcat::detail::DirectoryMode_Sync;
    }

    ASSERT_RESULT_SUCCESS(diffView.Make(g_DirectoryMetaViewHalf, remoteDirectories, NN_ARRAY_SIZE(remoteDirectories)));

    int indexes[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, DirectoryCountHalf);

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, DirectoryCountHalf);

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);
}

TEST(DiffView, DirectoryAdd)
{
    DirectoryDiffView diffView;

    nn::bcat::detail::ListDirectory remoteDirectories[nn::bcat::DeliveryCacheDirectoryCountMax] = {};

    // 51~ 追加
    for (int i = 0; i < nn::bcat::DeliveryCacheDirectoryCountMax; i++)
    {
        nn::bcat::detail::ListDirectory& entry = remoteDirectories[i];

        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "dir%03d", i + 1);
        entry.mode = nn::bcat::detail::DirectoryMode_Sync;
    }

    ASSERT_RESULT_SUCCESS(diffView.Make(g_DirectoryMetaViewHalf, remoteDirectories, NN_ARRAY_SIZE(remoteDirectories)));

    int indexes[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, DirectoryCountHalf);

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, DirectoryCountHalf);

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, DirectoryCountHalf);

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        EXPECT_EQ(i + DirectoryCountHalf, indexes[i]);
    }

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);
}

TEST(DiffView, DirectoryRemove)
{
    DirectoryDiffView diffView;

    nn::bcat::detail::ListDirectory remoteDirectories[nn::bcat::DeliveryCacheDirectoryCountMax] = {};

    // 全件削除

    ASSERT_RESULT_SUCCESS(diffView.Make(g_DirectoryMetaViewHalf, remoteDirectories, 0));

    int indexes[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, DirectoryCountHalf);

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);
}

TEST(DiffView, DirectoryUpdate)
{
    DirectoryDiffView diffView;

    nn::bcat::detail::ListDirectory remoteDirectories[DirectoryCountHalf] = {};

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        nn::bcat::detail::ListDirectory& entry = remoteDirectories[i];

        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "dir%03d", i + 1);
        entry.mode = nn::bcat::detail::DirectoryMode_Sync;

        // 全件更新
        entry.digest.value[0] = 1;
    }

    ASSERT_RESULT_SUCCESS(diffView.Make(g_DirectoryMetaViewHalf, remoteDirectories, NN_ARRAY_SIZE(remoteDirectories)));

    int indexes[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, DirectoryCountHalf);

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, DirectoryCountHalf);

    for (int i = 0; i < DirectoryCountHalf; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }
}

TEST(DiffView, DirectoryReplacement)
{
    DirectoryDiffView diffView;

    nn::bcat::detail::ListDirectory remoteDirectories[nn::bcat::DeliveryCacheDirectoryCountMax] = {};

    for (int i = 0; i < nn::bcat::DeliveryCacheDirectoryCountMax; i++)
    {
        nn::bcat::detail::ListDirectory& entry = remoteDirectories[i];

        // dir001 → dir1001
        nn::util::SNPrintf(entry.name.value, sizeof (entry.name.value), "dir1%03d", i + 1);
        entry.mode = nn::bcat::detail::DirectoryMode_Sync;
    }

    ASSERT_RESULT_SUCCESS(diffView.Make(g_DirectoryMetaView, remoteDirectories, NN_ARRAY_SIZE(remoteDirectories)));

    int indexes[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    // Local

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, nn::bcat::DeliveryCacheDirectoryCountMax);

    for (int i = 0; i < nn::bcat::DeliveryCacheDirectoryCountMax; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);

    // Remote

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_NotModified);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Add);
    EXPECT_EQ(count, nn::bcat::DeliveryCacheDirectoryCountMax);

    for (int i = 0; i < nn::bcat::DeliveryCacheDirectoryCountMax; i++)
    {
        EXPECT_EQ(i, indexes[i]);
    }

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Remove);
    EXPECT_EQ(count, 0);

    diffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), nn::bcat::detail::DiffType_Update);
    EXPECT_EQ(count, 0);
}
