﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

#include <nn/bcat/service/bcat_ArchiveDecryptionKeyGenerator.h>
#include <nn/bcat/service/bcat_ArchiveDecoder.h>
#include <nn/bcat/service/bcat_ArchiveVerifier.h>

namespace
{
    nn::Bit8 g_Encoded[1 * 1024 * 1024] = {};
    size_t g_EncodedSize = 0;

    nn::Bit8 g_Decoded[1 * 1024 * 1024] = {};
    size_t g_DecodedSize = 0;

    nn::Bit8 g_Answer[1 * 1024 * 1024] = {};
    size_t g_AnswerSize = 0;

    nn::ApplicationId AppId = {0x0100b7d0020fc000};
    const char* Passphrase = "c9c949818fef72b6eb5794266bf19eb55959e0626ffa5ae58f0b7b4976290542";

    nn::bcat::service::ArchiveHeader g_Header;
}

nn::Result LoadFile() NN_NOEXCEPT
{
    static nn::Bit8 s_MountRomCacheBuffer[4096];

    size_t mountRomCacheUseSize = 0;
    NN_RESULT_DO(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));

    NN_ABORT_UNLESS_LESS_EQUAL(mountRomCacheUseSize, sizeof (s_MountRomCacheBuffer));

    NN_RESULT_DO(nn::fs::MountRom("rom", s_MountRomCacheBuffer, sizeof (s_MountRomCacheBuffer)));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    {
        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, "rom:/sample.baf", nn::fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::ReadFile(&g_EncodedSize, handle, 0, g_Encoded, sizeof (g_Encoded)));

        NN_ABORT_UNLESS_LESS(sizeof (g_Header), g_EncodedSize);
        std::memcpy(&g_Header, g_Encoded, sizeof (g_Header));
    }

    {
        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, "rom:/sample.bin", nn::fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::ReadFile(&g_AnswerSize, handle, 0, g_Answer, sizeof (g_Answer)));
    }

    NN_RESULT_SUCCESS;
}

TEST(ArchiveDecoder, Initialize)
{
    ASSERT_RESULT_SUCCESS(LoadFile());
}

TEST(ArchiveDecoder, AnalyzeHeader)
{
    NN_LOG("Magic: %c%c%c%c\n",
        g_Header.magic[0], g_Header.magic[1], g_Header.magic[2], g_Header.magic[3]);

    EXPECT_TRUE(g_Header.magic[0] == 'b' && g_Header.magic[1] == 'c' && g_Header.magic[2] == 'a' && g_Header.magic[3] == 't');

    NN_LOG("Version: %d\n", g_Header.version);

    EXPECT_TRUE(g_Header.version == 1);

    switch (g_Header.encryptionType)
    {
    case nn::bcat::service::EncryptionType_NoEncrypted:
        NN_LOG("Enc: NoEncrypted\n");
        break;
    case nn::bcat::service::EncryptionType_AesCtr128:
        NN_LOG("Enc: AES CTR 128\n");
        break;
    case nn::bcat::service::EncryptionType_AesCtr192:
        NN_LOG("Enc: AES CTR 192\n");
        break;
    case nn::bcat::service::EncryptionType_AesCtr256:
        NN_LOG("Enc: AES CTR 256\n");
        break;
    default:
        NN_LOG("Enc: Unknown\n");
    }

    EXPECT_TRUE(g_Header.encryptionType == nn::bcat::service::EncryptionType_NoEncrypted ||
        g_Header.encryptionType == nn::bcat::service::EncryptionType_AesCtr128 ||
        g_Header.encryptionType == nn::bcat::service::EncryptionType_AesCtr192 ||
        g_Header.encryptionType == nn::bcat::service::EncryptionType_AesCtr256);

    switch (g_Header.signatureType)
    {
    case nn::bcat::service::SignatureType_RsaPkcs1Sha1:
        NN_LOG("Signature: RSA-PKCS1-SHA1\n");
        break;
    case nn::bcat::service::SignatureType_RsaPkcs1Sha256:
        NN_LOG("Signature: RSA-PKCS1-SHA256\n");
        break;
    case nn::bcat::service::SignatureType_RsaPssSha1:
        NN_LOG("Signature: RSA-PSS-SHA1\n");
        break;
    case nn::bcat::service::SignatureType_RsaPssSha256:
        NN_LOG("Signature: RSA-PSS-SHA256\n");
        break;
    default:
        NN_LOG("Signature: Unknown\n");
    }

    EXPECT_TRUE(g_Header.signatureType == nn::bcat::service::SignatureType_RsaPkcs1Sha1 ||
        g_Header.signatureType == nn::bcat::service::SignatureType_RsaPkcs1Sha256 ||
        g_Header.signatureType == nn::bcat::service::SignatureType_RsaPssSha1 ||
        g_Header.signatureType == nn::bcat::service::SignatureType_RsaPssSha256);

    NN_LOG("SaltIndex: %d\n", g_Header.saltIndex);

    EXPECT_TRUE(g_Header.saltIndex <= nn::bcat::service::ArchiveDecryptionKeyGenerator::SaltIndexMax);

    NN_LOG("Reserved: %02x %02x %02x %02x %02x %02x %02x %02x\n",
        g_Header.reserved[0], g_Header.reserved[1], g_Header.reserved[2], g_Header.reserved[3],
        g_Header.reserved[4], g_Header.reserved[5], g_Header.reserved[6], g_Header.reserved[7]);

    NN_LOG("IV: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
        g_Header.iv[0], g_Header.iv[1], g_Header.iv[2], g_Header.iv[3],
        g_Header.iv[4], g_Header.iv[5], g_Header.iv[6], g_Header.iv[7],
        g_Header.iv[8], g_Header.iv[9], g_Header.iv[10], g_Header.iv[11],
        g_Header.iv[12], g_Header.iv[13], g_Header.iv[14], g_Header.iv[15]);
}

TEST(ArchiveDecoder, Decode)
{
    nn::bcat::service::ArchiveDecryptionKey key;
    nn::bcat::service::ArchiveDecryptionKeyGenerator::Generate(&key, Passphrase, AppId, g_Header.saltIndex);

    g_DecodedSize = g_EncodedSize - sizeof (g_Header);

    nn::bcat::service::ArchiveDecoder::Decode(g_Decoded, g_Encoded + sizeof (g_Header), g_DecodedSize, g_Header, key);

    EXPECT_TRUE(g_DecodedSize == g_AnswerSize);
    EXPECT_TRUE(std::memcmp(g_Decoded, g_Answer, g_AnswerSize) == 0);

    nn::bcat::service::ArchiveDecoder::Decode(g_Decoded, g_Encoded + sizeof (g_Header), g_DecodedSize, g_Header, Passphrase, AppId);

    EXPECT_TRUE(g_DecodedSize == g_AnswerSize);
    EXPECT_TRUE(std::memcmp(g_Decoded, g_Answer, g_AnswerSize) == 0);
}

TEST(ArchiveDecoder, Verify)
{
    nn::bcat::service::ArchiveVerifier verifier;

    verifier.Initialize(g_Header.signatureType);
    verifier.Update(g_Decoded, g_DecodedSize);

    EXPECT_TRUE(verifier.Verify(g_Header.signature, sizeof (g_Header.signature)));
}
