﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/arp/arp_Api.h>
#include <nn/arp/arp_Result.h>

#include <memory>

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/ns/ns_ApplicationManagerApi.h>

#define NNT_ARP_ASSERT_RESULT_SUCCESS(expression) \
    do { \
        nn::Result _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NN_LOG("%s(%d): Failure: " #expression " -> %03u-%04u (0x%08lx)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ASSERT_TRUE(_r.IsSuccess()); \
        } \
    } while (NN_STATIC_CONDITION(false))

#define NNT_ARP_ASSERT_RESULT_INCLUDED(result, expression) \
    do { \
        nn::Result _r = (expression); \
        if (!result::Includes(_r)) \
        { \
            NN_LOG("%s(%d): Not included in " #result ": " #expression " -> %03u-%04u (0x%08lx)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ASSERT_TRUE(result::Includes(_r)); \
        } \
    } while (NN_STATIC_CONDITION(false))

namespace {
bool CompareLaunchProperty(const nn::arp::ApplicationLaunchProperty& lhs, const nn::arp::ApplicationLaunchProperty& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.id.value, rhs.id.value);
    EXPECT_EQ(lhs.version, rhs.version);
    EXPECT_EQ(lhs.storageId, rhs.storageId);
    EXPECT_EQ(lhs.patchStorageId, rhs.patchStorageId);
    return true
        && (lhs.id.value == rhs.id.value)
        && (lhs.version == rhs.version)
        && (lhs.storageId == rhs.storageId)
        && (lhs.patchStorageId == rhs.patchStorageId);
}

bool CompareControlProperty(const nn::ns::ApplicationControlProperty& lhs, const nn::ns::ApplicationControlProperty& rhs) NN_NOEXCEPT
{
    return std::strncmp(lhs.displayVersion, rhs.displayVersion, sizeof(lhs.displayVersion)) == 0;
}
} // ~namespace <anonymous>

TEST(testArp, Basic)
{
    const nn::os::ProcessId PidToTest = {0x0001ull};
    nn::arp::DeleteProperties(PidToTest);

    const struct TestData {
        uint64_t id;
        uint32_t version;
        nn::ncm::StorageId storage;
        nn::ncm::StorageId patchStorage;
        char displayVersion[129];
    } Tests[] = {
        {
            0x0123456789ABCDEFull,
            0xFDB97531,
            nn::ncm::StorageId::Card,
            nn::ncm::StorageId::SdCard,
            "testtesttest"
        },
        {
            0x0000000000000001ull,
            0x00000001,
            nn::ncm::StorageId::BuildInUser,
            nn::ncm::StorageId::None,
            "TEST"
        },
    };

    for (const auto& t : Tests)
    {
        // テストデータ
        std::unique_ptr<nn::arp::ApplicationLaunchProperty> pExpectedLaunchProp(new nn::arp::ApplicationLaunchProperty());
        pExpectedLaunchProp->id.value = t.id;
        pExpectedLaunchProp->version = t.version;
        pExpectedLaunchProp->storageId = t.storage;
        pExpectedLaunchProp->patchStorageId = t.patchStorage;

        std::unique_ptr<nn::ns::ApplicationControlProperty> pExpectedCtrlProp(new nn::ns::ApplicationControlProperty());
        std::strncpy(pExpectedCtrlProp->displayVersion, t.displayVersion, sizeof(t.displayVersion));

        std::unique_ptr<nn::arp::ApplicationLaunchProperty> pLaunchProp(new nn::arp::ApplicationLaunchProperty());
        std::unique_ptr<nn::ns::ApplicationControlProperty> pCtrlProp(new nn::ns::ApplicationControlProperty());

        nn::ApplicationId applicationId = {t.id};

        // 未登録時の取得
        NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultNotRegistered, nn::arp::GetApplicationLaunchProperty(pLaunchProp.get(), PidToTest));
        NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultNotRegistered, nn::arp::GetApplicationLaunchProperty(pLaunchProp.get(), applicationId));
        NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultNotRegistered, nn::arp::GetApplicationControlProperty(pCtrlProp.get(), PidToTest));
        NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultNotRegistered, nn::arp::GetApplicationControlProperty(pCtrlProp.get(), applicationId));

        // 登録キャンセル
        {
            nn::arp::Registrar registrar;
            NNT_ARP_ASSERT_RESULT_SUCCESS(nn::arp::AcquireRegistrar(&registrar));
        }

        // 登録
        {
            nn::arp::Registrar registrar;
            NNT_ARP_ASSERT_RESULT_SUCCESS(nn::arp::AcquireRegistrar(&registrar));
            // NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultRegistryFull, nn::arp::AcquireRegistrar(&registrar));

            NNT_ARP_ASSERT_RESULT_SUCCESS(registrar.SetApplicationLaunchProperty(*pExpectedLaunchProp));
            NNT_ARP_ASSERT_RESULT_SUCCESS(registrar.SetApplicationControlProperty(*pExpectedCtrlProp));
            NNT_ARP_ASSERT_RESULT_SUCCESS(registrar.Issue(PidToTest));
        }

        // 取得
        NNT_ARP_ASSERT_RESULT_SUCCESS(nn::arp::GetApplicationLaunchProperty(pLaunchProp.get(), PidToTest));
        EXPECT_TRUE(CompareLaunchProperty(*pExpectedLaunchProp.get(), *pLaunchProp.get()));
        NNT_ARP_ASSERT_RESULT_SUCCESS(nn::arp::GetApplicationLaunchProperty(pLaunchProp.get(), applicationId));
        EXPECT_TRUE(CompareLaunchProperty(*pExpectedLaunchProp.get(), *pLaunchProp.get()));

        NNT_ARP_ASSERT_RESULT_SUCCESS(nn::arp::GetApplicationControlProperty(pCtrlProp.get(), PidToTest));
        EXPECT_TRUE(CompareControlProperty(*pExpectedCtrlProp.get(), *pCtrlProp.get()));
        NNT_ARP_ASSERT_RESULT_SUCCESS(nn::arp::GetApplicationControlProperty(pCtrlProp.get(), applicationId));
        EXPECT_TRUE(CompareControlProperty(*pExpectedCtrlProp.get(), *pCtrlProp.get()));

        // 削除と取得
        nn::arp::DeleteProperties(PidToTest);
        NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultNotRegistered, nn::arp::GetApplicationLaunchProperty(pLaunchProp.get(), PidToTest));
        NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultNotRegistered, nn::arp::GetApplicationLaunchProperty(pLaunchProp.get(), applicationId));
        NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultNotRegistered, nn::arp::GetApplicationControlProperty(pCtrlProp.get(), PidToTest));
        NNT_ARP_ASSERT_RESULT_INCLUDED(nn::arp::ResultNotRegistered, nn::arp::GetApplicationControlProperty(pCtrlProp.get(), applicationId));
    }
}
