﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/applet/applet.h>

#include "../Common/applet_Configs.h"

//-----------------------------------------------------------------------------

void CheckAndProcessMessageImpl(nn::oe::Message expectMessage, int line) NN_NOEXCEPT;
void WaitForFocusStateChangedImpl(nn::oe::FocusState expect, int line) NN_NOEXCEPT;

#define CheckAndProcessMessage(expected) CheckAndProcessMessageImpl(expected, __LINE__)
#define WaitForFocusStateChanged(expected) WaitForFocusStateChangedImpl(expected, __LINE__)

namespace {

char g_StartupParamBuffer[ nn::applet::StartupParamSizeMax ];
char g_ReturnValueBuffer[ nn::applet::ReturnValueSizeMax ];

const size_t RequestCommandSize = 128 * 1024;
const size_t ReplyMessageSize   = 128 * 1024;
NN_ALIGNAS(4096) char g_RequestCommandBuffer[ RequestCommandSize ];
NN_ALIGNAS(4096) char g_ReplayMessageBuffer[ ReplyMessageSize ];

const size_t OptionalStartupParamSize = 2 * 1024 * 1024;
const size_t OptionalReturnValueSize  = 2 * 1024 * 1024;
NN_ALIGNAS(4096) char g_OptionalStartupParamBuffer[ OptionalStartupParamSize ];
NN_ALIGNAS(4096) char g_OptionalReturnValueBuffer[ OptionalReturnValueSize ];
char g_OptionalReturnValueBuffer2[ OptionalReturnValueSize ];

const size_t TransferStorageBufferSize = 1 * 1024 * 1024;
NN_ALIGNAS(4096) char g_TransferStorageBuffer[ TransferStorageBufferSize ];

void DumpMemory(const char* buffer)
{
    NN_LOG("OE: buffer=");
    for (int i=0; i<16; ++i)
    {
        NN_LOG(" %02x", buffer[i]);
    }
    NN_LOG("\n");
}

void FillMemory(char* buffer, size_t size, char seed)
{
    char c = seed;
    for (int i=0; i<size; ++i)
    {
        buffer[i] = c;

        c = (c < 0x7e) ? c + 1 : ' ';
    }
}

void CheckMemory(char* buffer, size_t size, char seed)
{
    DumpMemory(buffer);
    NN_LOG("OE: seed= 0x%02x\n", seed);

    char c = seed;
    for (int i=0; i<size; ++i)
    {
        NN_ABORT_UNLESS(buffer[i] == c);

        c = (c < 0x7e) ? c + 1 : ' ';
    }
}

}

//-----------------------------------------------------------------------------
//  ライブラリアプレットの Foreground 起動
//
void InvokeLibraryAppletInForegroundState() NN_NOEXCEPT
{
    // ライブラリアプレット起動準備
    NN_LOG("(2000)\n");
    NN_LOG("APL: Invoke nn::applet::CreateLibraryApplet( FG 起動 )\n");

    // ライブラリアプレットの作成
    nn::applet::LibraryAppletHandle la1;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateLibraryApplet(&la1, nn::applet::AppletId_LibraryAppletAuth, nn::applet::LibraryAppletMode_AllForeground));
    NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_LibraryAppletAuth, GetLibraryAppletId(la1));
    NN_ABORT_UNLESS_EQUAL(nn::applet::LibraryAppletMode_AllForeground, GetLibraryAppletMode(la1));

    {
        // 小さいデータの入力
        FillMemory(g_StartupParamBuffer, sizeof(g_StartupParamBuffer), 'A');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_StartupParamBuffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_StartupParamBuffer, sizeof(g_StartupParamBuffer)));
        PushToInChannel(la1, storageHandle);
    }
    {
        // 大きなデータの入力
        FillMemory(g_OptionalStartupParamBuffer, sizeof(g_OptionalStartupParamBuffer), '0');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateLargeStorage(&storageHandle, g_OptionalStartupParamBuffer, sizeof(g_OptionalStartupParamBuffer), false));
        PushToInChannel(la1, storageHandle);
    }
    {
        // 転送ストレージの入力
        FillMemory(g_TransferStorageBuffer, sizeof(g_TransferStorageBuffer), '2');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateTransferStorage(&storageHandle, g_TransferStorageBuffer, sizeof(g_TransferStorageBuffer)));
        PushToInChannel(la1, storageHandle);
    }
    {
        // 終了用バッファの入力
        FillMemory(g_OptionalReturnValueBuffer, sizeof(g_OptionalReturnValueBuffer), '1'); // 書く必要ある？
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateLargeStorage(&storageHandle, g_OptionalReturnValueBuffer, sizeof(g_OptionalReturnValueBuffer), true));
        PushToInChannel(la1, storageHandle);
    }

    // LA1 起動
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::StartLibraryApplet(la1));

    // アプリケーションに対して BG 遷移要求
    NN_LOG("(2150)\n");
    NN_LOG("APL: Wait nn::oe::FocusState_OutOfFocus\n");
    WaitForFocusStateChanged(nn::oe::FocusState_OutOfFocus);

    // ライブラリアプレットの終了待機
    NN_LOG("(2200)\n");
    NN_LOG("APL: Wait for LA1 exit\n");
    WaitSystemEvent(GetLibraryAppletExitEvent(la1));

    // ライブラリアプレットの終了確認
    NN_LOG("APL: Join LA1\n");
    JoinLibraryApplet(la1);
    NN_ABORT_UNLESS(GetLibraryAppletExitReason(la1) == nn::applet::LibraryAppletExitReason_Normal);

    {
        // 小さいデータの出力
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(TryPopFromOutChannel(&storageHandle, la1));
        NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_ReturnValueBuffer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer)));
        ReleaseStorage(storageHandle);
        CheckMemory(g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer), 'D');
    }
    {
        // 大きいデータの出力
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(TryPopFromOutChannel(&storageHandle, la1));
        NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_OptionalReturnValueBuffer2));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_OptionalReturnValueBuffer2, sizeof(g_OptionalReturnValueBuffer2)));
        ReleaseStorage(storageHandle);
        CheckMemory(g_OptionalReturnValueBuffer2, sizeof(g_OptionalReturnValueBuffer2), '2');
    }

    // ハンドルのクローズ
    nn::applet::CloseLibraryApplet(la1);

    // アプリケーションに対して FG 遷移要求
    NN_LOG("(2250)\n");
    NN_LOG("APL: Wait for nn::oe::FocusState_InFocus\n");
    WaitForFocusStateChanged(nn::oe::FocusState_InFocus);
}

void InvokeLibraryAppletExpectedAbnormallyExited() NN_NOEXCEPT
{
    // ライブラリアプレット起動準備
    NN_LOG("(2300)\n");
    NN_LOG("APL: Invoke nn::applet::CreateLibraryApplet( FG 起動 )\n");

    // ライブラリアプレットの作成
    nn::applet::LibraryAppletHandle la1;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateLibraryApplet(&la1, nn::applet::AppletId_LibraryAppletAuth, nn::applet::LibraryAppletMode_AllForeground));
    NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_LibraryAppletAuth, GetLibraryAppletId(la1));
    NN_ABORT_UNLESS_EQUAL(nn::applet::LibraryAppletMode_AllForeground, GetLibraryAppletMode(la1));

    {
        // 小さいデータの入力
        FillMemory(g_StartupParamBuffer, sizeof(g_StartupParamBuffer), 'Z');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_StartupParamBuffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_StartupParamBuffer, sizeof(g_StartupParamBuffer)));
        PushToInChannel(la1, storageHandle);
    }

    // LA1 起動
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::StartLibraryApplet(la1));

    // LA1 異常終了待機
    NN_LOG("(2310)\n");
    NN_LOG("APL: Join LA1\n");
    JoinLibraryApplet(la1);
    NN_ABORT_UNLESS(GetLibraryAppletExitReason(la1) == nn::applet::LibraryAppletExitReason_Abnormal);

    // ハンドルのクローズ
    nn::applet::CloseLibraryApplet(la1);
}

//-----------------------------------------------------------------------------
//  ライブラリアプレットの Background 起動
//
void InvokeLibraryAppletInBackgroundState() NN_NOEXCEPT
{
    //-------------------------------------------------------------------------
    // ライブラリアプレット起動準備
    NN_LOG("(3000)\n");
    NN_LOG("APL: Invoke nn::applet::CreateLibraryAppletHandle( BG 起動 )\n");

    // ライブラリアプレットの作成
    nn::applet::LibraryAppletHandle la2;
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateLibraryApplet(&la2, nn::applet::AppletId_LibraryAppletCabinet, nn::applet::LibraryAppletMode_PartialForeground));

    // LA2 起動
    NN_LOG("(3100)\n");
    NN_LOG("APL: Invoke nn::applet::StartLibraryApplet(la2)\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::StartLibraryApplet(la2));

    // ライブラリアプレットに IPC コマンドを送信
    NN_LOG("(3200)\n");
    NN_LOG("APL: Invoke nn::applet::PushToInteractiveInChannel()\n");
    {
        FillMemory(g_RequestCommandBuffer, sizeof(g_RequestCommandBuffer), 'G');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_RequestCommandBuffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_RequestCommandBuffer, sizeof(g_RequestCommandBuffer)));
        PushToInteractiveInChannel(la2, storageHandle);
    }

    //-------------------------------------------------------------------------
    // LA が BG 起動し、アプリが FG 状態を維持している場合に、
    // SA からの FG 遷移要求を正しくハンドリングできるか
    //-------------------------------------------------------------------------
    // SA からの BG 遷移要求を待機
    NN_LOG("APL: Wait for nn::oe::FocusState_Background\n");
    WaitForFocusStateChanged(nn::oe::FocusState_Background);

    // SA からの FG 遷移要求を待機
    NN_LOG("APL: Wait for nn::oe::FocusState_InFocus\n");
    WaitForFocusStateChanged(nn::oe::FocusState_InFocus);


    //-------------------------------------------------------------------------
    // ライブラリアプレットから IPC 返答を受信
    NN_LOG("(3300)\n");
    NN_LOG("APL: Wait nn::applet::GetPopFromInteractiveOutChannelEvent(la2)\n");
    nn::os::WaitSystemEvent(GetPopFromInteractiveOutChannelEvent(la2));
    NN_LOG("APL: Invoke nn::applet::TryPopFromInteractiveOutChannel()\n");
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(TryPopFromInteractiveOutChannel(&storageHandle, la2));
        NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_ReplayMessageBuffer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_ReplayMessageBuffer, sizeof(g_ReplayMessageBuffer)));
        ReleaseStorage(storageHandle);
        CheckMemory(g_ReplayMessageBuffer, sizeof(g_ReplayMessageBuffer), 'H');
    }

    //-------------------------------------------------------------------------
    // ライブラリアプレットに空メッセージを投げて終了を要求
    NN_LOG("(3400)\n");
    NN_LOG("APL: Invoke nn::applet::RequestLibraryAppletExit()\n");
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, 0));
        PushToInteractiveInChannel(la2, storageHandle);
    }

    // ライブラリアプレットの終了待機
    NN_LOG("(3500)\n");
    NN_LOG("APL: Join LA2\n");
    JoinLibraryApplet(la2);
    NN_ABORT_UNLESS(GetLibraryAppletExitReason(la2) == nn::applet::LibraryAppletExitReason_Normal);

    // ハンドルのクローズ
    nn::applet::CloseLibraryApplet(la2);
}

//-----------------------------------------------------------------------------
//  ライブラリアプレット呼出しテスト
//
void MainSequenceTestOfLibraryAppletInvoking() NN_NOEXCEPT
{
    auto id = nn::applet::GetAppletResourceUserId();
    NN_LOG("APL: ARUID = 0x%016llx\n", id.lower);

    // ここに来た時点で FG 状態

    InvokeLibraryAppletInForegroundState();
    InvokeLibraryAppletExpectedAbnormallyExited();

    // デフォルトのマスターボリューム期待値の取得とチェック
    float applicationVolume;
    float libraryAppletVolume;
    nn::oe::GetExpectedVolumeBalance( &applicationVolume, &libraryAppletVolume );
    NN_LOG("APL: Default Application's expected volume   = %1.2f\n", applicationVolume);
    NN_LOG("APL: Default LibraryApplet's expected volume = %1.2f\n", libraryAppletVolume);
    NN_ABORT_UNLESS(applicationVolume   == 0.25f);
    NN_ABORT_UNLESS(libraryAppletVolume == 1.0f );

    // マスターボリューム期待値の変更とチェック
    nn::oe::SetExpectedVolumeBalance( 0.5f, 0.2f );
    {
        float volume1;
        float volume2;
        nn::oe::GetExpectedVolumeBalance( &volume1, &volume2 );
        NN_ABORT_UNLESS(volume1 == 0.5f);
        NN_ABORT_UNLESS(volume2 == 0.2f );
    }

    InvokeLibraryAppletInBackgroundState();

    // マスターボリューム期待値をテスト前の値に戻す
    nn::oe::SetExpectedVolumeBalance( applicationVolume, libraryAppletVolume );
}

