﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/applet/applet.h>

#include <mutex>

#include "../Common/applet_Common.h"
#include "../Common/applet_Configs.h"

//-----------------------------------------------------------------------------

void MainSequenceTestOfLibraryAppletInvoking(nn::os::SystemEventType* pEvent, nn::ae::ApplicationHandle handle) NN_NOEXCEPT;

void WaitInterval() NN_NOEXCEPT
{
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(200) );
}

//-----------------------------------------------------------------------------

namespace {

class MenuControl
{
public:
    MenuControl() NN_NOEXCEPT :
        m_OrderToResume(nn::os::EventClearMode_AutoClear),
        m_ReadyForSuspend(nn::os::EventClearMode_AutoClear),
        m_OrderToSuspend(false)
    {
    }

    // Resume 要求 と 要求待ち
    void Resume() NN_NOEXCEPT
    {
        m_OrderToResume.Signal();
    }

    void WaitForResuming() NN_NOEXCEPT
    {
        m_OrderToResume.Wait();
    }

    // Suspend 要求 と 完了、要求チェック
    void OrderToSuspend() NN_NOEXCEPT
    {
        m_OrderToSuspend = true;
    }

    bool IsSuspendOrdered() NN_NOEXCEPT
    {
        return m_OrderToSuspend;
    }

    void ReadyToSuspend() NN_NOEXCEPT
    {
        m_ReadyForSuspend.Signal();
    }

    void WaitForReadyToSuspend() NN_NOEXCEPT
    {
        m_ReadyForSuspend.Wait();
        m_OrderToSuspend = false;
    }

private:
    nn::os::Event       m_OrderToResume;
    nn::os::Event       m_ReadyForSuspend;
    bool                m_OrderToSuspend;
};

MenuControl g_MenuControl;


NN_OS_ALIGNAS_THREAD_STACK  uint8_t g_Stack[0x4000];

}   // namespace anonymous

//-----------------------------------------------------------------------------
//  疑似メニュースレッドです。
//
void PseudoMenu(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    // 最初は Suspend 状態で起動し ResumePseudoMenu() で動き出す。

    int count = 0;
    for (;;)
    {
        // ResumePseudoMenu() されるまで待ち
        g_MenuControl.WaitForResuming();

        auto id = nn::applet::GetAppletResourceUserId();
        NN_LOG("SA: ARUID = 0x%016llx\n", id.lower);

        nn::os::TimerEvent  pseudoVsync(nn::os::EventClearMode_AutoClear);
        pseudoVsync.StartPeriodic(0, nn::TimeSpan::FromMicroSeconds(16667));

        int vcount = 60;
        for (;;)
        {
            pseudoVsync.Wait();

            if (++vcount >= 60)
            {
                vcount = 0;
                NN_LOG("SA: === Pseudo Menu === (count: %d)\r", count++);
            }

            if (g_MenuControl.IsSuspendOrdered())
            {
                NN_LOG("\n");
                break;
            }
        }

        g_MenuControl.ReadyToSuspend();
    }
}

//-----

inline void SuspendPseudoMenu() NN_NOEXCEPT
{
    g_MenuControl.OrderToSuspend();
    g_MenuControl.WaitForReadyToSuspend();
}

inline void ResumePseudoMenu() NN_NOEXCEPT
{
    g_MenuControl.Resume();
}

//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x200000 );
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}


//-----------------------------------------------------------------------------
//  制御用のメインループです。
//
void MainLoop(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    ResumePseudoMenu();

    for (;;)
    {
        auto message = nn::ae::WaitForNotificationMessage(pEvent);
        switch (message)
        {
            case nn::ae::Message_ChangeIntoForeground:
                    {
                        nn::ae::AcquireForegroundRights();
                        NN_LOG("SA: Changed into Foreground\n");
                        ResumePseudoMenu();
                    }
                    break;

            case nn::ae::Message_ChangeIntoBackground:
                    {
                        SuspendPseudoMenu();
                        NN_LOG("SA: Changed into Background\n");
                        nn::ae::ReleaseForegroundRights();
                    }
                    break;

            case nn::ae::Message_ApplicationExited:
                    NN_LOG("SA: Detect the application exited.\n");
                    // この後に Message_ChangeIntoForeground が届く
                    break;

            case nn::ae::Message_Exit:
                    // 終了要求（システムアプレットには来ない）
                    // メインスレッドからリターンするためループを抜ける
                    return;

            default:
                    NN_ABORT("SA: Received unknown message= 0x%08x\n", message);
                    break;
        }
    }
}

//-----------------------------------------------------------------------------

void StarterSequenceTest(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_LOG("SA: Launch Starter Application\n");
    auto handle = nn::ae::CreateSystemApplication({0x0100000000001012});
    nn::ae::StartApplication(handle);

    nn::ae::RequestApplicationGetForeground(handle);
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    NN_LOG("SA: Wait for starter message via SystemGeneralChannel\n");
    nn::os::WaitSystemEvent(nn::ae::GetPopFromSystemGeneralChannelEvent());
    {
        nn::applet::StorageHandle storage;
        NN_ABORT_UNLESS(nn::ae::TryPopFromSystemGeneralChannel(&storage));
        NN_LOG("SA: Receive starter message via SystemGeneralChannel\n");
        char buffer[256];
        NN_ABORT_UNLESS_EQUAL(sizeof(buffer), GetStorageSize(storage));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storage, 0, buffer, sizeof(buffer)));
        CheckMemory(buffer, sizeof(buffer), '1');
        nn::applet::ReleaseStorage(storage);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::RequestToGetForeground());
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);

    WaitInterval();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::RequestApplicationGetForeground(handle));
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    NN_LOG("SA: Wait for home button\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectShortPressingHomeButton);
    NN_LOG("SA: ignore home button\n");

    NN_LOG("SA: Wait for starter message via SystemGeneralChannel\n");
    nn::os::WaitSystemEvent(nn::ae::GetPopFromSystemGeneralChannelEvent());
    {
        nn::applet::StorageHandle storage;
        NN_ABORT_UNLESS(nn::ae::TryPopFromSystemGeneralChannel(&storage));
        NN_LOG("SA: Receive starter message via SystemGeneralChannel\n");
        char buffer[256];
        NN_ABORT_UNLESS_EQUAL(sizeof(buffer), GetStorageSize(storage));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storage, 0, buffer, sizeof(buffer)));
        CheckMemory(buffer, sizeof(buffer), '2');
        nn::applet::ReleaseStorage(storage);
    }

    NN_LOG("SA: Wait for home button\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectShortPressingHomeButton);

    NN_LOG("SA: Request exit to Starter\n");
    nn::ae::RequestApplicationExit(handle);
    WaitForAllMessageArrived("SA", pEvent, nn::ae::Message_ChangeIntoForeground, nn::ae::Message_ApplicationExited);
    NN_ABORT_UNLESS_EQUAL(nn::ae::ApplicationResult_NormallyExited, nn::ae::GetApplicationResult(handle));

    nn::ae::CloseApplicationHandle(handle);
}

void MainApplicationLanuchTest(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    char g_Buffer[1024];
    {
        NN_LOG("SA: Launch APL 0 with 'X'\n");
        auto handle = nn::ae::CreateApplication(appletSequenceOeApplication);
        NN_ABORT_UNLESS_EQUAL(appletSequenceOeApplication.value, nn::ae::GetApplicationId(handle).value);
        {
            FillMemory(g_Buffer, sizeof(g_Buffer), 'X');
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_Buffer)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
            nn::ae::PushApplicationLaunchParameter(handle, nn::applet::LaunchParameterKind_Account, storageHandle);
        }
        nn::ae::StartApplication(handle);
        nn::ae::RequestApplicationGetForeground(handle);

        CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);
        CheckAndProcessMessage("SA", pEvent, nn::ae::Message_LaunchApplicationRequested);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::RequestApplicationExit(handle));
        WaitForAllMessageArrived("SA", pEvent, nn::ae::Message_ChangeIntoForeground,
                                               nn::ae::Message_ApplicationExited);

        NN_ABORT_UNLESS_EQUAL(nn::ae::ApplicationResult_NormallyExited, nn::ae::GetApplicationResult(handle));
        nn::ae::CloseApplicationHandle(handle);
    }
    {
        NN_LOG("SA: prepare to launch APL 1 with 'Y'\n");
        nn::ae::ApplicationHandle handle;
        NN_ABORT_UNLESS(nn::ae::TryPopLaunchRequestedApplication(&handle));
        NN_ABORT_UNLESS_EQUAL(appletSequenceOeApplication.value, nn::ae::GetApplicationId(handle).value);
        {
            char g_Buffer[1024];
            FillMemory(g_Buffer, sizeof(g_Buffer), 'Y');
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_Buffer)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
            nn::ae::PushApplicationLaunchParameter(handle, nn::applet::LaunchParameterKind_Account, storageHandle);
        }
        NN_LOG("SA: Launch APL 1 with 'Y'\n");
        nn::ae::StartApplication(handle);
        nn::ae::RequestApplicationGetForeground(handle);

        CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);
        CheckAndProcessMessage("SA", pEvent, nn::ae::Message_LaunchApplicationRequested);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::RequestApplicationExit(handle));
        WaitForAllMessageArrived("SA", pEvent, nn::ae::Message_ChangeIntoForeground,
                                               nn::ae::Message_ApplicationExited);

        NN_ABORT_UNLESS_EQUAL(nn::ae::ApplicationResult_NormallyExited, nn::ae::GetApplicationResult(handle));
        nn::ae::CloseApplicationHandle(handle);
    }
    {
        NN_LOG("SA: prepare to launch APL 1 with 'Z'\n");
        nn::ae::ApplicationHandle handle;
        NN_ABORT_UNLESS(nn::ae::TryPopLaunchRequestedApplication(&handle));
        NN_ABORT_UNLESS_EQUAL(appletSequenceOeApplication.value, nn::ae::GetApplicationId(handle).value);
        {
            char g_Buffer[1024];
            FillMemory(g_Buffer, sizeof(g_Buffer), 'Z');
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_Buffer)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
            nn::ae::PushApplicationLaunchParameter(handle, nn::applet::LaunchParameterKind_Account, storageHandle);
        }
        NN_LOG("SA: Launch APL 2 with 'Z'\n");
        nn::ae::StartApplication(handle);
        nn::ae::RequestApplicationGetForeground(handle);

        CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);
        CheckAndProcessMessage("SA", pEvent, nn::ae::Message_LaunchApplicationRequested);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::RequestApplicationExit(handle));
        WaitForAllMessageArrived("SA", pEvent, nn::ae::Message_ChangeIntoForeground,
                                               nn::ae::Message_ApplicationExited);
        NN_ABORT_UNLESS_EQUAL(nn::ae::ApplicationResult_AbnormallyExited, nn::ae::GetApplicationResult(handle));
        nn::ae::CloseApplicationHandle(handle);
    }
    {
        nn::ae::ApplicationHandle handle;
        NN_ABORT_UNLESS(nn::ae::TryPopLaunchRequestedApplication(&handle));
        nn::ae::CloseApplicationHandle(handle);
    }
}

void MainApplicationSequenceTest(nn::os::SystemEventType* pEvent, nn::ae::ApplicationHandle handle) NN_NOEXCEPT
{
    nn::Result result;

    //--------------------------------------------
    // アプリ起動時のメッセージチェック
    NN_LOG("(1000)\n");
    NN_LOG("SA: Wait for message of application launched.\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    // ダミーのアプリ起動リクエストを受け、破棄
    NN_LOG("(1010)\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_LaunchApplicationRequested);
    {
        nn::ae::ApplicationHandle handle;
        NN_ABORT_UNLESS(nn::ae::TryPopLaunchRequestedApplication(&handle));
        nn::ae::CloseApplicationHandle(handle);
    }

    WaitInterval();

    // スリープハンドリングなしでのスリープシーケンス(BG)
    NN_LOG("(1050)\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::StartSleepSequence());
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_FinishedSleepSequence);

    WaitInterval();

    // スリープハンドリングありでのスリープシーケンス(BG)
    NN_LOG("(1060)\n");
    nn::ae::BeginSleepHandling();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::StartSleepSequence());
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_RequestToPrepareSleep);
    nn::ae::AllowToEnterSleepAndWait();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_FinishedSleepSequence);
    nn::ae::EndSleepHandling();

    WaitInterval();

    // アプリを BG に遷移させることに成功する
    NN_LOG("(1100)\n");
    NN_LOG("SA: Invoke nn::ae::RequestToGetForeground()\n");
    nn::ae::RequestToGetForeground();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);

    WaitInterval();

    // スリープハンドリングなしでのスリープシーケンス(FG)
    NN_LOG("(1150)\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::StartSleepSequence());
    // 一旦 BG になるシーケンスが入るが SuspendWindow()/ResumeWindow() によって
    // BG と FG のメッセージが相殺される場合があるのでケアしておく。
    {
        auto message = nn::ae::WaitForNotificationMessage(pEvent);
        PrintAppletMessage("SA", message);
        ProcessMessage("SA", message);
        if (message == nn::ae::Message_ChangeIntoBackground)
        {
            WaitForAllMessageArrived("SA", pEvent, nn::ae::Message_ChangeIntoForeground,
                                                   nn::ae::Message_FinishedSleepSequence);
        }
        else if (message == nn::ae::Message_FinishedSleepSequence)
        {
            // FG/BG のメッセージが相殺されたケース。何もしない。
        }
        else
        {
            NN_ABORT("SA: Received unexpected message=0x%08x\n", message);
        }
    }
    WaitInterval();

    // アプリを再度 FG に戻す
    NN_LOG("(1200)\n");
    NN_LOG("SA: Invoke nn::ae::RequestApplicationGetForeground()\n");
    nn::ae::RequestApplicationGetForeground(handle);
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    WaitInterval();

    // スリープシーケンス中での FG の確保
    NN_LOG("(1250)\n");

    // スリープハンドリングを開始
    NN_LOG("SA: Invoke nn::ae::BeginSleepHandling()\n");
    nn::ae::BeginSleepHandling();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::StartSleepSequence());
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_RequestToPrepareSleep);

    // エントランス相当の準備を想定して、スクリーンロックを開始
    NN_LOG("SA: Invoke nn::ae::LockForeground()\n");
    nn::ae::LockForeground();

    NN_LOG("SA: Invoke nn::ae::AllowToEnterSleepAndWait()\n");
    nn::ae::AllowToEnterSleepAndWait();
    WaitForAllMessageArrived("SA", pEvent, nn::ae::Message_ChangeIntoForeground,
                                           nn::ae::Message_FinishedSleepSequence);

    // SA でスクリーンロックを解除
    NN_LOG("SA: Invoke nn::ae::UnlockForeground()\n");
    nn::ae::UnlockForeground();

    // スリープハンドリングを終了
    NN_LOG("SA: Invoke nn::ae::EndSleepHandling()\n");
    nn::ae::EndSleepHandling();

    // アプリを再度 FG に戻す
    NN_LOG("(1260)\n");
    NN_LOG("SA: Invoke nn::ae::RequestApplicationGetForeground()\n");
    nn::ae::RequestApplicationGetForeground(handle);
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    WaitInterval();

#if 0
// NEW_AM: TODO: スリープ復帰からのメッセージに関しては要調整

    // スリープ要求を許諾される
    NN_LOG("(1600)\n");
    NN_LOG("SA: Invoke nn::ae::RequestToEnterSleep()\n");
    nn::ae::RequestToEnterSleep();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoSleep);

    NN_LOG("SA: Invoke nn::ae::AllowToEnterSleep()\n");
    nn::ae::AllowToEnterSleep();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ApprovedSleepRequest);

    // ここで SA が FG になるように要求する
    NN_LOG("(1650)\n");
    NN_LOG("SA: Invoke nn::ae::RequestToGetForeground()\n");
    nn::ae::RequestToGetForeground();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);
    WaitInterval();

    // 実際にスリープに入る
    // 現在の実装では、実際にはスリープに入らず、すぐに全アプレットに対して
    // Message_AwakeFromSleep が通知されて動作を再開する。
    NN_LOG("(1650)\n");
    NN_LOG("SA: Invoke nn::ae::EnterSleep()\n");
    result = nn::ae::EnterSleep();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_AwakeFromSleep);

    // アプリを再度 FG に戻す
    NN_LOG("(1660)\n");
    NN_LOG("SA: Invoke nn::ae::RequestApplicationGetForeground()\n");
    nn::ae::RequestApplicationGetForeground(handle);
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    WaitInterval();
#endif

}   // NOLINT(readability/fn_size)

void MainLibraryAppletSequenceTest(nn::os::SystemEventType* pEvent, nn::ae::ApplicationHandle handle) NN_NOEXCEPT
{
    //--------------------------------------------
    // FG 起動のライブラリアプレット１のテスト開始
    // 2000
    WaitInterval();
    NN_LOG("SA: Wait for Message_DetectShortPressingHomeButton\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectShortPressingHomeButton);

    // アプリから起動した LA1 を BG に遷移させることに成功する
    NN_LOG("SA: Invoke nn::ae::RequestToGetForeground()\n");
    nn::ae::RequestToGetForeground();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);

    WaitInterval();

    // アプリから起動した LA1 を再度 FG に戻す
    NN_LOG("SA: Invoke nn::ae::RequestApplicationGetForeground()\n");
    nn::ae::RequestApplicationGetForeground(handle);
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    WaitInterval();

    WaitInterval();

    // SA への汎用メッセージはまだ来ていない
    NN_ABORT_UNLESS(!TryWaitSystemEvent(nn::ae::GetPopFromSystemGeneralChannelEvent()));


    //--------------------------------------------
    // BG 起動のライブラリアプレット２のテスト開始
    // 3000
    WaitInterval();
    NN_LOG("SA: Wait for Message_DetectShortPressingHomeButton\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectShortPressingHomeButton);

    // LA2 が BG 起動なので、アプリを BG に遷移させることに成功する
    NN_LOG("SA: Invoke nn::ae::RequestToGetForeground()\n");
    nn::ae::RequestToGetForeground();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);

    WaitInterval();

    // SA への汎用メッセージの待機と取得
    NN_LOG("SA: Wait GetPopFromSystemGeneralChannelEvent()\n");
    WaitSystemEvent(nn::ae::GetPopFromSystemGeneralChannelEvent());
    NN_LOG("SA: TryPopFromSystemGeneralChannel('N')\n");
    {
        static char buffer[2048];
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromSystemGeneralChannel(&storageHandle));
        NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(buffer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, buffer, sizeof(buffer)));
        ReleaseStorage(storageHandle);
        CheckMemory(buffer, sizeof(buffer), 'N');
    }

    // LA2 が BG 起動なので、アプリを再度 FG に戻す
    NN_LOG("SA: Invoke nn::ae::RequestApplicationGetForeground()\n");
    nn::ae::RequestApplicationGetForeground(handle);
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoBackground);

    WaitInterval();
}

void MainApplicatoinExit(nn::os::SystemEventType* pEvent, nn::ae::ApplicationHandle handle) NN_NOEXCEPT
{
    //--------------------------------------------
    // 各種ボタン押下メッセージの送信と受信
    NN_LOG("(9000)\n");
    NN_LOG("SA: Invoke nn::ae::NotifyShortPressingHomeButtonForDebug()\n");
    nn::ae::NotifyShortPressingHomeButtonForDebug();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectShortPressingHomeButton);

    NN_LOG("SA: Invoke nn::ae::NotifyShortPressingPowerButtonForDebug()\n");
    nn::ae::NotifyShortPressingPowerButtonForDebug();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectShortPressingPowerButton);

    NN_LOG("SA: Invoke nn::ae::NotifyMiddlePressingPowerButtonForDebug()\n");
    nn::ae::NotifyMiddlePressingPowerButtonForDebug();
#if 0
    // PowerButton 中押し（３秒）は OverlayApplet が取得することになったので、
    // 上記 API を呼んでも SA には通知が来ないことが正しい
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectMiddlePressingPowerButton);
#endif

    WaitInterval();
    WaitInterval();

    // 上記の PowerButton 中押しによって、OA が InteractionMode となるため、
    // 以下の HomeButton 短押しは SA には通知されず、OA に横取りされる。
    // これにより、OA は InteractionMode が解除される。
    NN_LOG("SA: Invoke nn::ae::NotifyShortPressingHomeButtonForDebug()\n");
    nn::ae::NotifyShortPressingHomeButtonForDebug();

#if 0
    // NEW_AM: PowerButton 長押し（７秒）はそもそも通知が来ない。
    NN_LOG("SA: Invoke nn::ae::NotifyLongPressingPowerButtonForDebug()\n");
    nn::ae::NotifyLongPressingPowerButtonForDebug();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_DetectLongPressingPowerButton);
#endif

    WaitInterval();

    //--------------------------------------------
    // アプリ終了直前に一旦アプリを BG 状態にしておく
    NN_LOG("(9900)\n");
    NN_LOG("SA: Invoke nn::ae::RequestToGetForeground()\n");
    nn::ae::RequestToGetForeground();
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);

    // アプリが BG 状態のまま、アプリに対して終了要求を出す
    NN_LOG("(9910)\n");
    NN_LOG("SA: Invoke nn::ae::RequestApplicationExit()\n");
    nn::ae::RequestApplicationExit(handle);

    // アプリハンドルによる終了待機
    NN_LOG("SA: Wait for application exit.\n");
    {
        auto pExitEvent = nn::ae::GetApplicationExitEvent(handle);
        nn::os::WaitSystemEvent(pExitEvent);
    }

    NN_LOG("(9999)\n");
    CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ApplicationExited);
    NN_ABORT_UNLESS_EQUAL(nn::ae::ApplicationResult_NormallyExited, nn::ae::GetApplicationResult(handle));
    NN_LOG("SA: Detect application exited.\n");

    // 直前の RequestToGetForeground() により SA は既に FG 状態なので、
    // 以下のメッセージ通知は来ない。
    //  CheckAndProcessMessage("SA", pEvent, nn::ae::Message_ChangeIntoForeground);
}


//-----------------------------------------------------------------------------
//  本体起動理由の取得と表示
//
void DispAndCheckBootReason() NN_NOEXCEPT
{
    auto reason = nn::ae::GetBootReason();

    switch (reason)
    {
    case nn::ae::BootReason_ColdBoot:
            NN_LOG("SA: BootReason is ColdBoot\n");
            break;

    default:
            NN_ABORT("SA: Unknown boot reason = 0x%d\n", reason);
    }
}

//-----------------------------------------------------------------------------
//  メイン関数です。
//
void SystemAppletMenuMain(nn::ae::SystemAppletParameters* param)
{
    NN_UNUSED(param);

    NN_LOG("SA: Launched SystemAppletMenu.\n");

    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    auto aruid = nn::applet::GetAppletResourceUserId();
    NN_LOG("SA: ARUID = 0x%016llx\n", aruid.lower);
    NN_ABORT_UNLESS(aruid.lower != 0);

    // SIGLO-69853: ハンドルリークしないかの確認
    for (int i=0; i<1000; ++i)
    {
        nn::os::SystemEvent event;
        nn::ae::GetHdcpAuthenticationFailedEvent(&event);
    }
    // API が呼べることを確認
    nn::ae::LoadAndApplyIdlePolicySettings();

    nn::applet::SetInputDetectionSourceSet( nn::applet::InputSource_Pad |
                                            nn::applet::InputSource_Sensor |
                                            nn::applet::InputSource_DebugPad |
                                            nn::applet::InputSource_TouchScreen |
                                            nn::applet::InputSource_Keyboard |
                                            nn::applet::InputSource_Mouse |
                                            nn::applet::InputSource_UsbPort |
                                            nn::applet::InputSource_SdCardSlot |
                                            nn::applet::InputSource_GameCardSlot |
                                            0 );
    nn::applet::SetInputDetectionSourceSet( nn::applet::InputSource_All );


    // 本体起動理由を取得
    DispAndCheckBootReason();

    // SA 起動直後は FG 遷移要求がある
    CheckAndProcessMessage("SA", &event, nn::ae::Message_ChangeIntoForeground);

    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread(&thread, PseudoMenu, NULL, g_Stack, sizeof(g_Stack), nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread(&thread);

    StarterSequenceTest(&event);
    MainApplicationLanuchTest(&event);

    NN_LOG("SA: Start PseudoMenu thread.\n");

    //--------------------------------------------
    // アプリハンドルの生成と起動待機
    NN_LOG("SA: Launch application.\n");
    auto handle = nn::ae::LaunchApplication(appletSequenceOeApplication, true);
    for (int i = 0; i < 10; ++i)
    {
        WaitInterval();
    }

    MainApplicationSequenceTest(&event, handle);
    MainLibraryAppletSequenceTest(&event, handle);
    MainApplicatoinExit(&event, handle);
    MainSequenceTestOfLibraryAppletInvoking(&event, handle);

    NN_LOG("SA: Invoke nn::ae::CloseApplicationHandle().\n");
    nn::ae::CloseApplicationHandle(handle);

    MainLoop(&event);

    //----------

    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    NN_LOG("SA: End PseudoMenu thread.\n");

    NN_LOG("SA: Exit SystemAppletMenu from now.\n");
}

extern "C" void nnMain()
{
    NN_LOG("SA: Invoke nn::ae::InvokeSystemAppletMain().\n");

    nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, SystemAppletMenuMain);
}

