﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/applet/applet.h>

//-----------------------------------------------------------------------------

namespace {

class OverlayStateControl
{
public:
    // 状態遷移表：表内が状態遷移後の UserInteractionMode 状態を示す。
    // +---------------------+--------+--------+--------+--------+--------+
    // |                     |   Home Button   |       Power Button       |
    // | UserInteractionMode +--------+--------+--------+--------+--------+
    // |                     | Short  |  Long  | Short  | Middle |  Long  |
    // +=====================+========+========+========+========+========+
    // |        True         | False  | False  |  N/A   | False? |  N/A   |
    // +---------------------+--------+--------+--------+--------+--------+
    // |        False        |  N/A   | True   |  N/A   |  True  |  N/A   |
    // +---------------------+--------+--------+--------+--------+--------+

    void OnPressShortHomeButton() NN_NOEXCEPT
    {
        if (m_IsUserInteractionMode)
        {
            EndUserInteraction();
        }
        else
        {
            NN_LOG("OA: Unexpected !!\n");
        }
    }

    void OnPressLongHomeButton() NN_NOEXCEPT
    {

        if (m_IsUserInteractionMode)
        {
            EndUserInteraction();
        }
        else
        {
            BeginUserInteraction();
        }
    }

    void OnPressMiddlePowerButton() NN_NOEXCEPT
    {
        if (m_IsUserInteractionMode)
        {
            EndUserInteraction();
        }
        else
        {
            BeginUserInteraction();
        }
    }

private:
    void BeginUserInteraction() NN_NOEXCEPT
    {
        m_IsUserInteractionMode = true;
        nn::ae::SetTransparentVolumeRate(0.3f);
        nn::ae::BeginOverlayUserInteraction();
        nn::ae::EnableAutoSleepTimeAndDimmingTime();
        NN_LOG("OA: Invoked nn::ae::BeginOverlayUserInteraction()\n");
    }

    void EndUserInteraction() NN_NOEXCEPT
    {
        m_IsUserInteractionMode = false;
        nn::ae::SetTransparentVolumeRate(1.0f);
        nn::ae::DisableAutoSleepTimeAndDimmingTime();
        nn::ae::EndOverlayUserInteraction();
        NN_LOG("OA: Invoked nn::ae::EndOverlayUserInteraction()\n");
    }

private:
    bool    m_IsUserInteractionMode{false};
};

OverlayStateControl g_OverlayStateControl;

}   // namespace anonymous


//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x200000 );
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

//-----------------------------------------------------------------------------
//  オーバレイアプレットのメッセージハンドリング
//
bool HandleAppletMessage(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    auto message = nn::ae::WaitForNotificationMessage(pEvent);

    switch (message)
    {
        case nn::ae::Message_Exit:
                NN_LOG("OA: Received Message_Exit\n");
                return false;

        case nn::ae::Message_DetectShortPressingHomeButton:
                NN_LOG("OA: Received Message_DetectShortPressingHomeButton\n");
                g_OverlayStateControl.OnPressShortHomeButton();
                break;

        case nn::ae::Message_DetectLongPressingHomeButton:
                NN_LOG("OA: Received Message_DetectLongPressingHomeButton\n");
                g_OverlayStateControl.OnPressLongHomeButton();
                break;

        case nn::ae::Message_DetectShortPressingPowerButton:
                NN_LOG("OA: Received !!UNEXPECTED!! Message_DetectShortPressingPowerButton\n");
                break;

        case nn::ae::Message_DetectMiddlePressingPowerButton:
                NN_LOG("OA: Received Message_DetectMiddlePressingPowerButton\n");
                g_OverlayStateControl.OnPressMiddlePowerButton();
                break;

        case nn::ae::Message_DetectLongPressingPowerButton:
                NN_LOG("OA: Received !!UNEXPECTED!! Message_DetectLongPressingPowerButton\n");
                break;

        case nn::ae::Message_VrModeChanged:
                NN_LOG("OA: Received Message_VrModeChanged\n");
                NN_LOG("OA: ae::IsVrMode() returns %s.\n", nn::ae::IsVrMode() ? "true" : "false");
                break;

        case nn::ae::Message_ShowApplicationLogo:
                NN_LOG("OA: Received Message_ShowApplicationLogo for 0x%016llx\n", nn::ae::GetApplicationIdForLogo().value);
                break;

        case nn::ae::Message_HideApplicationLogo:
                NN_LOG("OA: Received Message_HideApplicationLogo\n");
                break;

        case nn::ae::Message_ForceHideApplicationLogo:
                NN_LOG("OA: Received Message_ForceHideApplicationLogo\n");
                break;

        default:
                NN_ABORT("OA: Received unexpected message= 0x%08x", message);
                break;
    }
    return true;
}

//-----------------------------------------------------------------------------
//  メイン関数です。
//
void OverlayAppletMain(nn::ae::OverlayAppletParameters* param)
{
    NN_UNUSED(param);

    NN_LOG("OA: Launched OverlayApplet.\n");

    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    auto aruid = nn::applet::GetAppletResourceUserId();
    NN_LOG("OA: ARUID = 0x%016llx\n", aruid.lower);
    NN_ABORT_UNLESS(aruid.lower != 0);

    for (;;)
    {
        if (!HandleAppletMessage(&event))
        {
            break;
        }
    }

    NN_LOG("OA: Exit OverlayApplet from now.\n");
}

extern "C" void nnMain()
{
    NN_LOG("OA: Invoke nn::ae::InvokeOverlayAppletMain().\n");

    nn::ae::InvokeOverlayAppletMain(nn::ae::AppletId_OverlayApplet, OverlayAppletMain);
}

